/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2004 Nokia. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult <richard@imendio.com>
 */

#include <config.h>
#include <string.h>
#include <glib.h>

#include "gnome-vfs-volume.h"
#include "gnome-vfs-drive.h"
#include "gnome-vfs-volume-monitor-private.h"

/*
 * DBUS daemon support code.
 */

static void
iter_append_string_or_null (DBusMessageIter *iter, const gchar *str)
{
	dbus_message_iter_append_string (iter, str ? str : "");
}

static void
iter_next_dict_key (DBusMessageIter *iter, const gchar *key)
{
	gchar *str;	

	str = dbus_message_iter_get_dict_key (iter);

	g_assert (strcmp (str, key) == 0);
	dbus_free (str);
}

static gchar *
iter_get_string_or_null (DBusMessageIter *iter)
{
	gchar *str;
	gchar *ret;

	str = dbus_message_iter_get_string (iter);
	if (str && str[0] == '\0') {
		dbus_free (str);
		return NULL;
	}

	ret = g_strdup (str);
	dbus_free (str);

	return ret;
}

gboolean
_gnome_vfs_dbus_utils_append_volume (DBusMessageIter *iter,
				     GnomeVFSVolume  *volume)
{
	DBusMessageIter        dict;
	GnomeVFSVolumePrivate *priv;
	GnomeVFSDrive         *drive;

	g_return_val_if_fail (iter != NULL, FALSE);
	g_return_val_if_fail (volume != NULL, FALSE);
	
	if (!dbus_message_iter_append_dict (iter, &dict)) {
		return FALSE;
	}

	priv = volume->priv;
	
	dbus_message_iter_append_dict_key (&dict, "id");
	dbus_message_iter_append_int32 (&dict, priv->id);

	dbus_message_iter_append_dict_key (&dict, "volume_type");
	dbus_message_iter_append_int32 (&dict, priv->volume_type);

	dbus_message_iter_append_dict_key (&dict, "device_type");
	dbus_message_iter_append_int32 (&dict, priv->device_type);

	dbus_message_iter_append_dict_key (&dict, "drive");
	drive = gnome_vfs_volume_get_drive (volume);
	if (drive != NULL) {
		dbus_message_iter_append_int32 (&dict, drive->priv->id);
		gnome_vfs_drive_unref (drive);
	} else {
		dbus_message_iter_append_int32 (&dict, 0);
	}

	dbus_message_iter_append_dict_key (&dict, "activation_uri");
	iter_append_string_or_null (&dict, priv->activation_uri);

	dbus_message_iter_append_dict_key (&dict, "filesystem_type");
	iter_append_string_or_null (&dict, priv->filesystem_type);

	dbus_message_iter_append_dict_key (&dict, "display_name");
	iter_append_string_or_null (&dict, priv->display_name);

	dbus_message_iter_append_dict_key (&dict, "icon");
	iter_append_string_or_null (&dict, priv->icon);

	dbus_message_iter_append_dict_key (&dict, "is_user_visible");
	dbus_message_iter_append_boolean (&dict, priv->is_user_visible);

	dbus_message_iter_append_dict_key (&dict, "is_read_only");
	dbus_message_iter_append_boolean (&dict, priv->is_read_only);

	dbus_message_iter_append_dict_key (&dict, "is_mounted");
	dbus_message_iter_append_boolean (&dict, priv->is_mounted);

	dbus_message_iter_append_dict_key (&dict, "device_path");
	iter_append_string_or_null (&dict, priv->device_path);

	dbus_message_iter_append_dict_key (&dict, "unix_device");
	dbus_message_iter_append_int32 (&dict, priv->unix_device);

	dbus_message_iter_append_dict_key (&dict, "gconf_id");
	iter_append_string_or_null (&dict, priv->gconf_id);

	return TRUE;
}

GnomeVFSVolume *
_gnome_vfs_dbus_utils_get_volume (DBusMessageIter       *dict,
				  GnomeVFSVolumeMonitor *volume_monitor)
{
	DBusMessageIter        iter;
	GnomeVFSVolume        *volume;
	GnomeVFSVolumePrivate *priv;
	gint32                 drive_id;

	g_return_val_if_fail (dict != NULL, NULL);
	g_return_val_if_fail (volume_monitor != NULL, NULL);
	
	g_assert (dbus_message_iter_get_arg_type (dict) == DBUS_TYPE_DICT);

	dbus_message_iter_init_dict_iterator (dict, &iter);

	/* Note: the volumes lock is locked in _init. */
	volume = g_object_new (GNOME_VFS_TYPE_VOLUME, NULL);

	priv = volume->priv;

	iter_next_dict_key (&iter, "id");
	priv->id = dbus_message_iter_get_int32 (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "volume_type");
	priv->volume_type = dbus_message_iter_get_int32 (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "device_type");
	priv->device_type = dbus_message_iter_get_int32 (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "drive");
	drive_id = dbus_message_iter_get_int32 (&iter);

	if (drive_id != 0) {
		priv->drive = gnome_vfs_volume_monitor_get_drive_by_id (volume_monitor,
									drive_id);
		
		if (priv->drive != NULL) {
			/* RH, changed to many volumes per drive for 2.8. */
			_gnome_vfs_drive_add_mounted_volume (priv->drive, volume);
			
			/* The drive reference is weak */
			gnome_vfs_drive_unref (priv->drive);
		}
	}
	
	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "activation_uri");
	priv->activation_uri = iter_get_string_or_null (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "filesystem_type");
	priv->filesystem_type = iter_get_string_or_null (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "display_name");
	priv->display_name = iter_get_string_or_null (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "icon");
	priv->icon = iter_get_string_or_null (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "is_user_visible");
	priv->is_user_visible = dbus_message_iter_get_boolean (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "is_read_only");
	priv->is_read_only = dbus_message_iter_get_boolean (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "is_mounted");
	priv->is_mounted = dbus_message_iter_get_boolean (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "device_path");
	priv->device_path = iter_get_string_or_null (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "unix_device");
	priv->unix_device = dbus_message_iter_get_int32 (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "gconf_id");
	priv->gconf_id = iter_get_string_or_null (&iter);
	
	return volume;
}

gboolean
_gnome_vfs_dbus_utils_append_drive (DBusMessageIter *iter,
				    GnomeVFSDrive   *drive)
{
	DBusMessageIter       dict;
	GnomeVFSDrivePrivate *priv;
	GnomeVFSVolume       *volume;

	g_return_val_if_fail (iter != NULL, FALSE);
	g_return_val_if_fail (drive != NULL, FALSE);
	
	if (!dbus_message_iter_append_dict (iter, &dict)) {
		return FALSE;
	}

	priv = drive->priv;

	dbus_message_iter_append_dict_key (&dict, "id");
	dbus_message_iter_append_int32 (&dict, priv->id);
	
	dbus_message_iter_append_dict_key (&dict, "device_type");
	dbus_message_iter_append_int32 (&dict, priv->device_type);

	dbus_message_iter_append_dict_key (&dict, "volume");
	volume = gnome_vfs_drive_get_mounted_volume (drive);
	if (volume != NULL) {
		dbus_message_iter_append_int32 (&dict, volume->priv->id);
		gnome_vfs_volume_unref (volume);
	} else {
		dbus_message_iter_append_int32 (&dict, 0);
	}

	dbus_message_iter_append_dict_key (&dict, "device_path");
	iter_append_string_or_null (&dict, priv->device_path);

	dbus_message_iter_append_dict_key (&dict, "activation_uri");
	iter_append_string_or_null (&dict, priv->activation_uri);

	dbus_message_iter_append_dict_key (&dict, "display_name");
	iter_append_string_or_null (&dict, priv->display_name);

	dbus_message_iter_append_dict_key (&dict, "icon");
	iter_append_string_or_null (&dict, priv->icon);

	dbus_message_iter_append_dict_key (&dict, "is_user_visible");
	dbus_message_iter_append_boolean (&dict, priv->is_user_visible);

	dbus_message_iter_append_dict_key (&dict, "is_connected");
	dbus_message_iter_append_boolean (&dict, priv->is_connected);

	return TRUE;
}

GnomeVFSDrive *
_gnome_vfs_dbus_utils_get_drive (DBusMessageIter       *dict,
				 GnomeVFSVolumeMonitor *volume_monitor)
{
	DBusMessageIter       iter;
	GnomeVFSDrive        *drive;
	GnomeVFSDrivePrivate *priv;
	gint32                volume_id;

	g_return_val_if_fail (dict != NULL, NULL);
	g_return_val_if_fail (volume_monitor != NULL, NULL);
	
	g_assert (dbus_message_iter_get_arg_type (dict) == DBUS_TYPE_DICT);

	dbus_message_iter_init_dict_iterator (dict, &iter);

	/* Note: the drives lock is locked in _init. */
	drive = g_object_new (GNOME_VFS_TYPE_DRIVE, NULL);

	priv = drive->priv;

	iter_next_dict_key (&iter, "id");
	priv->id = dbus_message_iter_get_int32 (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "device_type");
	priv->device_type = dbus_message_iter_get_int32 (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "volume");
	volume_id = dbus_message_iter_get_int32 (&iter);

	if (volume_id != 0) {
		GnomeVFSVolume *mounted_volume;

		/* RH, changed for many volumes per drive for 2.8. */
		mounted_volume = gnome_vfs_volume_monitor_get_volume_by_id (volume_monitor,
									    volume_id);

		if (mounted_volume) {
			priv->volumes = g_list_append (priv->volumes, mounted_volume);
			_gnome_vfs_volume_set_drive (mounted_volume, drive);
		}
	}

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "device_path");
	priv->device_path = iter_get_string_or_null (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "activation_uri");
	priv->activation_uri = iter_get_string_or_null (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "display_name");
	priv->display_name = iter_get_string_or_null (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "icon");
	priv->icon = iter_get_string_or_null (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "is_user_visible");
	priv->is_user_visible = dbus_message_iter_get_boolean (&iter);

	dbus_message_iter_next (&iter);
	iter_next_dict_key (&iter, "is_connected");
	priv->is_connected = dbus_message_iter_get_boolean (&iter);

	return drive;
}

