/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2004 Nokia. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <glib.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus-glib.h>

#include "gnome-vfs-volume-monitor-private.h"
#include "dbus-utils.h"

#define d(x)


/*
 * FileInfo messages
 */

gboolean
dbus_utils_message_iter_append_file_info (DBusMessageIter        *iter,
					  const GnomeVFSFileInfo *info)
{
	DBusMessageIter  dict;
	gchar           *str;

	g_return_val_if_fail (iter != NULL, FALSE);
	g_return_val_if_fail (info != NULL, FALSE);
	
	if (!dbus_message_iter_append_dict (iter, &dict)) {
		return FALSE;
	}
	
	dbus_message_iter_append_dict_key (&dict, "name");
	str = gnome_vfs_escape_path_string (info->name);
	dbus_message_iter_append_string (&dict, str);
	g_free (str);

	dbus_message_iter_append_dict_key (&dict, "valid_fields");
	dbus_message_iter_append_int32 (&dict, info->valid_fields);

	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_TYPE) {
		dbus_message_iter_append_dict_key (&dict, "type");
		dbus_message_iter_append_int32 (&dict, info->type);
	}

	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS) {
		dbus_message_iter_append_dict_key (&dict, "permissions");
		dbus_message_iter_append_int32 (&dict, info->permissions);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_FLAGS) {
		dbus_message_iter_append_dict_key (&dict, "flags");
		dbus_message_iter_append_int32 (&dict, info->flags);
	}
		
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_DEVICE) {
		dbus_message_iter_append_dict_key (&dict, "device");
		dbus_message_iter_append_int32 (&dict, info->device);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_INODE) {
		dbus_message_iter_append_dict_key (&dict, "inode");
		dbus_message_iter_append_int64 (&dict, info->inode);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_LINK_COUNT) {
		dbus_message_iter_append_dict_key (&dict, "link_count");
		dbus_message_iter_append_int32 (&dict, info->link_count);
	}

	/* FIXME: is permissions right here? */
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS) {
		dbus_message_iter_append_dict_key (&dict, "uid");
		dbus_message_iter_append_uint32 (&dict, info->uid);
	}
	
	/* FIXME: is permissions right here? */
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS) {
		dbus_message_iter_append_dict_key (&dict, "gid");
		dbus_message_iter_append_uint32 (&dict, info->gid);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SIZE) {
		dbus_message_iter_append_dict_key (&dict, "filesize");
		dbus_message_iter_append_int64 (&dict, info->size);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_BLOCK_COUNT) {
		dbus_message_iter_append_dict_key (&dict, "block_count");
		dbus_message_iter_append_int64 (&dict, info->block_count);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_IO_BLOCK_SIZE) {
		dbus_message_iter_append_dict_key (&dict, "io_block_size");
		dbus_message_iter_append_uint32 (&dict, info->io_block_size);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_ATIME) {
		dbus_message_iter_append_dict_key (&dict, "atime");
		dbus_message_iter_append_int32 (&dict, info->atime);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MTIME) {
		dbus_message_iter_append_dict_key (&dict, "mtime");
		dbus_message_iter_append_int32 (&dict, info->mtime);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_CTIME) {
		dbus_message_iter_append_dict_key (&dict, "ctime");
		dbus_message_iter_append_int32 (&dict, info->ctime);
	}

	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SYMLINK_NAME) {
		dbus_message_iter_append_dict_key (&dict, "symlink_name");
		str = gnome_vfs_escape_path_string (info->symlink_name);
		dbus_message_iter_append_string (&dict, str);
		g_free (str);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE) {
		dbus_message_iter_append_dict_key (&dict, "mime_type");
		dbus_message_iter_append_string (&dict, info->mime_type);
	}

	return TRUE;
}

gboolean
dbus_utils_message_append_file_info (DBusMessage            *message,
				     const GnomeVFSFileInfo *info)
{
	DBusMessageIter iter;

	g_return_val_if_fail (message != NULL, FALSE);
	g_return_val_if_fail (info != NULL, FALSE);

	dbus_message_append_iter_init (message, &iter);
	
	return dbus_utils_message_iter_append_file_info (&iter, info);
}

GnomeVFSFileInfo *
dbus_utils_message_iter_get_file_info (DBusMessageIter *dict)
{
	DBusMessageIter   iter;
	gchar            *key;
	GnomeVFSFileInfo *info;
	gchar            *str;

	g_assert (dbus_message_iter_get_arg_type (dict) == DBUS_TYPE_DICT);
  
	dbus_message_iter_init_dict_iterator (dict, &iter);

	key = dbus_message_iter_get_dict_key (&iter);
	if (strcmp (key, "name") != 0) {
		dbus_free (key);
		return NULL;
	}
	dbus_free (key);
	
	info = gnome_vfs_file_info_new ();

	str = dbus_message_iter_get_string (&iter);
	info->name = gnome_vfs_unescape_string (str, NULL);
	dbus_free (str);

	dbus_message_iter_next (&iter);
	key = dbus_message_iter_get_dict_key (&iter);
	if (strcmp (key, "valid_fields") != 0) {
		dbus_free (key);
		gnome_vfs_file_info_unref (info);
		
		return NULL;
	}
	dbus_free (key);

	info->valid_fields = dbus_message_iter_get_int32 (&iter);

	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_TYPE) {
		dbus_message_iter_next (&iter);
		info->type = dbus_message_iter_get_int32 (&iter);
	}

	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS) {
		dbus_message_iter_next (&iter);
		info->permissions = dbus_message_iter_get_int32 (&iter);
	}

	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_FLAGS) {
		dbus_message_iter_next (&iter);
		info->flags = dbus_message_iter_get_int32 (&iter);
	}

	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_DEVICE) {
		dbus_message_iter_next (&iter);
		info->device = dbus_message_iter_get_int32 (&iter);
	}

	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_INODE) {
		dbus_message_iter_next (&iter);
		info->inode = dbus_message_iter_get_int64 (&iter);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_LINK_COUNT) {
		dbus_message_iter_next (&iter);
		info->link_count = dbus_message_iter_get_int32 (&iter);
	}

	/* FIXME: is permissions right here? */
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS) {
		dbus_message_iter_next (&iter);
		info->uid = dbus_message_iter_get_uint32 (&iter);
	}
	
	/* FIXME: is permissions right here? */
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS) {
		dbus_message_iter_next (&iter);
		info->gid = dbus_message_iter_get_uint32 (&iter);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SIZE) {
		dbus_message_iter_next (&iter);
		info->size = dbus_message_iter_get_int64 (&iter);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_BLOCK_COUNT) {
		dbus_message_iter_next (&iter);
		info->block_count = dbus_message_iter_get_int64 (&iter);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_IO_BLOCK_SIZE) {
		dbus_message_iter_next (&iter);
		info->io_block_size = dbus_message_iter_get_uint32 (&iter);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_ATIME) {
		dbus_message_iter_next (&iter);
		info->atime = dbus_message_iter_get_int32 (&iter);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MTIME) {
		dbus_message_iter_next (&iter);
		info->mtime = dbus_message_iter_get_int32 (&iter);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_CTIME) {
		dbus_message_iter_next (&iter);
		info->ctime = dbus_message_iter_get_int32 (&iter);
	}

	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SYMLINK_NAME) {
		dbus_message_iter_next (&iter);
		str = dbus_message_iter_get_string (&iter);
		info->symlink_name = gnome_vfs_unescape_string (str, NULL); 
		dbus_free (str);
	}
	
	if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE) {
		dbus_message_iter_next (&iter);
		str = dbus_message_iter_get_string (&iter);
		info->mime_type = g_strdup (str);
		dbus_free (str);
	}
	
	return info;
}

GList *
dbus_utils_message_get_file_info_list (DBusMessage *message)
{
	DBusMessageIter   iter;
	GnomeVFSFileInfo *info;
	GList            *list;

	g_return_val_if_fail (message != NULL, NULL);

	if (!dbus_message_iter_init (message, &iter)) {
		return NULL;
	}

	/* First skip the result code (which has already been checked). */
	if (!dbus_message_iter_next (&iter)) {
		return NULL;
	}
	
	list = NULL;	
	do {
		info = dbus_utils_message_iter_get_file_info (&iter);
		if (info) {
			list = g_list_prepend (list, info);
		}
	} while (dbus_message_iter_next (&iter));

	list = g_list_reverse (list);

	return list;
}


/*
 * Volume messages
 */

void
dbus_utils_message_append_volume_list (DBusMessage *message, GList *volumes)
{
	DBusMessageIter  iter;
	GList           *l;
	GnomeVFSVolume  *volume;
	
	g_return_if_fail (message != NULL);

	if (!volumes) {
		return;
	}

	dbus_message_append_iter_init (message, &iter);
		
	for (l = volumes; l; l = l->next) {
		volume = l->data;

		_gnome_vfs_dbus_utils_append_volume (&iter, volume);
	}
}

void
dbus_utils_message_append_drive_list (DBusMessage *message, GList *drives)
{
	DBusMessageIter  iter;
	GList           *l;
	GnomeVFSDrive   *drive;
	
	g_return_if_fail (message != NULL);

	if (!drives) {
		return;
	}

	dbus_message_append_iter_init (message, &iter);
		
	for (l = drives; l; l = l->next) {
		drive = l->data;

		_gnome_vfs_dbus_utils_append_drive (&iter, drive);
	}
}

void
dbus_utils_message_append_volume (DBusMessage *message, GnomeVFSVolume *volume)
{
	DBusMessageIter  iter;
	
	g_return_if_fail (message != NULL);
	g_return_if_fail (volume != NULL);

	dbus_message_append_iter_init (message, &iter);
	_gnome_vfs_dbus_utils_append_volume (&iter, volume);
}

void
dbus_utils_message_append_drive (DBusMessage *message, GnomeVFSDrive *drive)
{
	DBusMessageIter  iter;
	
	g_return_if_fail (message != NULL);
	g_return_if_fail (drive != NULL);

	dbus_message_append_iter_init (message, &iter);
	_gnome_vfs_dbus_utils_append_drive (&iter, drive);
}
