/*
* DSP audio backend for ESD
*
*
* Copyright (c) 2005 Nokia Corporation
*
* @contact: Makoto Sugano <makoto.sugano@nokia.com>
*
* ESD is free software; you can redistribute it and/or modify it under the terms of
* the GNU General Public License as published by the Free Software Foundation;
* either version 2, or (at your option) any later version.
*
* ESD is distributed in the hope that it will be useful, but WITHOUT ANY
* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
* FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along with 
* ESD; see the file COPYING.  If not, write to the Free Software Foundation, 
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*
*/

#if 0
#define DBG_PRINT(...) printf(__VA_ARGS__)
#else
#define DBG_PRINT(...) ((void)(0))
#endif

#ifdef HAVE_MACHINE_SOUNDCARD_H
#  include <machine/soundcard.h>
#else
#  ifdef HAVE_SOUNDCARD_H
#    include <soundcard.h>
#  else
#    include <sys/soundcard.h>
#  endif
#endif

#include <dsp/pcm_socket_node.h>
#include <dsp/interface_common.h>
#include <dsp/audio_socket_node.h>

#include <sys/mman.h>
#include <sys/poll.h>

#define DSP_MODE_ERROR 0xE0
#define DSP_RESET_MAX_ITER 10
#define DSP_VOL_SCALE_DEFAULT 16423;
#define DSP_VOL_POWER_DEFAULT 0;
#define ESD_RESET_WAIT 2

#define ARCH_esd_audio_devices
const char *esd_audio_devices()
{
    return "/dev/dsptask/pcm1";
}


AUDIO_STATUS_INFO status_info;
AUDIO_PARAMS_DATA pcm_params_data;
VOLUME_DATA pcm_volume_data;
PANNING_DATA pcm_panning_data;
AUDIO_INIT_STATUS pcm_init_status;

int pcm_drv;
int buf;
const char *device = "/dev/dsptask/pcm1";

short int pcm_state=STATE_UNINITIALISED;                            
short int pcm_bridge_buffer[BRIDGE_BUFFER_SIZE];
short int *pcm_mmap_buf_addr;

int dsp_hard_reset(void);

int
dsp_check_read()
{
  struct pollfd pfd;

  pfd.fd = pcm_drv;
  pfd.events = POLLIN;

  if(poll(&pfd,1,0)>0) return 1;
  else return 0;
}

void dsp_read_junk(void) {
  while( dsp_check_read() )
    read(pcm_drv,&pcm_bridge_buffer[0],sizeof(short int));    
}

/**
 * try to open the DSP device
 * @dev Name of the device
 * Returns If successfull, descriptor for device. Othervise -1
 *
 */
 
int dsp_open_device(const char *dev, int mode)
{
  int ret = 0;
  if ((ret = open(dev, mode, 0)) == -1) {
      // Opening device failed
      DBG_PRINT("ESD: Can't open the device %s\n", dev);
      perror(dev);
      return( -1 );
  }
  return ret;
}

/**
 * This method tries to initialize and mmap() DSP
 *
 * Returns 1 if successfull, -1 otherwise
 */

int dsp_init()
{
  AUDIO_STATUS_INFO statInfo;
  AUDIO_INIT_STATUS init_status;
  DSP_CMD_STATUS cmd;

  int len;
  short int tmp;
  
  statInfo.dsp_cmd = DSP_CMD_STATE;
  len = write(pcm_drv, &statInfo, sizeof(short int));
  if(len == -1) {
    pcm_state = DSP_MODE_ERROR;
    DBG_PRINT("failed to query dsp state\n");
    return -1;
  }

  len = read(pcm_drv, &statInfo, sizeof(AUDIO_STATUS_INFO));
  if(len == -1) {
    pcm_state = DSP_MODE_ERROR;
    DBG_PRINT("failed to read dsp return value\n");
    return -1;
  }

  if(statInfo.status == STATE_UNINITIALISED) {

    cmd.dsp_cmd = DSP_CMD_INIT;
    len = write(pcm_drv, &cmd, sizeof(short int));
    if(len == -1) {
      pcm_state = DSP_MODE_ERROR;
      return -1;
    }
    
    len = read(pcm_drv, &init_status, sizeof(AUDIO_INIT_STATUS));
    if(len == -1) {
      DBG_PRINT("Error reading status from DSP\n");
      return -1;
    }
    else if(init_status.init_status != DSP_OK) {
      DBG_PRINT("INIT_STATUS FAIL: %d\n", init_status.init_status);
      return -1;
    }

    pcm_init_status.stream_ID = init_status.stream_ID;
    pcm_init_status.bridge_buffer_size = init_status.bridge_buffer_size;
    pcm_init_status.mmap_buffer_size = init_status.mmap_buffer_size;
  }
  else {

    pcm_init_status.stream_ID = statInfo.stream_ID;
    pcm_init_status.bridge_buffer_size = statInfo.bridge_buffer_size;
    pcm_init_status.mmap_buffer_size = statInfo.mmap_buffer_size;

    tmp = DSP_CMD_STOP;
    len = write(pcm_drv, &tmp, sizeof(short int));
    if(len == -1) {
      pcm_state = DSP_MODE_ERROR;
      return -1;
    }
  
    // Read the "answer"
    read(pcm_drv, &cmd, sizeof(DSP_CMD_STATUS));
  }

  DBG_PRINT("PCM mmap buf size: %d\n", pcm_init_status.mmap_buffer_size);
  // mmap                                                                                                                           
  pcm_mmap_buf_addr = (char *) mmap(0,
      pcm_init_status.mmap_buffer_size*sizeof(short int),
      PROT_READ | PROT_WRITE, MAP_SHARED, pcm_drv, 0);

  if(pcm_mmap_buf_addr == NULL) {
    DBG_PRINT("Cannot mmap data buffer");
    return -1;
  }

  return 1;
  
}


/**
 * Set basic parameters for DSP
 *
 * Returns 1 if successfull, -1 otherwise
 */

int dsp_setparams()
{
  int value = 0, test = 0;

  // set the sound driver number playback rate
  //buf = esd_audio_rate;
  switch(esd_audio_rate)
  {
    case 48000: pcm_params_data.sample_rate = SAMPLE_RATE_48KHZ; break;
    case 44100: pcm_params_data.sample_rate = SAMPLE_RATE_44_1KHZ; break;
    case 32000: pcm_params_data.sample_rate = SAMPLE_RATE_32KHZ; break;
    case 24000: pcm_params_data.sample_rate = SAMPLE_RATE_24KHZ; break;
    case 22050: pcm_params_data.sample_rate = SAMPLE_RATE_22_05KHZ; break;
    case 16000: pcm_params_data.sample_rate = SAMPLE_RATE_16KHZ; break;
    case 12000: pcm_params_data.sample_rate = SAMPLE_RATE_12KHZ; break;
    case 11025: pcm_params_data.sample_rate = SAMPLE_RATE_11_025KHZ; break;
    case  8000: pcm_params_data.sample_rate = SAMPLE_RATE_8KHZ; break;
    default: DBG_PRINT("\nUnsupported sample rate.\n\n");getchar();
    return -1;
  }     

  dsp_read_junk();

  pcm_params_data.dsp_cmd = DSP_CMD_SET_PARAMS;
  pcm_params_data.stream_priority = 0;
  
  // set the sound driver audio format for playback
  value = test = ( (esd_audio_format & ESD_MASK_BITS) == ESD_BITS16 )
                   ? DSP_AFMT_S16_LE : DSP_AFMT_U8;
  pcm_params_data.audio_fmt = value;
  
  // set the sound driver number of channels for playback
  value = test = ( ( ( esd_audio_format & ESD_MASK_CHAN) == ESD_STEREO )
        ? 2 : 1 );
  pcm_params_data.number_channels = value;
  
  pcm_params_data.ds_stream_ID = 0;
  write(pcm_drv,&pcm_params_data,sizeof(pcm_params_data));
  read(pcm_drv,pcm_bridge_buffer,sizeof(DSP_CMD_STATUS));

  if(((DSP_CMD_STATUS *)pcm_bridge_buffer)->status != DSP_OK) {
    switch(((DSP_CMD_STATUS *)pcm_bridge_buffer)->status) {
      case DSP_ERROR_STATE: DBG_PRINT("\nError setting parameters: DSP_ERROR_STATE\n\n");break;
      case DSP_ERROR_RATE: DBG_PRINT("\nError setting parameters: DSP_ERROR_RATE\n\n");break;
      case DSP_ERROR_CHANNELS: DBG_PRINT("\nError setting parameters: DSP_ERROR_CHANNELS\n\n");break;
      case DSP_ERROR_DS_ID: DBG_PRINT("\nError setting parameters: DSP_ERROR_DS_ID\n\n");break;
      case DSP_ERROR_GENERAL: DBG_PRINT("\nError setting parameters: DSP_ERROR_GENERAL\n\n");break;
      case DSP_ERROR_FMT: DBG_PRINT("\nError setting parameters: DSP_ERROR_FMT\n\n");break;
      default:  DBG_PRINT("\nDSP init: Unknown error\n\n");break;
    }
    return -1;
  }
  
  DBG_PRINT("ESD setparams ok!\n");
  return 1;
}


/**
 * Set volume
 *
 * Returns 1 if successfull, -1 otherwise
 */

int dsp_set_volume() {
  pcm_volume_data.dsp_cmd = DSP_CMD_SET_VOLUME;  
  pcm_volume_data.scale = DSP_VOL_SCALE_DEFAULT;
  pcm_volume_data.power2 = DSP_VOL_POWER_DEFAULT;
  write(pcm_drv,&pcm_volume_data,sizeof(pcm_volume_data));
  read(pcm_drv,pcm_bridge_buffer,sizeof(DSP_CMD_STATUS));
  if(((DSP_CMD_STATUS *)pcm_bridge_buffer)->status != DSP_OK) {
    DBG_PRINT("\nError setting up volume.\n");
    return -1;
  }
  return 1;
}


/**
 * Set panning
 *
 * Returns 1 if successfull, -1 otherwise
 */

int dsp_set_panning() {
  pcm_panning_data.dsp_cmd = DSP_CMD_SET_PANNING;
  pcm_panning_data.left_gain = 0x4000;
  pcm_panning_data.right_gain = 0x4000;
  pcm_panning_data.steps = 300;
  write(pcm_drv,&pcm_panning_data,sizeof(pcm_panning_data));
  read(pcm_drv,pcm_bridge_buffer,sizeof(DSP_CMD_STATUS));
  if(((DSP_CMD_STATUS *)pcm_bridge_buffer)->status != DSP_OK) {
    DBG_PRINT("\nError setting up panning.\n");
    return -1;
  }
  return 1;
}


#define ARCH_esd_audio_open
int esd_audio_open()
{

  int value = 0, test = 0, turn = 0;
  
  short int tmp;
  int len, ret;
  int mode = O_RDWR;
    
  DBG_PRINT("Initializing DSP...\n");
  // open the sound device
    
  for ( turn=0; turn < DSP_RESET_MAX_ITER; turn++ ) {

    DBG_PRINT("ESD: Round %d...\n", turn);
    /* Try to open the device */
    ret = dsp_open_device(device, O_RDWR);
    if ( ret == -1 ) {
      /* sleep for a moment */
      sleep(ESD_RESET_WAIT);
    } else {  
      break;
    }
  }
    
  if ( ret == -1 ) {
    return -2;
  }
    
  pcm_drv = ret;
  
  dsp_read_junk();

  ret = dsp_init();
  if ( ret == -1 ) {
    return -2;
  }
  
  ret = dsp_setparams();
  if ( ret == -1 ) {
    return -2;
  }
  
  ret = dsp_set_volume();
  if ( ret == -1 ) {
    return -2;
  }
  
  ret = dsp_set_panning();
  if ( ret == -1 ) {
    return -2;
  }
  
  DBG_PRINT("\nPCM play ...\n");
  pcm_bridge_buffer[0] = DSP_CMD_PLAY;
  write(pcm_drv,pcm_bridge_buffer,sizeof(short int));
  pcm_state = DSP_CMD_PLAY;

  DBG_PRINT("DSP backend successfully initalized.\n");
  
  esd_audio_fd = pcm_drv;

  return pcm_drv;
}


#define ARCH_esd_audio_flush
void esd_audio_flush()
{
//    DBG_PRINT("Flush called\n");
    return;
}


#define ARCH_esd_audio_close
void esd_audio_close()
{
  unsigned short tmp;
  DSP_CMD_STATUS cmd;
  int len;

  DBG_PRINT("Close called\n");
  if(pcm_mmap_buf_addr != NULL) {
    munmap(pcm_mmap_buf_addr,
           pcm_init_status.mmap_buffer_size * sizeof(short int));
    pcm_mmap_buf_addr = NULL;
  }

  if(pcm_drv != -1) {
    
    // Read the junk
    dsp_read_junk();

    cmd.dsp_cmd = DSP_CMD_STOP;
    DBG_PRINT("Writing DSP_CMD_STOP\n");
    len = write(pcm_drv, &cmd, sizeof(short int));
    if(len != -1) {
      DBG_PRINT("Reading response to DSP_CMD_STOP\n");
      read(pcm_drv, &cmd, sizeof(DSP_CMD_STATUS));
    }
    if ( cmd.status == DSP_OK ) {
      DBG_PRINT("Got OK response\n");
    } else {
      DBG_PRINT("Error situation, trying to close...\n");    
    }
    
    cmd.dsp_cmd = DSP_CMD_CLOSE;
    DBG_PRINT("  Writing DSP_CMD_CLOSE\n");
    len = write(pcm_drv, &cmd, sizeof(short int));
    if(len != -1) {
      DBG_PRINT("  Reading response to DSP_CMD_CLOSE\n");
      read(pcm_drv, &cmd, sizeof(DSP_CMD_STATUS));
    }
    DBG_PRINT("  Got response\n");
    
    close(pcm_drv);
    
    pcm_state = STATE_UNINITIALISED;
    pcm_drv = -1;
  }
}

#define ARCH_esd_audio_pause
void esd_audio_pause()
{
  // DSP pausing not needed
  return;
}
                                                                                                                

#define ARCH_esd_audio_write
int esd_audio_write( void *buffer, int buf_size )
{
  int ret = 0;
  
  if ( ( buf_size == 0 )  ||
       ( buffer == NULL ) ||
       ( pcm_state == DSP_MODE_ERROR ) ) {
    DBG_PRINT("Empty buffer received\n");
    return 0;
  }

  ret = read(pcm_drv,&pcm_bridge_buffer[0],sizeof(short int));
  if ( ret == -1 ) {
    return 0;
  }
  
  if (pcm_bridge_buffer[0]==DSP_CMD_DATA_WRITE) {
    ret = read(pcm_drv,&pcm_bridge_buffer[1],sizeof(short int));
    if ( ret == -1 ) {
      return 0;
    }
    
    if(pcm_bridge_buffer[1]==DSP_ERROR_STATE) {
      DBG_PRINT("\nDSP_CMD_DATA_WRITE: Wrong pcm node state.\n");
      if ( ! dsp_hard_reset() ) {
        DBG_PRINT("Critical problem with DSP, giving up\n");
        pcm_state = DSP_MODE_ERROR;
        return -1;
      }
    }
    
    ret = read(pcm_drv,&pcm_bridge_buffer[2],sizeof(short int));
    if ( ret == -1 ) {
      return 0;
    }
    
    if(pcm_state == DSP_CMD_PLAY) {
      pcm_bridge_buffer[0] = DSP_CMD_DATA_WRITE;
      //DBG_PRINT("\nReceived: %d\n", buf_size);
      memcpy(pcm_mmap_buf_addr, (short int*)(buffer), buf_size);
      pcm_bridge_buffer[1] = buf_size/2;
      ret = write(pcm_drv,pcm_bridge_buffer,2*sizeof(short int));
      if ( ret == -1 ) {
        return 0;
      }
      
    }
  }

  return buf_size;
}

/* This method does the dsp node hard reinit.
 * Node is closed and then opened and initalized again.
 *
 * @return if hard reset went ok, 1 is returned. Otherwise 0
 */
 
int dsp_hard_reset(void)
{
  DBG_PRINT("Commencing hard reset, closing node...\n");
  esd_audio_close();
  
  // Sleep for a while so DSP can settle down
  usleep(50000);
  
  DBG_PRINT("Reopening and initalizing node...\n");
  if ( esd_audio_open() < 0 ) {
    DBG_PRINT("Hard init failed. Try resetting the DSP.\n");
    return 0;
  }
  else {
    DBG_PRINT("Hard init finished ok.\n");
  }
  return 1;
}
