/**
  @file pin-handler.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>
#include <glib.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#define DEFAULT_TIMEOUT (10 * 1000)
#define DEFAULT_PIN     "1234"

/** Startup config options */
static struct {
    gint   timeout;   /**< seconds to stay alive */
    gchar *pin;       /**< PIN to reply with */
    gchar *object;    /**< D-BUS path to listen on */
    gchar *bda;       /**< Specific BDA to catch */
} cfg;

/* For getopt */
static struct option const long_options[] = {
    {"help",     no_argument,       0, 'h'},
    {"pin",      required_argument, 0, 'p'},
    {"timeout",  required_argument, 0, 't'},
    {"object",   required_argument, 0, 'o'},
    {"bda",      required_argument, 0, 'b'},
    {NULL, 0, NULL, 0}
};

static char *program_name;

static gboolean        success    = TRUE;
static DBusConnection *connection = NULL;
static GMainLoop      *event_loop = NULL;

/* Print usage information and exit with status */
static void usage(int status) {
    printf("%s - BT PIN Handler test application %s\n", program_name, VERSION);

    printf("Compilation flags: ");
#ifdef DEBUG
    printf("+DEBUG ");
#else
    printf("-DEBUG ");
#endif

    printf(
     "\nUsage: %s [OPTION]...\n"
     "Options:\n"
     "-h, --help             Display this help and exit\n"
     "-p, --pin              PIN to reply with\n"
     "-t, --timeout          Seconds to stay alive\n"
     "-o, --object           D-BUS path to listen on\n"
     "-b, --bda              Specific BDA to catch\n"
     "\n", program_name);

    exit(status);
}

/* Process commandline options. Returns index of first
 * non-option argument */
static int decode_switches(int argc, char *argv[])
{
    int c;

    /* Set defaults */
    memset(&cfg, 0, sizeof(cfg));
    cfg.timeout = DEFAULT_TIMEOUT;
    cfg.pin     = g_strdup(DEFAULT_PIN);

    while ((c = getopt_long(argc, argv,
                    "h"   /* help      */
                    "V"   /* version   */
                    "p:"  /* pin       */
                    "o:"  /* object    */
                    "b:"  /* bda       */
                    "t:"  /* timeout   */
                    ,long_options, (int *) 0)) != EOF) {
        switch (c) {
            case 'V':
                printf("GW SDP Application %s\n", VERSION);
                exit(EXIT_SUCCESS);

            case 'h':
                usage(EXIT_SUCCESS);

            case 'p':
                g_free(cfg.pin);
                cfg.pin = g_strdup(optarg);
                break;

            case 'o':
                g_free(cfg.object);
                cfg.object = g_strdup(optarg);
                break;

            case 'b':
                g_free(cfg.bda);
                cfg.bda = g_strdup(optarg);
                break;

            case 't':
                cfg.timeout = atoi(optarg);
                break;

            default:
                usage(EXIT_FAILURE);
        }
    }

    if (cfg.object == NULL) {
        fprintf(stderr, "ERROR: You must specify the object path\n");
        usage(EXIT_FAILURE);
    }

    return optind;
}

static gboolean send_request(const gchar *method) {
    DBusMessage *request, *reply;

    request = dbus_message_new_method_call("com.nokia.btpin",
                                           "/com/nokia/btpin",
                                           "com.nokia.btpin",
                                           method);
    if (cfg.bda) {
        dbus_message_append_args(request,
                                 DBUS_TYPE_STRING, cfg.object,
                                 DBUS_TYPE_STRING, cfg.bda,
                                 DBUS_TYPE_INVALID);
    }
    else {
        dbus_message_append_args(request,
                                 DBUS_TYPE_STRING, cfg.object,
                                 DBUS_TYPE_INVALID);
    }

    reply = dbus_connection_send_with_reply_and_block(connection, request,
                                                      -1, NULL);
    dbus_connection_flush(connection);

    dbus_message_unref(request);

    if (reply == NULL) {
        fprintf(stderr, "sending failed (got NULL reply)\n");
        return FALSE;
    }

    if (dbus_message_get_type(reply) == DBUS_MESSAGE_TYPE_ERROR) {
        fprintf(stderr, "got error D-BUS reply\n");
        dbus_message_unref(reply);
        return FALSE;
    }

    dbus_message_unref(reply);

    return TRUE;
}

static gboolean timeout_cb(gpointer unused_data) {
    printf("Unregistering..."); fflush(stdout);
    if (!send_request("unregister")) {
        fprintf(stderr, "Registering handler failed\n");        
    }
    printf("done.\n");

    g_main_loop_quit(event_loop);

    return FALSE;
}

/* Handler for D-Bus requests */
static DBusHandlerResult pinreq_handler(DBusConnection *connection,
                                        DBusMessage    *message,
                                        void           *user_data) {

    DBusMessage *reply;

    printf("Got PIN request. Replying with PIN: %s\n", cfg.pin);

    reply = dbus_message_new_method_return(message);
    dbus_message_append_args(reply,
                             DBUS_TYPE_STRING, cfg.pin,
                             DBUS_TYPE_INVALID);
    dbus_connection_send(connection, reply, NULL);
    dbus_connection_flush(connection);

    printf("Reply sent.\n");

    dbus_message_unref(reply);

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusObjectPathVTable test_vtable = {
    .message_function    = pinreq_handler,
    .unregister_function = NULL
};

int main(int argc, char *argv[]) {
    gint i;

    program_name = argv[0];
    i = decode_switches(argc, argv);

    event_loop = g_main_loop_new(NULL, TRUE);

    connection = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
    if (connection == NULL) {
        fprintf(stderr, "D-BUS connection failed\n");
        exit(EXIT_FAILURE);
    }

    if (!dbus_connection_register_object_path(connection,
                                              cfg.object,
                                              &test_vtable,
                                              NULL)) {
        fprintf(stderr, "Registering object path \"%s\" failed\n",
                cfg.object);
        exit(EXIT_FAILURE);
    }

    dbus_connection_setup_with_g_main(connection, NULL);

    printf("Registering..."); fflush(stdout);
    if (!send_request("register")) {
        fprintf(stderr, "Registering handler failed\n");        
        exit(EXIT_FAILURE);
    }
    printf("done.\n");

    (void) g_timeout_add(cfg.timeout,
                         (GSourceFunc)timeout_cb,
                         NULL);

    printf("Entering mainloop\n");
    g_main_loop_run(event_loop);

    if (success) {
        printf("Tests completed successfully\n");
        exit(EXIT_SUCCESS);
    }
    else {
        printf("Tests failed\n");
        exit(EXIT_FAILURE);
    }
}
