/**
  @file key.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include <string.h>
#include <glib.h>
#include <bluetooth/bluetooth.h>

#include "log.h"
#include "io.h"
#include "key.h"

#define TMP_FILE "/tmp/.btlk_tmp_file"

static char *key_file = NULL;

/* A lot of this code is from hcid.h and security.c in bluez-utils */

struct link_key {
    bdaddr_t sba;
    bdaddr_t dba;
    uint8_t  key[16];
    uint8_t  type;
    time_t   time;
};

/* Link Key handling */

static struct link_key *get_link_key(int fd, bdaddr_t *sba, bdaddr_t *dba) {
    struct link_key k;
    struct link_key *key = NULL;
    int r;

    if (sba == NULL && dba == NULL)
        return NULL;

    while ((r = io_read(fd, &k, sizeof(k)))) {
        if (r < sizeof(k)) {
            if (r < 0)
                error("Link key database read failed. %s(%d)",
                        strerror(errno), errno);
            break;
        }


        if ((!sba || !bacmp(&k.sba, sba)) &&
            (!dba || !bacmp(&k.dba, dba))) {
            key = g_memdup(&k, sizeof(k));
            break;
        }
    }

    return key;
}

static int open_key_file(int opts) {
    int fd;

    g_assert(key_file != NULL);

    fd = open(key_file, O_RDWR | O_NONBLOCK | O_CREAT | opts, 0);
    if (fd < 0)
        error("Link key database open failed. %s(%d)",
                strerror(errno), errno);

    lockf(fd, F_LOCK, 0);

    return fd;
}

static void close_key_file(int fd) {
    lockf(fd, F_ULOCK, 0);
    close(fd);
}

static void show_link_key(struct link_key *key) {
    int i;
    char *tstr, *p;
    char sba[18], dba[18];

    ba2str(&key->sba, sba);
    ba2str(&key->dba, dba);

    tstr = ctime(&key->time);
    p = strchr(tstr, '\n');
    if (p != NULL)
        *p = '\0';

    printf("%s|0x%02X|%s|%s|", tstr, key->type, sba, dba);

    for (i = 0; i < 16; i++)
        printf("%02X", key->key[i]);

    printf("\n");

    return;
}

static void __list_link_keys(int fd, bdaddr_t *sba, bdaddr_t *dba) {
    struct link_key k;
    int r;

    while ((r = io_read(fd, &k, sizeof(k)))) {
        if (r < sizeof(k)) {
            if (r < 0)
                error("Link key database read failed. %s(%d)",
                        strerror(errno), errno);
            break;
        }

        if ((!sba || !bacmp(&k.sba, sba)) &&
            (!dba || !bacmp(&k.dba, dba))) {
            show_link_key(&k);
        }
    }
}

gboolean have_link_key(const char *sba, const char *dba) {
    struct link_key *key;
    gboolean ret;
    bdaddr_t sa, da, *sptr, *dptr;
    int fd;

    if (sba == NULL)
        sptr = NULL;
    else {
        if (str2ba(sba, &sa) < 0)
            return FALSE;
        sptr = &sa;
    }

    if (dba == NULL)
        dptr = NULL;
    else {
        if (str2ba(dba, &da) < 0)
            return FALSE;
        dptr = &da;
    }

    fd = open_key_file(0);
    if (fd < 0)
        key = NULL;
    else {
        key = get_link_key(fd, sptr, dptr);
        close_key_file(fd);
    }

    if (key == NULL)
        ret = FALSE;
    else
        ret = TRUE;

    g_free(key);
    return ret;
}

gboolean remove_link_key(const char *sba, const char *dba) {
    int tfd, fd;
    struct link_key k;
    bdaddr_t sa, da;
    gboolean found;
    int r;

    if (sba == NULL && dba == NULL) {
        error("Deleting all keys prohibited");
        return FALSE;
    }

    if (sba != NULL && str2ba(sba, &sa) < 0)
        return FALSE;

    if (dba != NULL && str2ba(dba, &da) < 0)
        return FALSE;


    tfd = open(TMP_FILE, O_RDWR | O_CREAT | O_TRUNC | O_NONBLOCK);
    if (tfd < 0) {
        error("open: %s", strerror(errno));
        return FALSE;
    }
    unlink(TMP_FILE);

    fd = open_key_file(0);
    if (fd < 0) {
        close(tfd);
        return FALSE;
    }

    found = FALSE;
    while ((r = io_read(fd, &k, sizeof(k)))) {
        if (r < sizeof(k)) {
            if (r < 0)
                error("Link key database read failed. %s(%d)",
                        strerror(errno), errno);
            break;
        }

        if ((!sba || !bacmp(&k.sba, &sa)) &&
            (!dba || !bacmp(&k.dba, &da))) {
            found = TRUE;
            continue; 
        }

        io_write(tfd, &k, sizeof(k));
    }

    if (!found) {
        report("No matching keys found");
        close(tfd);
        close_key_file(fd);
        return FALSE;
    }

    close_key_file(fd);
    /* XXX: Potential problem here: hcid may write to the file before we
     *      reopen it
     */
    fd = open_key_file(O_TRUNC);
    if (fd < 0) {
        close(tfd);
        return FALSE;
    }

    lseek(tfd, 0, SEEK_SET);
    while ((r = io_read(tfd, &k, sizeof(k)))) {
        if (r < sizeof(k)) {
            if (r < 0)
                error("Link key database read failed. %s(%d)",
                        strerror(errno), errno);
            break;
        }
        io_write(fd, &k, sizeof(k));
    }
   

    close(tfd);
    close_key_file(fd);

    return TRUE;
}

gboolean list_link_keys(const char *sba, const char *dba) {
    int fd;
    bdaddr_t sa, da, *sptr, *dptr;

    if (sba == NULL)
        sptr = NULL;
    else {
        if (str2ba(sba, &sa) < 0)
            return FALSE;
        sptr = &sa;
    }

    if (dba == NULL)
        dptr = NULL;
    else {
        if (str2ba(dba, &da) < 0)
            return FALSE;
        dptr = &da;
    }

    fd = open_key_file(0);
    if (fd < 0)
        return FALSE;

    __list_link_keys(fd, sptr, dptr);

    close_key_file(fd);
    return TRUE;
}

void key_file_init(const char *filename) {
    if (key_file != NULL)
        g_free(key_file);

    if (filename == NULL)
        key_file = g_strdup(DEFAULT_KEY_FILE);
    else
        key_file = g_strdup(filename);
}

