/**
  @file btpin-dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "log.h"
#include "dbus.h"
#include "dbus-helper.h"
#include "btpin-stack.h"
#include "bttools-dbus.h"
#include "btpin-dbus.h"

#define BLUEZ_PIN_AGENT_PATH "/org/bluez/PinAgent"

#define TIMEOUT (1000 * 60)

extern GMainLoop *event_loop;

static const char bda_sgn[]	= { DBUS_TYPE_STRING, DBUS_TYPE_STRING, DBUS_TYPE_INVALID };
static const char nobda_sgn[]	= { DBUS_TYPE_STRING, DBUS_TYPE_INVALID };

static gchar *read_pin(void) {
    char pin[128], *ptr;
    FILE *pin_file;

    pin_file = fopen(PIN_FILE, "r");
    if (pin_file == NULL) {
        error("Unable to open %s: %s", PIN_FILE, strerror(errno));
        return g_strdup(DEFAULT_PIN);
    }
    
    ptr = fgets(pin, sizeof(pin), pin_file);
    fclose(pin_file);
    if (ptr == NULL) {
        error("Unable to read %s", PIN_FILE);
        return g_strdup(DEFAULT_PIN);
    }

    /* Remove possible newline */
    ptr = strchr(pin, '\n');
    if (ptr != NULL)
        *ptr = '\0';

    return g_strdup(pin);
}

static DBusHandlerResult service_exit_filter(DBusConnection *connection,
                                             DBusMessage *message,
                                             void *user_data) {
    char *service; 

    if (dbus_message_is_signal(message,
                               DBUS_INTERFACE_ORG_FREEDESKTOP_DBUS,
                               "ServiceOwnerChanged")) {
        char *old, *new;

        if (!get_dbus_args(message,
                    DBUS_TYPE_STRING, &service,
                    DBUS_TYPE_STRING, &old,
                    DBUS_TYPE_STRING, &new,
                    DBUS_TYPE_INVALID)) {
            error("Invalid arguments for ServiceOwnerChanged signal");
            return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }

        dbus_free(new);

        /* Service was created */
        if (old[0] == '\0') {
            dbus_free(old);
            return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }

        dbus_free(old);
    }
    else
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;


    if (!btpin_stack_unregister(service, NULL, NULL)) {
        dbus_free(service);
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }
    else {
        debug("Service \"%s\" exited without first calling \"unregister\"",
                service);
        dbus_free(service);
        return DBUS_HANDLER_RESULT_HANDLED;
    }
}
static gboolean send_pin_reply(DBusConnection *connection,
                               DBusMessage    *message,
                               const gchar    *pin) {
    DBusMessage *reply;
    reply = new_dbus_method_return(message);
    if (pin != NULL) {
        debug("Replying with PIN: %s", pin);
        append_dbus_args(reply,
                         DBUS_TYPE_STRING, pin,
                         DBUS_TYPE_INVALID);
    }
    else
        debug("Replying with error");
    return send_and_unref(connection, reply);
}

static void notify_cb(DBusPendingCall *pending,
                      DBusMessage     *message) {
    DBusConnection *connection;
    DBusMessage *reply;
    gchar *pin = NULL;

    reply = dbus_pending_call_get_reply(pending);
    if (reply == NULL)
        error("dbus_pending_call_get_reply returned NULL!");
    if (dbus_message_get_type(reply) == DBUS_MESSAGE_TYPE_ERROR)
        error("Got error reply: %s", dbus_message_get_error_name(reply));
    else {
        if (!get_dbus_args(reply,
                           DBUS_TYPE_STRING, &pin,
                           DBUS_TYPE_INVALID))
            error("Invalid arguments in PIN reply");
        else
            debug("Got reply from handler: %s", pin);
    }

    if (reply != NULL)
        dbus_message_unref(reply);

    connection = get_dbus_connection();

    if (!send_pin_reply(connection, message, pin))
        error("Sending PIN reply failed");

    dbus_message_unref(message);
}

/* Handler for D-Bus requests */
static DBusHandlerResult bluez_req_handler(DBusConnection     *connection,
                                           DBusMessage        *message,
                                           void               *user_data) {
    gchar *bdastr = NULL, *service = NULL, *path = NULL;
    unsigned char *bda = NULL;
    dbus_bool_t out;
    guint bdalen;

    if (!get_dbus_args (message,
                        DBUS_TYPE_BOOLEAN, &out,
                        DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE, &bda, &bdalen,
                        DBUS_TYPE_INVALID)) {
        error("PinRequest with invalid arguments received");
        if (!send_invalid_args(connection, message))
            error("Unable to send invalid_args error");
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    if (bdalen != 6) {
        error("PinRequest with invalid BDA length: %u", bdalen);
        if (!send_invalid_args(connection, message))
            error("Unable to send invalid_args error");
        dbus_free(bda);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    bdastr = g_strdup_printf("%02X:%02X:%02X:%02X:%02X:%02X",
                             bda[5], bda[4], bda[3],
                             bda[2], bda[1], bda[0]);

    debug("PinRequest(%s, %s)", out ? "out" : "in", bdastr);

    if (!btpin_stack_get_handler(bdastr, &service, &path)) {
        /* Fallback */
        gchar *pin;
        debug("No PIN-handler matched. Using fallback");
        pin = read_pin();
        if (!send_pin_reply(connection, message, pin))
            error("Sending PIN reply failed");
        g_free(pin);
    }
    else {
        DBusPendingCall *pending;
        DBusMessage *newmsg;

        debug("Found matching handler at (%s, %s)", service, path);

        newmsg = new_dbus_method_call(service,
                                      path,
                                      dbus_message_get_interface(message),
                                      dbus_message_get_member(message));
        append_dbus_args(newmsg,
                         DBUS_TYPE_BOOLEAN, out,
                         DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE, bda, bdalen,
                         DBUS_TYPE_INVALID);

        if (!dbus_connection_send_with_reply(connection, newmsg,
                                             &pending, TIMEOUT))
            die("Out of memory during dbus_connection_send_with_reply");

        dbus_connection_flush(connection);
        debug("PIN request was sent to handler. Waiting for reply.");

        if (!dbus_pending_call_set_notify(pending,
                                          (DBusPendingCallNotifyFunction)notify_cb,
                                          message,
                                          (DBusFreeFunction)dbus_message_unref))
            die("Out of memory during dbus_pending_call_set_notify");

        dbus_message_ref(message);
    }

    dbus_free(bda);
    g_free(bdastr);
    g_free(service);
    g_free(path);

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult btpin_req_handler(DBusConnection     *connection,
                                           DBusMessage        *message,
                                           void               *user_data) {

    char *path, *bda;
    const char *service;

    if (dbus_message_has_signature(message, bda_sgn))
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &path,
                             DBUS_TYPE_STRING, &bda,
                             DBUS_TYPE_INVALID);
    else if (dbus_message_has_signature(message, nobda_sgn)) {
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &path,
                             DBUS_TYPE_INVALID);
        bda = NULL;
    }
    else {
        send_invalid_args(connection, message);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    service = dbus_message_get_sender(message);

    /* Handler "register" method call */
    if (dbus_message_is_method_call(message,
                                    BTPIN_INTERFACE,
                                    BTPIN_REQ_REGISTER)) {

        debug("Method call: register(%s, %s, %s)",
                service, path, bda ? bda : "any");

        if (!btpin_stack_register(service, path, bda)) {
            error("Registering PIN-handler failed!");
            if (!send_dbus_error(connection, message, BTPIN_ERROR_DUPLICATE))
                error("Sending error reply failed");
        }
        else {
            DBusMessage *reply;

            reply = new_dbus_method_return(message);
            if (!send_and_unref(connection, reply))
                error("Sending D-BUS reply failed");
        }

        dbus_free(path);
        if (bda != NULL)
            dbus_free(bda);

        return DBUS_HANDLER_RESULT_HANDLED;
    }

    /* Handler "unregister" method call */
    if (dbus_message_is_method_call(message,
                                    BTPIN_INTERFACE,
                                    BTPIN_REQ_UNREGISTER)) {

        debug("Method call: unregister(%s, %s, %s)",
                service, path, bda ? bda : "any");

        if (!btpin_stack_unregister(service, path, bda)) {
            error("Unregistering PIN-handler failed!");
            if (!send_dbus_error(connection, message, BTPIN_ERROR_NOMATCH))
                error("Sending error reply failed");
        }
        else {
            DBusMessage *reply;

            reply = new_dbus_method_return(message);
            if (!send_and_unref(connection, reply))
                error("Sending D-BUS reply failed");
        }


        dbus_free(path);
        if (bda != NULL)
            dbus_free(bda);

        return DBUS_HANDLER_RESULT_HANDLED;
    }
                                    
    dbus_free(path);
    if (bda != NULL)
        dbus_free(bda);

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusObjectPathVTable bluez_req_vtable = {
    .message_function    = bluez_req_handler,
    .unregister_function = NULL
};

static DBusObjectPathVTable btpin_req_vtable = {
    .message_function    = btpin_req_handler,
    .unregister_function = NULL
};

/* Create bindings for D-Bus handlers */
void init_dbus_handlers(DBusConnection *connection) {
    DBusError derr;

    debug("My base service is %s",
            dbus_bus_get_base_service(connection));

    dbus_error_init(&derr);
    dbus_connection_register_object_path(connection,
                                         BLUEZ_PIN_AGENT_PATH,
                                         &bluez_req_vtable,
                                         &derr);
    if (dbus_error_is_set(&derr)) {
        error("register_object_path(%s) failed: %s",
                BLUEZ_PIN_AGENT_PATH, derr.message);
        dbus_error_free(&derr);
    }

    dbus_bus_acquire_service(connection,
                             BTPIN_SERVICE,
                             DBUS_SERVICE_FLAG_PROHIBIT_REPLACEMENT,
                             &derr);
    if (dbus_error_is_set(&derr)) {
        error("Aquiring service %s failed: %s",
                BTPIN_SERVICE, derr.message);
        dbus_error_free(&derr);
    }

    dbus_connection_register_object_path(connection,
                                         BTPIN_PATH,
                                         &btpin_req_vtable,
                                         &derr);
    if (dbus_error_is_set(&derr)) {
        error("register_object_path(%s) failed: %s",
                BTPIN_PATH, derr.message);
        dbus_error_free(&derr);
    }

    if (!dbus_connection_add_filter(connection,
                                    service_exit_filter, NULL, NULL))
        die("Out of memory during dbus_connection_add_filter");

    dbus_bus_add_match(connection,
            "interface=" DBUS_INTERFACE_ORG_FREEDESKTOP_DBUS
            ",member=ServiceOwnerChanged",
            &derr);
    if (dbus_error_is_set(&derr)) {
        error("dbus_add_match() failed: %s", derr.message);
        dbus_error_free(&derr);
    }
}

void destroy_dbus_handlers(DBusConnection *connection) {
    dbus_connection_unregister_object_path(connection, BLUEZ_PIN_AGENT_PATH);
}
