/**
  @file btname-op.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <glib.h>

#include "log.h"
#include "btname-op.h"

#define BTDEV           "hci0"
#define HCICONFIG       "/usr/sbin/hciconfig"
#define NAMEFILE        "/etc/bluetooth/name"

static gboolean set_btname(const char *name) {
    pid_t pid;

    pid = fork();
    if (pid < 0) {
        error("fork: %s", strerror(errno));
        return FALSE;
    }

    if (pid == 0) {
        execl(HCICONFIG, HCICONFIG, BTDEV, "name", name, NULL);
        _exit(EXIT_FAILURE);
    }
    else {
        int status = 0;

        if (waitpid(pid, &status, 0) < 0) {
            error("waitpid: %s", strerror(errno));
            return FALSE;
        }

        if (WEXITSTATUS(status) != EXIT_SUCCESS) {
            error("hciconfig exited with error status: %d\n",
                    WEXITSTATUS(status));
            return FALSE;
        }
    }

    return TRUE;
}

static gboolean valid_name(const char *name) {
    if (strlen(name) > MAXNAMELEN)
        return FALSE;
    return g_utf8_validate(name, -1, NULL);
}

gboolean get_name(char *name, int name_len) {
    FILE *file;

    file = fopen(NAMEFILE, "r");
    if (file == NULL) {
        if (gethostname(name, name_len) < 0) {
            error("gethostname: %s", strerror(errno));
            return FALSE;
        }
    }
    else {
        char *ptr;
        if (!fgets(name, name_len, file)) {
            error("Reading from %s failed\n", NAMEFILE);
            fclose(file);
            return FALSE;
        }
        fclose(file);
        ptr = strchr(name, '\n');
        if (ptr)
            *ptr = '\0';
    }

    return TRUE;
}

gboolean set_name(const char *name) {
    char oldname[MAXNAMELEN];
    FILE *file;

    if (!valid_name(name)) {
        error("Invalid name!");
        return FALSE;
    }

    if (!get_name(oldname, sizeof(oldname))) {
        error("Could not get old name!");
        return FALSE;
    }

    if (!set_btname(name)) {
        error("Setting name with hciconfig failed!");
        return FALSE;
    }

    file = fopen(NAMEFILE, "w+");
    if (file == NULL) {
        error("fopen: %s", strerror(errno));
        (void) set_btname(oldname);
        return FALSE;
    }

    fprintf(file, "%s\n", name);
    fclose(file);

    return TRUE;
}

