/* 
 * Framebuffer Progressbar
 * 
 * Compile with:
 *	gcc -s -O2 -Wall -o fb-progress fb-progress.c
 * 
 * This reads from a fifo number of steps and shows corresponding
 * progress bar position at the bottom of the framebuffer.  Before
 * starting, it will switch to the given console (virtual terminal).
 * 
 * NOTES
 * - This doesn't disable/handle VT switches
 *   (to be able to do screen redraws when they happen etc)
 * - Framebuffer and its signal handling was originally based on
 *   the code from Torsten Scherer's W window system m68k-linux
 *   graphics initialization code
 * - The screen width has to be taken from finfo.line_length
 *   instead of vinfo.xres(_virtual) as some screens might include
 *   padding there.  Framebuffer is panned to the bottom so that
 *   the graphics are visible.
 *
 * Changelog:
 * - 1. version:
 *   output progress bar to fb, switch VT on start & end
 *   and get progress updates from the fifo as numbers
 * - 2. version:
 *   remove progress number handling, update only on '#'
 *   and terminate cleanly on sigTERM
 * - 3. version:
 *   catch vt switching, add verbose option,
 *   rewrote argument handling so that user can select whether:
 *   - vt is switched,
 *   - screen is cleared at startup, and
 *   - progress bar is advanced once a sec or only
 *     when something is written to the fifo
 * 
 * This is free software; you can redistribute it and/or modify it
 * under the terms specified in the GNU Public Licence (GPL).
 *
 * Copyright (C) 1998 Torsten Scherer (fb init and its signal handling)
 * Copyright (C) 2005 Nokia Corporation
 * Author: Eero Tamminen
 * Contact: Kimmo Hmlinen <kimmo.hamalainen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <linux/fb.h>
#include <linux/vt.h>
#include <linux/kd.h>
#include "omapfb.h"

/* named pipe for progress communication */
#define PROGRESS_FIFO "/tmp/progress"
#define PROGRESS_HEIGHT 20

/* expected framebuffer properties */
#define CONSOLE	"/dev/tty0"
#define FB_DEV	"/dev/fb0"

/* colors from 24-bit 888 RGB to 16-bit 565 RGB
 * - clears to white
 * - bar is green
 */
#define COLOR_CLEAR\
	((0xff << 8) & 0xf800)|((0xff << 3) & 0x07e0)|((0xff >> 3) & 0x001f)
#define COLOR_BAR\
	((0x00 << 8) & 0xf800)|((0x40 << 3) & 0x07e0)|((0xa3 >> 3) & 0x001f)

typedef unsigned short uint16_t;

typedef struct {
	int fd;		/* framebuffer device file descriptor */
	int wd;		/* screen width in pixels (shorts) */
	int ht;		/* visible screen height */
	void *mem;	/* memory mapped framebuffer */
	size_t size;	/* size of mapped memory in bytes */

	int dirty;	/* whether framebuffer needs updating */
	int dx1, dx2;	/* + the dirty area */
	int dy1, dy2;

	int sig;	/* when the console switched, set to signal */
} myfb_t;

typedef struct {
	/* whether to listen a fifo for and advance progress with that */
	int use_fifo;
	/* whether to switch my own console/VT */
	int use_vt;
	/* set when program should exit (due to SIGTERM etc) */
	int do_exit;
	/* be verbose */
	int verbose;
} myoptions_t;

static myfb_t Fb;
static myoptions_t Options;


/* open the console device, return its file descriptor or -1 */
static int console_open(void)
{
	int fd;
	if ((fd = open(CONSOLE, O_RDWR)) < 0) {
		perror("open(" CONSOLE ")");
		return -1;
	}
	return fd;
}

/* changes to the given console
 * returns current console or -1 on error
 */
static int console_switch(int vt)
{
	struct vt_stat vtstat;
	int fd, ret;

	if (!Options.use_vt) {
		return 0;
	}

	if ((fd = console_open()) < 0) {
		return -1;
	}
	if (ioctl(fd, VT_GETSTATE, &vtstat)) {
		perror("ioctl(VT_GETSTATE)");
		close(fd);
		return -1;
	}
	ret = vtstat.v_active;
	if (ioctl(fd, VT_ACTIVATE, vt)) {
		perror("ioctl(VT_ACTIVATE)");
		close(fd);
		return -1;
	}
	if (ioctl(fd, VT_WAITACTIVE, vt)) {
		perror("ioctl(VT_WAITACTIVATE)");
		close(fd);
		return -1;
	}
	close(fd);

	if (Options.verbose) {
		printf("fb-progress: previous console %d, switched to %d\n",
		       ret, vt);
	}
	return ret;
}

/* open console and do given ioctl operation with given value,
 * show message if verbose and name with perror.
 * return zero on success, negative on error
 */
static int console_ioctl(int op, int val, const char *name, const char *msg)
{
	int fd;

	if (Options.verbose) {
		printf("fb-progress: %s\n", msg);
	}
	if ((fd = console_open()) >= 0) {
		if (ioctl(fd, op, val) == 0) {
			return 0;
		} else {
			perror(name);
		}
		close(fd);
	}
	Options.do_exit = 1;
	return -1;
}

/* vt switch signal (SIGUSR*) handler, mark which signal was delivered
 * and ignore all handled signals until this is handled
 */
static void signal_handle(int sig)
{
	struct sigaction sa;

	sa.sa_flags = 0;
	sa.sa_handler = SIG_IGN;
	sigaction(SIGUSR1, &sa, NULL);
	sigaction(SIGUSR2, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);
	
	/* which signal */
	Fb.sig = sig;
}

/* set up signal handlers for termination and vtswitch
 */
static void signal_set_handlers(void)
{
	struct sigaction sa;
	sa.sa_handler = signal_handle;
	sigemptyset(&sa.sa_mask);
	sa.sa_flags = 0;	
	sigaction(SIGUSR1, &sa, NULL);
	sigaction(SIGUSR2, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);
}

/* this routine will be called at signal delivery by mainloop
 * when all graphics has been done, therefore we needn't any
 * kind of semaphore for the screen...
 * 
 * it returns true if screen needs redrawing.
 */
static int signal_process(void)
{
	switch(Fb.sig) {

	case SIGUSR1:
		if (console_ioctl(VT_RELDISP, 1, "VT_RELDISP", "Release VT")) {
			/* error */
			return 0;
		}
		break;

	case SIGUSR2:
		if (console_ioctl(VT_ACKACQ, 1, "VT_ACKACQ", "Acquire VT")) {
			/* error */
			return 0;
		}
		break;

	case SIGTERM:
		Options.do_exit = 1;
		return 0;
	default:
		fprintf(stderr, "fb-progress: Unknown signal %d\n", Fb.sig);
		Options.do_exit = 1;
		return 0;
	}
	/* no signal */
	Fb.sig = 0;

	/* signals ignored until now, set handler back */
	signal_set_handlers();
	
	/* screen needs redraw */
	return 1;
}

/* init vt switch catchup
 * return 0 on success, -1 on error
 */
static int signal_init(void)
{
	struct vt_mode vt;
	int fd;

	if (Options.verbose) {
		printf("fb-progress: Catch VT switches\n");
	}
	signal_set_handlers();

	if ((fd = console_open()) < 0) {
		return -1;
	}
	/* catch vty switches
	 */
	vt.mode = VT_PROCESS;	/* take control */
	vt.waitv = 0;		/* do not hang writes when not active */
	vt.relsig = SIGUSR1;	/* signal to send on release request */
	vt.acqsig = SIGUSR2;	/* signal to send on acquisition notice */
	vt.frsig = SIGUSR1;	/* signal to use to force VT switch */
	if (ioctl(fd, VT_SETMODE, &vt)) {
		perror("ioctl(VT_SETMODE)");
		return -1;
	}
	close(fd);
	return 0;
}

/* initialize framebuffer and return struct to it or NULL for error
 */
static myfb_t* fb_init(void)
{
	struct fb_var_screeninfo vinfo;
	struct fb_fix_screeninfo finfo;
	
	if ((Fb.fd = open(FB_DEV, O_RDWR)) < 0) {
		perror("open(" FB_DEV ")");
		return NULL;
	}
	if (ioctl(Fb.fd, FBIOGET_FSCREENINFO, &finfo)) {
		perror("ioctl(FBIOGET_FSCREENINFO)");
		close(Fb.fd);
		return NULL;
	}

	if (ioctl(Fb.fd, FBIOGET_VSCREENINFO, &vinfo)){
		perror("ioctl(FBIOGET_VSCREENINFO)");
		close(Fb.fd);
		return NULL;
	}
	if (Options.verbose) {
		printf("fb-progress: using %ix%i of %ix%i pixels,\n"
		       "\t%i bits per pixel, with line_len %i\n",
		       vinfo.xres, vinfo.yres,
		       vinfo.xres_virtual, vinfo.yres_virtual,
		       vinfo.bits_per_pixel,
		       finfo.line_length);
	}
	Fb.wd = finfo.line_length / sizeof(uint16_t);
	Fb.size = finfo.line_length * vinfo.yres;
	Fb.wd = vinfo.xres;
	Fb.ht = vinfo.yres;

	Fb.mem = mmap(0, Fb.size, PROT_READ | PROT_WRITE, MAP_SHARED, Fb.fd, 0);
	if (Fb.mem < 0) {
		perror("mmap(" FB_DEV ")");
		close(Fb.fd);
		return NULL;
	}
	if (Options.verbose) {
		printf("fb-progress: mapped %ik videoram to %p\n",
		       Fb.size >> 10, Fb.mem);
	}

	/* pan to start of fb */
	vinfo.xoffset = 0;
	vinfo.yoffset = 0;
	if (ioctl(Fb.fd, FBIOPAN_DISPLAY, &vinfo)) {
		perror("ioctl(FBIOPAN_DISPLAY)");
		close(Fb.fd);
		return NULL;
	}
	Fb.dirty = 0;
	return &Fb;
}

static void fb_dirty(int x1, int y1, int x2, int y2)
{
	if (x1 < 0  || y1 < 0 || x2 < x1 || y2 < y1 ||
	    x2 > Fb.wd || y2 > Fb.ht) {
		fprintf(stderr,
			"Error: invalid flush region (%d,%d),(%d,%d)\n",
			x1, y1, x2, y2);
		return;
	}
	if (!Fb.dirty) {
		Fb.dx1 = x1;
		Fb.dy1 = y1;
		Fb.dx2 = x2;
		Fb.dy2 = y2;
		Fb.dirty = 1;
		return;
	}
	if (x1 < Fb.dx1) {
		Fb.dx1 = x1;
	}
	if (y1 < Fb.dy1) {
		Fb.dy1 = y1;
	}
	if (x2 > Fb.dx2) {
		Fb.dx2 = x2;
	}
	if (y2 > Fb.dy2) {
		Fb.dy2 = y2;
	}
}

static void fb_flush(void)
{
	struct fb_update_window update;

	if (Fb.mem && Fb.dirty) {
		update.x = Fb.dx1;
		update.y = Fb.dy1;
		update.width = Fb.dx2 - Fb.dx1;
		update.height = Fb.dy2 - Fb.dy1;
		if (Options.verbose) {
			printf("fb-progress: update %dx%d+%d+%d\n",
			       update.width, update.height,
			       update.x, update.y);
		}
		if (ioctl(Fb.fd, OMAPFB_UPDATE_WINDOW, &update) < 0) {
			perror("ioctl(OMAPFB_UPDATE_WINDOW)");
		}
		Fb.dirty = 0;
	}
}

static void fb_clear(void)
{
	uint16_t *dst, *end;
	dst = Fb.mem;
	end = Fb.mem + Fb.size;
	while (dst < end) {
		*dst++ = COLOR_CLEAR;
	}
	fb_dirty(0, 0, Fb.wd, Fb.ht);
}

static void fb_exit(void)
{
	if (Fb.fd) {
		if (Options.verbose) {
			printf("fb-progress: close fb\n");
		}
		/* close frame buffer */
		munmap(Fb.mem, Fb.size);
		close(Fb.fd);
		Fb.mem = NULL;
	}
}

/* draw progress bar between steps 'start' and 'end' when the
 * maximum number of steps is 'max'. return negative on error
 */
static int draw_steps(int start, int end, int max)
{
	int x, w, h, off;
	uint16_t *dst;

	if (end < start) {
		fprintf(stderr, "ERROR: end step (%d) is smaller than start (%d)\n", end, start);
		return -1;
	}
	if (end > max) {
		fprintf(stderr, "ERROR: end step (%d) is larger than max step (%d)\n", end, max);
		return -1;
	}
	/* convert step indeces to co-ordinates */
	start = start * Fb.wd / max;
	end = end * Fb.wd / max;
	w = end - start;
	h = PROGRESS_HEIGHT;

	/* destination offset is calculated from the screen bottom */
	dst = (uint16_t *)(Fb.mem + Fb.size) - Fb.wd * h + start;
	off = Fb.wd - w;

	while (--h >= 0) {

		x = w;
		while (--x >= 0) {
			*dst++ = COLOR_BAR;
		}
		dst += off;
	}

	fb_dirty(start, Fb.ht - PROGRESS_HEIGHT, end, Fb.ht);
	return 0;
}

/* make and open given file as a fifo, and return its fd or negative
 * value for an error
 */
static int fifo_open(const char *path)
{
	int fd;
	if (Options.verbose) {
		printf("fb-progress: open fifo %s\n", path);
	}
	if (mkfifo(path, (S_IRUSR|S_IWUSR|S_IWOTH)) < 0) {
		goto error;
	}
	/* allow me to read and others to write to this regardless of umask */
	if (chmod(path, (S_IRUSR|S_IWUSR|S_IWOTH)) < 0) {
		goto error;
	}
	/* open without waiting for writers */
	fd = open(path, O_RDONLY|O_NONBLOCK);
	if (fd < 0) {
		goto error;
	}
	return fd;

error:
	perror(path);
	unlink(path);
	return -1;
}

/* close and unlink the fifo file
 */
static void fifo_close(const char *path, int fd)
{
	if (Options.verbose) {
		printf("fb-progress: close/remove fifo %s\n", path);
	}
	if (close(fd) < 0) {
		perror(path);
	}
	if (unlink(path) < 0) {
		perror(path);
	}
}

/* reads chars from fifo and adds value accordingly
 * returns new value or zero for intrerrupt/error
 */
static int fifo_read(int fd)
{
	int i, bytes, ret, count = 0;
	fd_set rfds;
	char buf[8];

	FD_ZERO(&rfds);
	FD_SET(fd, &rfds);
	/* as the fifo is non-blocking, I need to check
	 * whether there's something to read...
	 */
	ret = select(fd+1, &rfds, NULL, NULL, NULL);
	if (ret < 0) {
		if (errno == EINTR) {
			/* interrupted */
			return 0;
		}
		perror("Progress fifo select()");
		Options.do_exit = 1;
		return 0;
	}
	bytes = read(fd, buf, sizeof(buf));
	if (bytes < 0) {
		if (errno == EINTR) {
			/* interrupted */
			return 0;
		}
		perror("Progres fifo read()");
		Options.do_exit = 1;
		return 0;
	}
	if (!bytes) {
		return 0;
	}
	
	for (i = 0; i < bytes; i++) {
		if (buf[i] == '#') {
			/* advance to next step */
			count++;
		}
		/* all other characters are ignored */
	}
	return count;
}

/* show error message, usage, and exit */
static void usage(const char *name, const char *error)
{
	fprintf(stderr, "\nERROR: %s!\n\n", error);
	printf("Usage: %s [options] <steps>\n\n"
	       "A progress bar for the the framebuffer, show at the bottom.\n"
	       "<steps> defines how many steps the progress bar has.\n\n"
	       "Options:\n"
	       "-v\t\tverbose\n"
	       "-c\t\tclear screen at startup\n"
	       "-s\t\tadvance one step each second (the default), or\n"
	       "-f\t\tread progress from fifo (each '#' advances one step)\n"
	       "-t <vt>\t\tswitch to given virtual terminal while showing the progress\n"
	       "-i <step>\tinitial step (< steps)\n\n"
	       "Examples:\n"
	       "\t%s -s -t 3 -c 30\n"
	       "\tsleep 30\n"
	       "\t%s -f -i 1 3\n"
	       "\techo \"##\" > %s\n",
	       name, name, name, PROGRESS_FIFO);
	exit(1);
}

int main(int argc, const char *argv[])
{
	int clear = 0, fd = 0, vt = 0, count, i;
	int steps, step = 1, oldstep = 0;
	myfb_t *fb;

	/* ---------- parse args ----------- */
	i = 0;
	while (++i < argc) {
		if (argv[i][0] != '-') {
			/* not an option */
			break;
		}
		if (argv[i][2]) {
			/* not a single letter option */
			usage(*argv, "Unknown option");
		}
		switch(argv[i][1]) {
		case 't':
			if (++i >= argc) {
				usage(*argv, "-v <vt> argument missing");
			}
			vt = atoi(argv[i]);
			if (vt < 1 || vt > 10) {
				usage(*argv, "Illegal console/VT number");
			}
			Options.use_vt = 1;
			break;
		case 'i':
			if (++i >= argc) {
				usage(*argv, "-i <initial step> argument missing");
			}
			step = atoi(argv[i]);
			/* error checking done later */
			break;
		case 'f':
			Options.use_fifo = 1;
			break;
		case 's':
			Options.use_fifo = 0;
			break;
		case 'v':
			Options.verbose = 1;
			break;
		case 'c':
			clear = 1;
			break;
		default:
			usage(*argv, "Unknown option");
			break;
		}
	}
	if (i+1 != argc) {
		usage(*argv, "Number of steps missing");
	}
	steps = atoi(argv[i]);
	if (steps < 1 || steps > 255) {
		usage(*argv, "Invalid number of steps");
	}
	if (step < 0 || step >= steps) {
		usage(*argv, "Invalid number of initial steps");
	}
	
	/* --------- setup signals + framebuffer ----------- */

	if (signal_init() < 0) {
		return -1;
	}
	
	vt = console_switch(vt);
	if (vt < 0) {
		return -1;
	}
	if (!(fb = fb_init())) {
		console_switch(vt);
		return -1;
	}
	if (fb->ht < PROGRESS_HEIGHT) {
		fprintf(stderr, "Error: progress bar higher (%d) than screen (%d)\n", PROGRESS_HEIGHT, fb->ht);
		fb_exit();
		console_switch(vt);
		return -1;
	}
	if (clear) {
		fb_clear();
	}
        draw_steps(oldstep, step, steps);
	fb_flush();

	/* ----- setup, read&progress and close progress fifo -------- */

	if (Options.use_fifo) {
		fd = fifo_open(PROGRESS_FIFO);
		if (fd < 0) {
			fb_exit();
			console_switch(vt);
			return -1;
		}
		printf("Opened fifo: " PROGRESS_FIFO "\n");
	}
	
	while (step < steps && !Options.do_exit) {

		if (Options.use_fifo) {
			count = fifo_read(fd);
			if (!count) {
				continue;
			}
			step += count;
			if (step > steps) {
				step = steps;
			}
		} else {
			/* without progress fifo, just advance
			 * the progress bar once a sec
			 */
			sleep(1);
			step++;
		}
		if (Fb.sig) {
			if (signal_process()) {
				/* need to redraw stuff on screen and
				 * wait a while so that the screen
				 * switcher has had time to draw its
				 * own stuff
				 */
				sleep(1);
				if (clear) {
					fb_clear();
				}
				oldstep = 0;
			}
		}
		if (draw_steps(oldstep, step, steps) < 0) {
			break;
		}
		fb_flush();

		if (Options.verbose) {
			printf("fb-progress: oldstep %d, newstep %d\n",
			       oldstep, step);
		}
		oldstep = step;
	}
	if (Options.use_fifo) {
		fifo_close(PROGRESS_FIFO, fd);
	}
	fb_exit();
	console_switch(vt);
	return 0;
}
