/*
 * This file is part of libosso-help-sdk
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Karoliina Salminen <karoliina.t.salminen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef OSSO_HELPLIB_PRIVATE_H
#define OSSO_HELPLIB_PRIVATE_H

#include "osso-helplib.h"

#include <stdio.h>

// Note: considered using 'g_get_tmp_dir()' but that would only
//       take a detour on some env.vars and default to "/tmp"
//       anyhow.
//
#define HELP_HTML_TMPFILE   "/tmp/osso-help.htm"

// Env.vars we observe:
//
#define OSSO_HELP_PATH  "OSSO_HELP_PATH"
#define OSSO_HELP_CSS   "OSSO_HELP_CSS"

// Default CSS filename (if OSSO_HELP_CSS not given):
//
#define DEFAULT_CSS   "help-default.css"

// Paths and extensions used for mapping <graphics filename=..>
// if the 'filename' param is relative (does not start with '/')
//
#define HELPLIB_PICTURE_PATHS   \
    "/usr/share/icons/hicolor/40x40/hildon/%s.png", \
    "/usr/share/icons/hicolor/34x34/hildon/%s.png", \
    "/usr/share/icons/hicolor/50x50/hildon/%s.png", \
    "/usr/share/icons/hicolor/scalable/hildon/%s.png", \
    "/usr/share/icons/hicolor/26x26/hildon/%s.png", \
    "/usr/share/osso-help/graphics/%s.png", \
    "/usr/share/icons/hicolor/26x26/osso/%s.png", \
    "/usr/share/icons/hicolor/scalable/osso/%s.png", \
    "/usr/share/icons/hicolor/40x40/osso/%s.png", \
    "/usr/share/icons/hicolor/26x26/osso/qgn_list_hw_button_%s.png", \
    "/usr/share/pixmaps/osso-mahjong/%s.png"   /*"undo_redo_buttons"*/

// Path used for the HTML embedded (go, web_link, zoom) icons
//tvh:12-July-05
//#define HELPLIB_ICO_PATH "/usr/share/icons/hicolor/26x26/osso/"
#define HELPLIB_ICO_PATH "/usr/share/icons/hicolor/26x26/hildon/"

// Note: The help keys ("/fname/a/[../][NN]") are generally very
//       short, but a long filename "controlpanelappletlanguageandregionalsettings.xml",
//       yes, such a help file actually exists... ?:o will be a problem.
//       The filename part in the above name is 45 chars.
//
#define HELP_KEY_MAXLEN 60

// Longer titles than this (if any?) simply get truncated
//
// AK 20-May-05: Raised the size to 80 for UTF-8 compatibility
//      (specs say max. 40 chars, Russian UTF-8 takes two bytes each)
//
#define HELP_TITLE_MAXLEN 80

#define HELP_KEY_OR_TITLE_MAXLEN  (HELP_KEY_MAXLEN)   // longer of the two

// Size of Help dialog (exterior)
//
#define HELP_DIALOG_WIDTH   480    // was: 450 (not fitting Italian title)
#define HELP_DIALOG_HEIGHT  280    // was: 250

// Alternative domain to be used in dialog
#define HELPUI_DOMAIN "osso-help"

// Timeout for waiting the Browser plugging
#define PLUG_TIMEOUT 10000

/*--- Private .so interface ---*/

const gchar *ossohelp_getpath( const gchar *language );

const gchar *ossohelp_getcss( void );

enum e_html_style {
    OSSOHELP_HTML_TOPIC,        // BCT + contents (normal)
    OSSOHELP_HTML_BCT,          // Just BCT
    OSSOHELP_HTML_DIALOG,       // no BCT, no title, no links, (almost) nothing :)
};

enum e_HelpStrId {
	HELP_UI_STR_CLOSE= 1,
	HELP_UI_STR_TITLE,
	HELP_UI_TAG_TIP,
	HELP_UI_TAG_NOTE,
	HELP_UI_TAG_EXAMPLE,
	HELP_UI_TAG_WARNING,
	HELP_UI_TAG_IMPORTANT
};

struct s_help_trail {
    char key[HELP_KEY_MAXLEN];
    char title[HELP_TITLE_MAXLEN];
};


/**
  Listing help topics available on the system, globally or for a specific
  branch (s.a. for a specific application) only.

  @param h_ref      Reference to an iteration handle (initially set to NULL)
  @param branch_id  Help ID of the branch to cover (NULL = global list) 

  @return   Help ID available on the system ("//fname/[...[/]]NN" string),
            NULL = end of list.

  @note     The returned Help ID pointer is valid only until the next 'next'
            call. If you need a permanent copy, you should make one.

  @note     The function only travers a certain folder level, not recursing
            into subfolders automatically. If you need to do that (= list
            an 'expanded' tree) just recurse your own functions whenever
            an id ending with '/' is received. 

  @see      'helptest.c' for a sample
            
  @code
        h_OssoHelpWalker h= NULL;
        const char *branch= "mail"; // (Help ID format may change)
        const char *id;
        <p>
        while( (id= ossohelp_next(&h, branch)) != NULL )
            {
            ...do something with 'id'...  
            recurse if you want if entry ends with '/'
            }
*/


typedef struct s_OssoHelpWalker *h_OssoHelpWalker;

const char *ossohelp_next( h_OssoHelpWalker *h_ref,
                           const char *branch_id );

#endif  // OSSO_HELPLIB_PRIVATE_H

