import sys, os, optparse, gettext
from linda import common, checks, output, eggs, parse_changes
from linda.common import dprint
_ = gettext.gettext

def format_callback(option, opt, value, parser):
    if value.lower() == 'help':
        print _("Output formats:")
        for x in output.registry.keys():
            if x == '':
                # Built-in, nothing to see here
                continue
            if output.registry[x].__doc__ is not None:
                docstr = output.registry[x].__doc__
            else:
                docstr = 'No doc string.'
            print "  - %s: %s" % (x, docstr)
        sys.exit(0)
    else:
        if output.registry.has_key(value):
            setattr(parser.values, option.dest, value) 
        else:
            print _("Linda: Output format %s not defined, falling back to normal.") % value

def unpack_callback(option, opt, value, parser):
    if value in (1, 2):
        setattr(parser.values, option.dest, value)
        setattr(parser.values, 'no_cull', 1)
    else:
        raise optparse.OptionValueError(_("unpack level must be 1 or 2"))

def config_callback(option, opt, value, parser):
    if os.path.isdir(value):
        setattr(parser.values, option.dest, os.path.abspath(value))
    else:
        raise optparse.OptionValueError(_("directory %s doesn't exist") % value)

def lab_root_callback(option, opt, value, parser):
    if os.path.isdir(value):
	    setattr(parser.values, option.dest, os.path.abspath(value))
    else:
        raise optparse.OptionValueError(_("directory %s doesn't exist") % value)

def list_checks_callback(option, opt, value, parser):
    if value in ('bin1', 'bin2', 'src1', 'src2', 'all'):
        setattr(parser.values, option.dest, value)
    else:
        raise optparse.OptionValueError(_("%s isn't a valid check list. Valid args are {bin,src}{1,2} or all.") % value)

def override_callback(option, opt, value, parser):
    if os.path.exists(os.path.abspath(value)):
        setattr(parser.values, option.dest, os.path.abspath(value))
    else:
        raise optparse.OptionValueError(_("%s can't be found.") % value)

class CmdLineParser:
    def __init__(self):
        class LindaParser(optparse.OptionParser):
            def print_version(self):
                print _("Linda, version VERSION")
                sys.exit(0)
            def error(self, msg):
                self.print_usage(sys.stderr)
                print _("%s: error: %s") % ('linda', msg)
                sys.exit(7)
        self.parser = LindaParser(usage=_('%prog [options] [.dsc|.deb|.udeb|.changes] ...'))
        self.parser.add_option('-c', '--config', action='callback', \
            type='string', dest='config', callback=config_callback, \
            help=_('specify config directory'))
        self.parser.add_option('-C', '--checks', dest='check_string', \
            help=_('specify checks to run'))
        self.parser.add_option('-d', '--debug', action='count', \
            dest='debug', help=_('set debug'))
        self.parser.add_option('-f', '--format', action='callback', \
            type='string', dest='format', callback=format_callback, \
            help=_('set output format, -f help shows them'))
        self.parser.add_option('-i', '--info', action='store_true', \
            dest='info', \
            help=_('show more information about errors and warnings'))
        self.parser.add_option('-I', '--informational', action='store_true', \
            dest='informational', help=_('also displays Informational tags'))
        self.parser.add_option('-l', '--lab', action='callback', \
            type='string', dest='lab_root', callback=lab_root_callback, \
            help=_('set directory lab should be in'))
        self.parser.add_option('-L', '--list-checks', action='callback', \
            dest='list_checks', type='string', callback=list_checks_callback, \
            help=_('list checks and exit'))
        self.parser.add_option('-m', '--more-overrides', action='callback', \
            dest='more_overrides', type='string', callback=override_callback, \
            help=_('parse overrides from another file'))
        self.parser.add_option('-n', '--no-cull', action='store_true', \
            dest='no_cull', help=_('don\'t delete the lab after tests are completed'))
        self.parser.add_option('-o', '--show-overridden', \
            action='store_true', dest='override_over', \
            help=_('display errors that are overridden'))
        self.parser.add_option('-p', '--print-overrides', \
            action='store_true', dest='print_over', \
            help=_('print parsed overrides and exit'))
        self.parser.add_option('-q', '--quiet', action='store_true', \
            dest='quiet', help=_('quieten me a little bit'))
        self.parser.add_option('-t', '--show-tag', action='store_true', \
            dest='show_tag', help=_('display tag as well'))
        self.parser.add_option('-T', '--traceback', action='store_true', \
            dest='printtrace', help=_('show a traceback from any exception'))
        self.parser.add_option('-u', '--unpack', action='callback', \
            callback=unpack_callback, dest='unpack_level', type='int', \
            help=_('set unpack level. implies --no-cull'))
        self.parser.add_option('-v', '--verbose', action='count', \
            dest='verbose', help=_('turn verbose mode on'))
        self.parser.add_option('-V', '--version', action='version', \
            help=_('display version information'))

    def parse_args(self):
        options, args = self.parser.parse_args()
        for x in common.options.keys():
            if x in ('files', 'archs', 'lab_directory', 'lang', 'quiet'):
                continue
            if getattr(options, x) is not None:
                common.options[x] = getattr(options, x)
        if not options.quiet:
            eggs.LindaEasterEggs()
        usable_files = filter(os.path.exists, args)
        for i in args:
            if i not in usable_files:
                print _("File %s doesn't exist, or not readable.") % i
        allowable_files = ('.deb', '.udeb', '.dsc')
        for i in usable_files:
            throw, keep = os.path.split(i)
            throw, ext = os.path.splitext(i)
            if ext in allowable_files:
                file = os.path.abspath(i)
                common.options['files'].append(file)
            if ext == '.deb':
                dprint(_("%s is a Debian archive.") % keep)
            elif ext == '.udeb':
                dprint(_("%s is a uDebian archive.") % keep)
            elif ext == '.changes':
                changes_path = os.path.abspath(i)
                keeper, tail = os.path.split(changes_path)
                tmp_files = parse_changes(i, keeper)
                if tmp_files:
                    common.options['files'].append(changes_path)
                for x in tmp_files:
                    throw, ch_ext = os.path.splitext(x)
                    if ch_ext in allowable_files:
                        file_keep, file_tail = os.path.split(x)
                        dprint(_("Adding file %s from %s.") % (file_tail, \
                            tail))
                        common.options['files'].append(x)
            elif ext == '.dsc':
                dprint(_("%s is a DSC.") % keep)
            else:
                print _("%s has an unknown extension %s.") % (keep, ext[1:])
        self.kill_checks()
        if common.options['list_checks']:
            checks.print_checks()
            if not common.options['files']:
                sys.exit(0)
        if not common.options['files']:
            print _("Linda: No valid files specified.")
            sys.exit(3)
        if options.debug:
            dprint(_("Lang: %s.") % common.options['lang'])
    def kill_checks(self):
        if common.options['check_string']:
            checks.cull_checks()
            no_check = 0
            for x in ('binary', 'source'):
                for y in (1, 2):
                    if not checks.registry[x][y]:
                        no_check += 1
            if no_check == 4 and not common.options['list_checks']:
                print _("Linda: No checks left, what's the point?")
                sys.exit(4)

