import sys, os, re, string, traceback, operator, gettext
from linda import common, output, overrides
from linda.common import file_to_packagename, file_to_version, dprint, \
    aps_not_valid, get_source_package_dir
_ = gettext.gettext
tmp_file_str = ''

class LindaChecker:
    def __init__(self, lab, file):
        self.lab = lab
        self.err_dict = ErrDict()
        self.file = file
        self.pkg_name = file_to_packagename(file)
        self.version = file_to_version(file)
        if self.version.find('-') != -1:
            self.isnative = 0
        else:
            self.isnative = 1
        tmp_src_dir = get_source_package_dir()
        if tmp_src_dir:
            self.pkg_dir = tmp_src_dir

    def signal_error(self, tag=None, message=[]):
        if tag is None:
            tag = self.tag
        self.err_dict.add_error(tag, message)

class ErrDict:
    def __init__(self):
        self.errs = {}

    def add_error(self, key, data=[]):
        if self.errs.has_key(key):
            # It already exists; append.
            self.errs[key].append(data)
        else:
            self.errs[key] = [data]

    def __getitem__(self, key): 
        return self.errs[key]

    def __delitem__(self, key): 
        del self.errs[key]

    def keys(self):
        return self.errs.keys()

registry = {}

def register(klass):
    global tmp_file_str
    klass.__file__ = tmp_file_str
    for type in ('source', 'binary', 'changes', 'udeb'):
        for level in (1, 2):
            if hasattr(klass, 'check_' + type + '_' + repr(level)):
                registry.setdefault(type, {}).setdefault(level, []).append\
                    (klass)

def apply(lab, file, type, level):
    if not registry.has_key(type):
        dprint(_("No checks registered with a type of %s.") % type)
        return None
    if not registry[type].has_key(level):
        dprint(_("No checks registered with a type of %s, and a level " +\
            "of %d.") % (type, level))
        return None
    for klass in registry[type][level]:
        dprint(_("Instantiating check: %s") % klass.__name__)
        instance = klass(lab, file)
        method = getattr(instance, 'check_' + type + '_' + repr(level))
        if not file.endswith('changes'):
            os.chdir(instance.lab)
        if hasattr(common.out_class, 'group_pre_print'):
            common.out_class.add_type(type, level)
            common.out_class.group_pre_print()
        dprint(_("Calling %s.%s().") % (klass.__name__, method.__name__))
        try:
            method()
        except KeyboardInterrupt:
            print _("Check %s interrupted.") % klass.__name__
        except StandardError:
            excptype, value = sys.exc_info()[:2]
            print _("Check %s failed. Exception %s thrown (%s).") % \
                (klass.__name__, excptype.__name__, value)
            if common.options['printtrace']:
                traceback.print_exc(file=sys.stdout)
        try:
            ep = PrintErrors(instance.__class__.__name__.lower(), \
                instance.err_dict, instance.pkg_name, instance)
            ep.print_error()
        except ErrorPrintingException:
            print _("Printing for %s failed. No desc found." % klass.__name__)
        except StandardError:
            excptype, value = sys.exc_info()[:2]
            print _("Output method %s failed. Exception %s thrown (%s).") % \
                (common.out_class.__class__.__name__, excptype.__name__, value)
            if common.options['printtrace']:
                traceback.print_exc(file=sys.stdout)

def register_all(dir):
    global tmp_file_str
    for file in os.listdir(dir):
        file = os.path.join(dir, file)
        if file.endswith('.py'):
            try:
                tmp_file_str = file
                local_checks = {'_': gettext.gettext, 'dprint': dprint, \
                    'aps_not_valid': aps_not_valid, 'os': os}
                execfile(file, local_checks)
            except StandardError:
                print _("Failed to import %s.") % file
                traceback.print_exc(file=sys.stdout)

def cull_checks():
    checks_to_run = common.options['check_string'].replace(' ', '').split(',')
    list_to_cull = []
    for pkg_type in ('binary', 'source'):
        for level in (1, 2):
            for x in registry[pkg_type][level]:
                class_str = x.__name__.lower()
                not_found = 0
                for y in checks_to_run:
                    if not re.match(y, class_str, re.IGNORECASE + re.DOTALL):
                        not_found += 1
                if not_found == len(checks_to_run):
                    list_to_cull.append(x)
            for x in list_to_cull:
                registry[pkg_type][level].remove(x)
            list_to_cull = []

def print_checks():
    mapping = {'bin': 'binary', 'src': 'source'}
    check_types = []
    check_levels = []
    if common.options['list_checks'] == 'all':
        check_types.extend(['binary', 'source'])
        check_levels.extend([1, 2])
    else:
        check_types.append(mapping[common.options['list_checks'][:-1]])
        check_levels.append(int(common.options['list_checks'][-1]))
    for pkg_type in check_types:
        for level in check_levels:
            if registry[pkg_type][level]:
                print "%s checks (level %d): " % (pkg_type.title(), level)
            for x in registry[pkg_type][level]:
                if x.__doc__ is not None:
                    docstr = x.__doc__
                else:
                    docstr = "No doc string."
                print " - %s: %s" % (x.__name__, docstr)

class PrintErrors:
    def __init__(self, file, errs, pkg_name, instance):
        if file.endswith('check'):
            self.file = file[:-5]
        else:
            self.file = file
        self.errs = errs
        self.pkg_name = pkg_name
        self.instance = instance
        try:
            self.output_var = self.parse_desc()
        except IOError:
            raise ErrorPrintingException("Failed to open desc file for %s" % \
                self.file)
    
    def set_desc(self, tag):
        if len(common.options['lang']) == 4:
            things_to_try = [common.options['lang'][0] + '_' + \
                common.options['lang'][1] + common.options['lang'][2] + \
                common.options['lang'][3], common.options['lang'][0] + '_' + \
                common.options['lang'][1], common.options['lang'][0] + \
                common.options['lang'][2] + common.options['lang'][3], \
                common.options['lang'][0]]
        elif len(common.options['lang']) == 2:
            things_to_try = [common.options['lang'][0] + '_' + \
                common.options['lang'][1], common.options['lang'][0]]
        elif len(common.options['lang']) == 1:
            things_to_try = [common.options['lang'][0]]
        dprint(_("Things to add onto Descriptions: %s.") % things_to_try, 2)
        for i in things_to_try:
            if self.output_var[tag].has_key('Description-' + i):
                dprint(_("Description finding: Found Description-%s.") % i, 2)
                return 'Description-' + i
        dprint(_("Description finding: Nothing found."), 2)
        return 'Description'

    def parse_desc(self):
        if self.instance.__file__:
            check_dir = os.path.split(os.path.split\
                (self.instance.__file__)[0])[0]
        else:
            check_dir = common.linda_root
        fp_desc_file = os.path.join(check_dir, 'desc', (self.file + '.desc'))
        output_var = {}
        f = open(fp_desc_file)
        desc_array = [[]]
        tmp_counter = 0
        for k in f.xreadlines():
            if k == '\n':
                tmp_counter += 1
                desc_array.append([])
                continue
            desc_array[tmp_counter].append(k)
        if desc_array[-1] == []:
            del desc_array[-1]
        f.close()
        to_del = []
        for x in range(len(desc_array)):
            seen = 0
            for y in desc_array[x]:
                if re.match(r'^(Tag:|Type:|Description:| )', y):
                    seen += 1
            if seen < 4:
                to_del.insert(0, x)
        for x in to_del:
            del desc_array[x]
            dprint(_("Ignoring broken stanza %d in desc file for %s.") % \
                ((x + 1), desc_file))
        # Stick the extended descriptions together.
        for x in desc_array:
            len_tmp_array = range(len(x))
            len_tmp_array.reverse()
            for y in len_tmp_array:
                if x[y].startswith(' '):
                    if x[y-1].startswith(' '):
                        x[y-1] = x[y-1] + x[y]
                        del x[y]
        for x in desc_array:
            cur_desc = ''
            for k in x:
                if k.find(':'):
                    cur_line = k.split(':')
                if k.startswith('Tag:'): 
                    tag = cur_line[1][:-1].strip()
                    output_var[tag] = {}
                elif k.startswith('Type:'):
                    output_var[tag]['Type'] = cur_line[1][:-1].strip()
                elif k.startswith('Description'):
                    cur_desc = cur_line[0]
                    output_var[tag][cur_desc] = []
                    # The only time cur_line is going to contain more than one
                    # element is when there is a ':' in the string.
                    output_var[tag][cur_desc].append(cur_line[1])
                    if len(cur_line) >= 2:
                        for z in cur_line[2:]:
                            output_var[tag][cur_desc][0] += ':' + z
                    output_var[tag][cur_desc][0] = \
                        output_var[tag][cur_desc][0][:-1].strip()
                    output_var[tag]['seen'] = 0
                elif k.startswith('Justification:'):
                    output_var[tag]['Justification'] = cur_line[1][:-1].strip()
                elif k.startswith(' '):
                    output_var[tag][cur_desc].append(k)
                elif k.startswith('#'):
                    continue
        dprint(_("Desc file (%s): %s") % (self.file, output_var), 3)
        return output_var

    def print_error(self):
        sorted_keys = self.errs.keys()
        sorted_keys.sort()
        for x in sorted_keys:
            ov_ret = 0
            if self.output_var.has_key(x):
                if common.overrides.has_key(x) and not \
                    common.options['override_over']:
                    ov_ret = overrides.check_overrides(self.errs[x], \
                        self.output_var[x], x)
                    if ov_ret:
                        continue
            else:
                dprint(_("No such tag %s in %s.desc") % (x, file))
                continue
            if self.output_var[x]['Type'] == 'Informational' and not \
                common.options['informational']:
                dprint(_("%s is Informational, skipping.") % x)
                continue
            desc_str = 'Description'
            self.errs[x].sort()
            for y in self.errs[x]:
                desc_str = self.set_desc(x)
                formatarg = len(re.findall(r'(?<!%)%\d*[asdful]', \
                    self.output_var[x][desc_str][0]))
                if formatarg != len(y):
                    dprint(_("Format args for %s don't match Description. " +\
                        "(%d vs %d)") % (x, formatarg, len(y)))
                    continue
                tmp_tuple = (x, y, self.output_var[x], self.pkg_name, desc_str)
                output.show_error(common.options['format'], tmp_tuple)
            if common.options['info'] and self.output_var[x][desc_str][1] \
                and not self.output_var[x]['seen']: 
                self.output_var[x]['seen'] = 1
                print self.output_var[x][desc_str][1][:-1]
                if self.output_var[x].has_key('Justification'):
                    print " Justification: %s" % \
                        self.output_var[x]['Justification']
        if hasattr(common.out_class, 'group_post_print'):
            common.out_class.group_post_print()

class ErrorPrintingException(Exception):
    pass

