/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Christopher Blizzard. Portions created by Christopher Blizzard are Copyright (C) Christopher Blizzard.  All Rights Reserved.
 * Portions created by the Initial Developer are Copyright (C) 2001
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Christopher Blizzard <blizzard@mozilla.org>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "EmbedProgress.h"

#include <nsXPIDLString.h>
#include <nsIChannel.h>
#include <nsIWebProgress.h>
#include <nsIDOMWindow.h>

// TONIKITOO
#include <nsString.h>
#include <nsStringAPI.h>
#include <nsEmbedString.h>

#include "nsIURI.h"
#include "nsCRT.h"


PRInt32 aMaxTotalProgress;

EmbedProgress::EmbedProgress(void)
{
  mOwner = nsnull;
  isFramed = FALSE;
}

EmbedProgress::~EmbedProgress()
{
}

NS_IMPL_ISUPPORTS2(EmbedProgress,
		   nsIWebProgressListener,
		   nsISupportsWeakReference)

nsresult
EmbedProgress::Init(EmbedPrivate *aOwner)
{
  mOwner = aOwner;
  return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnStateChange(nsIWebProgress *aWebProgress,
			     nsIRequest     *aRequest,
			     PRUint32        aStateFlags,
			     nsresult        aStatus)
{
	// give the widget a chance to attach any listeners
	mOwner->ContentStateChange();

	// get the uri for this request
	nsXPIDLCString uriString;
	RequestToURIString(aRequest, getter_Copies(uriString));
	nsString tmpString;
	CopyUTF8toUTF16(uriString, tmpString);

	// if we've got the start flag, emit the signal
	if ((aStateFlags & GTK_MOZ_EMBED_FLAG_IS_NETWORK) && 
	(aStateFlags & GTK_MOZ_EMBED_FLAG_START))
	{
		gtk_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
			moz_embed_signals[START_LOADING], (const char *) uriString);
	}

	// is it the same as the current URI?
	if (mOwner->mURI.Equals(tmpString))
	{
	// for people who know what they are doing
	gtk_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
			moz_embed_signals[NET_STATE],
			aStateFlags, aStatus);
	}
	gtk_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
			moz_embed_signals[NET_STATE_ALL],
			(const char *) uriString,
			(gint)aStateFlags, (gint)aStatus);
	// and for stop, too
	if ((aStateFlags & GTK_MOZ_EMBED_FLAG_IS_NETWORK) && 
	(aStateFlags & GTK_MOZ_EMBED_FLAG_STOP))
	{

		// TONIKITOO: IMPLEMENTING THE FINISH_STATUS (???) and HAS_FRAMES (FROM BIA) !!!!!
		// Make sure that this is the primary frame got finished and not just a subframe.  
		if (aWebProgress) {

			nsCOMPtr<nsIDOMWindow> domWindow;
			nsCOMPtr<nsIDOMWindow> topDomWindow;

			aWebProgress->GetDOMWindow(getter_AddRefs(domWindow));

			// get the root dom window
			if (domWindow)
				domWindow->GetTop(getter_AddRefs(topDomWindow));

			if (domWindow != topDomWindow)
				isFramed = PR_TRUE;

			else isFramed = PR_FALSE;
		}

                int pageSize = GetPageSize (mOwner->mOwningWidget);
		
		gtk_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
			moz_embed_signals[FINISHED_LOADING], 0, pageSize, isFramed);

		// let our owner know that the load finished
		mOwner->ContentFinishedLoading();

		// getting the encoding
		int curEncoding;
		curEncoding = GetEncoding (mOwner->mOwningWidget);
		printf ("[DEBUGING] Encoding: %d\n", curEncoding);

		gtk_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
			moz_embed_signals[ENCODING],
			curEncoding);

		}

	return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnProgressChange(nsIWebProgress *aWebProgress,
				nsIRequest     *aRequest,
				PRInt32         aCurSelfProgress,
				PRInt32         aMaxSelfProgress,
				PRInt32         aCurTotalProgress,
				PRInt32         aMaxTotalProgress)
{

	nsXPIDLCString uriString;
	RequestToURIString(aRequest, getter_Copies(uriString));

	nsString tmpString;
	CopyUTF8toUTF16(uriString, tmpString);

	// is it the same as the current uri?
	if (mOwner->mURI.Equals(tmpString)) {

		// TONIKITOO -> Trigger, they get some negative values sometimes (-1)
		if (aMaxTotalProgress < 0) aMaxTotalProgress = 1000;
		if (aCurTotalProgress < 0) aCurTotalProgress = 0;

		gtk_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
				moz_embed_signals[LOADING_PROGRESS],
				aCurTotalProgress, aMaxTotalProgress);
	}

	gtk_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
			moz_embed_signals[PROGRESS_ALL],
			(const char *)uriString,
			aCurTotalProgress, aMaxTotalProgress);

	::aMaxTotalProgress = aMaxTotalProgress;

	return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnLocationChange(nsIWebProgress *aWebProgress,
				nsIRequest     *aRequest,
				nsIURI         *aLocation)
{
  nsCAutoString newURI;
  NS_ENSURE_ARG_POINTER(aLocation);
  aLocation->GetSpec(newURI);

  // Make sure that this is the primary frame change and not
  // just a subframe.
  PRBool isSubFrameLoad = PR_FALSE;
  if (aWebProgress) {
    nsCOMPtr<nsIDOMWindow> domWindow;
    nsCOMPtr<nsIDOMWindow> topDomWindow;

    aWebProgress->GetDOMWindow(getter_AddRefs(domWindow));

    // get the root dom window
    if (domWindow)
      domWindow->GetTop(getter_AddRefs(topDomWindow));

    if (domWindow != topDomWindow)
      isSubFrameLoad = PR_TRUE;
  }

  if (!isSubFrameLoad) {
    mOwner->SetURI(newURI.get());
    gtk_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
		    moz_embed_signals[URL_CHANGED], newURI.get());
  }

  return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnStatusChange(nsIWebProgress  *aWebProgress,
			      nsIRequest      *aRequest,
			      nsresult         aStatus,
			      const PRUnichar *aMessage)
{
  // need to make a copy so we can safely cast to a void *
  PRUnichar *tmpString = nsCRT::strdup(aMessage);

  gtk_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
		  moz_embed_signals[STATUS_CHANGE],
		  NS_STATIC_CAST(void *, aRequest),
		  NS_STATIC_CAST(int, aStatus),
		  NS_STATIC_CAST(void *, tmpString));

  nsMemory::Free(tmpString);

  return NS_OK;
}

NS_IMETHODIMP
EmbedProgress::OnSecurityChange(nsIWebProgress *aWebProgress,
				nsIRequest     *aRequest,
				PRUint32         aState)
{

	printf ("[DEBUGING] OnSecurityChange = '%d'\n", aState);

	gtk_signal_emit(GTK_OBJECT(mOwner->mOwningWidget),
		  moz_embed_signals[SECURITY_MODE_CHANGED],
		  NS_STATIC_CAST(void *, aRequest),
		  aState);
	return NS_OK;
}

/* static */
void
EmbedProgress::RequestToURIString(nsIRequest *aRequest, char **aString)
{
  // is it a channel
  nsCOMPtr<nsIChannel> channel;
  channel = do_QueryInterface(aRequest);
  if (!channel)
    return;
  
  nsCOMPtr<nsIURI> uri;
  channel->GetURI(getter_AddRefs(uri));
  if (!uri)
    return;
  
  nsCAutoString uriString;
  uri->GetSpec(uriString);

  *aString = strdup(uriString.get());
}

int
EmbedProgress::GetEncoding (GtkMozEmbed *engine) {
 
        char *charset = nsnull;
        nsIWebBrowser *webBrowser = nsnull;
 
        gtk_moz_embed_get_nsIWebBrowser (GTK_MOZ_EMBED(engine), &webBrowser);
        nsCOMPtr<nsIDocCharset> docCharset = do_GetInterface (webBrowser);
 
        docCharset->GetCharset (&charset);
 
        if (!charset)
		return 0;

	printf("[DEBUGING] Charset: %s\n", charset);
 
        //G_WEBENGINE_ENCODING_AUTOMATIC, 0
        if (!strcmp(charset,"UTF-8")) return 1;
        if (!strcmp(charset,"UTF-16")) return 2;
        if (!strcmp(charset,"US-ASCII")) return 3;
        if (!strcmp(charset,"ISO-8859-1")) return 4;
        if (!strcmp(charset, "ISO-8859-2")) return 5;
        if (!strcmp(charset, "ISO-8859-3")) return 6;
        if (!strcmp(charset, "ISO-8859-4")) return 7;
        if (!strcmp(charset, "ISO-8859-5")) return 8;
        if (!strcmp(charset, "ISO-8859-6")) return 9;
        if (!strcmp(charset, "ISO-8859-7")) return 10;
        if (!strcmp(charset, "ISO-8859-8")) return 11;
        if (!strcmp(charset, "ISO-8859-9")) return 12;
        if (!strcmp(charset, "ISO-8859-10")) return 13;
        if (!strcmp(charset, "ISO-8859-11")) return 14;
        if (!strcmp(charset, "ISO-8859-13")) return 15;
        if (!strcmp(charset, "ISO-8859-14")) return 16;
        if (!strcmp(charset, "ISO-8859-15")) return 17;
        if (!strcmp(charset, "KOI-8-R")) return 18;
        if (!strcmp(charset, "KOI-8-U")) return 19;
        if (!strcmp(charset, "CP-1250")) return 20;
        if (!strcmp(charset, "CP-1251")) return 21;
        if (!strcmp(charset, "CP-1252")) return 22;
        if (!strcmp(charset, "CP-1253")) return 23;
        if (!strcmp(charset, "CP-1254")) return 24;
        if (!strcmp(charset, "CP-1255")) return 25;
        if (!strcmp(charset, "CP-1256")) return 26;
        if (!strcmp(charset, "CP-1257")) return 27;
        if (!strcmp(charset, "CP-1258")) return 28;
        if (!strcmp(charset, "Shift_JIS")) return 29;
        if (!strcmp(charset, "ISO-2022-JP")) return 30;
        if (!strcmp(charset, "BIG-5")) return 31;
        if (!strcmp(charset, "EUC-JP")) return 32;
        if (!strcmp(charset, "GB-2312")) return 33;
        if (!strcmp(charset, "VISCII")) return 34;
        if (!strcmp(charset, "EUC-KR")) return 35;
        if (!strcmp(charset, "HZ-GB-2312")) return 36;
        if (!strcmp(charset, "GBK")) return 37;
        if (!strcmp(charset, "EUC-TW")) return 38;
        if (!strcmp(charset, "WS2")) return 39;
        if (!strcmp(charset, "IBM-866")) return 40;
        if (!strcmp(charset, "TCVN")) return 41;
        if (!strcmp(charset, "VPS")) return 42;
        else return 0;
//        G_WEBENGINE_ENCODING_JAPANESE_AUTOMATIC,
//        G_WEBENGINE_ENCODING_CHINESE_AUTOMATIC
}

nsresult
EmbedProgress::GetCacheEntryDescriptor(const nsAString &aKey, nsICacheEntryDescriptor **aCacheEntryDescriptor){
 
        nsresult rv = NS_OK;
 
        nsCOMPtr<nsICacheService> cacheService = do_GetService(NS_CACHESERVICE_CONTRACTID);
        NS_ENSURE_TRUE (cacheService, NS_ERROR_FAILURE);
        char *url = g_strdup (GulCString (aKey).get());
        g_strdelimit (url, "#", '\0'); /* snip fragment, see bug #161201 */
 
        const char *cacheTypes[] = { "HTTP", "FTP" };
        for (unsigned int i = 0 ; i < G_N_ELEMENTS (cacheTypes); i++)
        {
                nsCOMPtr<nsICacheSession> cacheSession;
                cacheService->CreateSession(cacheTypes[i],
                                            nsICache::STORE_ANYWHERE,
                                            PR_TRUE,
                                            getter_AddRefs(cacheSession));
                NS_ENSURE_TRUE (cacheSession, NS_ERROR_FAILURE);
 
                cacheSession->SetDoomEntriesIfExpired(PR_FALSE);
                
                nsCOMPtr<nsICacheEntryDescriptor> cacheEntryDescriptor;
 
                rv = cacheSession->OpenCacheEntry(GulDependentCString(url),
                                                  1,PR_FALSE,  aCacheEntryDescriptor);
              if (NS_SUCCEEDED (rv)) break;
        }
        g_free (url);
        return rv;
}
 
 
int
EmbedProgress::GetPageSize (GtkMozEmbed *engine) {
 
        GulString url;
        
        PRUint32 dataSize=0;
 
        nsIWebBrowser *webBrowser = nsnull;
 
        gtk_moz_embed_get_nsIWebBrowser (GTK_MOZ_EMBED(engine), &webBrowser);
 
        nsCOMPtr<nsIDOMWindow> DOMWindow;
        webBrowser->GetContentDOMWindow (getter_AddRefs (DOMWindow));
 
        nsCOMPtr<nsIDOMDocument> domDocument;
        DOMWindow->GetDocument (getter_AddRefs (domDocument));
 
        nsCOMPtr<nsIDOMNSDocument> doc = do_QueryInterface(domDocument);
        NS_ENSURE_TRUE (doc, NS_ERROR_FAILURE);
 
        nsCOMPtr<nsIDOMLocation> DOMLocation;
        doc->GetLocation (getter_AddRefs (DOMLocation));
        NS_ENSURE_TRUE (DOMLocation, NS_ERROR_FAILURE);
 
        DOMLocation->ToString (url);
 
        nsCOMPtr<nsICacheEntryDescriptor> cacheEntryDescriptor;
        GetCacheEntryDescriptor(url, getter_AddRefs(cacheEntryDescriptor));
 
        if (cacheEntryDescriptor)
        {
                cacheEntryDescriptor->GetDataSize (&dataSize);
        }
        return dataSize;
}


//TONIKITOO
/*nsresult 
EmbedProgress::VerifyIfFramed (GtkMozEmbed *embed, nsIDOMNode *node, PRBool *framed)
{
	nsresult result;

	nsCOMPtr<nsIWebBrowser> wb;
	g_return_val_if_fail(embed != NULL, FALSE);
printf ("2\n");
	gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed),getter_AddRefs(wb));
	g_return_val_if_fail(wb != NULL, FALSE);

	nsCOMPtr<nsIDOMDocument> mainDocument;
	result = GetMainDocument (wb, getter_AddRefs(mainDocument));
	if (NS_FAILED(result) || !mainDocument) return NS_ERROR_FAILURE;
printf ("3\n");	

	///////////////

	if (!node) return NS_ERROR_FAILURE;
printf ("1\n"); 
	nsCOMPtr<nsIDOMDocument> nodeDocument;
	result = node->GetOwnerDocument (getter_AddRefs(nodeDocument));
	if (NS_FAILED(result) || !nodeDocument) return NS_ERROR_FAILURE;
printf ("4\n"); 

	///////////////
	*framed = (mainDocument != nodeDocument);

        return NS_OK;
}

nsresult 
EmbedProgress::GetMainDocument (nsIWebBrowser *mWebBrowser, nsIDOMDocument **aDOMDocument)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = 
		GetDocShell (mWebBrowser, getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	return contentViewer->GetDOMDocument (aDOMDocument);
}

nsresult 
EmbedProgress::GetDocShell (nsIWebBrowser *mWebBrowser, nsIDocShell **aDocShell)
{
        nsCOMPtr<nsIDocShellTreeItem> browserAsItem;
        browserAsItem = do_QueryInterface(mWebBrowser);
        if (!browserAsItem) return NS_ERROR_FAILURE;

        // get the owner for that item
        nsCOMPtr<nsIDocShellTreeOwner> treeOwner;
        browserAsItem->GetTreeOwner(getter_AddRefs(treeOwner));
        if (!treeOwner) return NS_ERROR_FAILURE;

        // get the primary content shell as an item
        nsCOMPtr<nsIDocShellTreeItem> contentItem;
        treeOwner->GetPrimaryContentShell(getter_AddRefs(contentItem));
        if (!contentItem) return NS_ERROR_FAILURE;

        // QI that back to a docshell
        nsCOMPtr<nsIDocShell> DocShell;
        DocShell = do_QueryInterface(contentItem);
        if (!DocShell) return NS_ERROR_FAILURE;

        *aDocShell = DocShell.get();

        NS_IF_ADDREF(*aDocShell);
        
        return NS_OK;
}
*/
