/* Copyright (C) 2003 Jamey Sharp.
 * This file is licensed under the MIT license. See the file COPYING. */

#define _GNU_SOURCE /* for PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP */
#include <features.h>

#include "Xlibint.h"
#include "locking.h"
#include "xclint.h"
#include <X11/XCB/xcbint.h>

#include <pthread.h>

static void _XLockDisplay(Display *dpy)
{
    pthread_mutex_lock(&XCBConnectionOfDisplay(dpy)->iolock);
    _XGetXCBBufferIf(dpy, _XBufferUnlocked);
    ++dpy->xcl->lock_count;
}

void XLockDisplay(Display* dpy)
{
    LockDisplay(dpy);
    /* We want the threads in the reply queue to all get out before
     * XLockDisplay returns, in case they have any side effects the
     * caller of XLockDisplay was trying to protect against.
     * XLockDisplay puts itself at the head of the event waiters queue
     * to wait for all the replies to come in.
     * TODO: Restore this behavior on XCB.
     */
}

static void _XUnlockDisplay(Display *dpy)
{
    --dpy->xcl->lock_count;
    _XPutXCBBufferIf(dpy, _XBufferUnlocked);
    pthread_mutex_unlock(&XCBConnectionOfDisplay(dpy)->iolock);
}

void XUnlockDisplay(Display* dpy)
{
    UnlockDisplay(dpy);
}

/* returns 0 if initialized ok, -1 if unable to allocate
   a mutex or other memory */
int _XInitDisplayLock(Display *dpy)
{
    pthread_mutex_t lock = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;
    XCBConnectionOfDisplay(dpy)->iolock = lock;

    dpy->lock_fns = (struct _XLockPtrs*)Xmalloc(sizeof(struct _XLockPtrs));
    if (dpy->lock_fns == NULL)
	return -1;

    dpy->lock = 0;
    dpy->lock_fns->lock_display = _XLockDisplay;
    dpy->lock_fns->unlock_display = _XUnlockDisplay;

    return 0;
}

void _XGetXCBBuffer(Display *dpy)
{
    static const xReq dummy_request;

    XCBConnection *c = XCBConnectionOfDisplay(dpy);
    _xcb_out_force_sequence_wrap(c);

    dpy->bufptr = dpy->buffer = c->out.queue + c->out.queue_len;
    dpy->bufmax = c->out.queue + sizeof(c->out.queue);

    if(c->out.last_request)
	dpy->last_req = c->out.last_request;
    else
	dpy->last_req = (char *) &dummy_request;

    dpy->request = c->out.request;
    dpy->last_request_read = c->in.request_read;
}

void _XPutXCBBuffer(Display *dpy)
{
    XCBConnection *c = XCBConnectionOfDisplay(dpy);
    int len = dpy->bufptr - dpy->buffer;

    if(len)
    {
	struct iovec iov;
	c->out.queue_len += len;
	iov.iov_base = (caddr_t) dpy->buffer;
	iov.iov_len = len;
	_XBeforeFlush(dpy, &iov);
    }

    /* crash applications that use these without locking the display */
    dpy->last_req = dpy->buffer = dpy->bufptr = dpy->bufmax = 0;

    c->out.request = dpy->request;
}

/*  */

void _XGetXCBBufferIf(Display *dpy, enum _XBufferCondition locked)
{
    if(!dpy->xcl->lock_count)
	locked = !locked;
    if(locked)
	_XGetXCBBuffer(dpy);
}

void _XPutXCBBufferIf(Display *dpy, enum _XBufferCondition locked)
{
    if(!dpy->xcl->lock_count)
	locked = !locked;
    if(locked)
	_XPutXCBBuffer(dpy);
}
