/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2004 Nokia. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <gw-obex.h>
#include <bt-dbus.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus-glib-lowlevel.h>

#include "om-dbus.h"

#define d(x) 

/* Gwcond communication. Talks to gwcond to get the device.
 *
 * This API can be accessed from any thread, since it creates it's own context
 * and uses that for the dbus connection.
 */

typedef struct {
	DBusConnection *dbus_conn;
	GMainContext   *context;
	GMainLoop      *loop;
} Connection;

static Connection *
get_gwcond_connection (void)
{
	DBusConnection *dbus_conn;
	Connection     *conn;
	DBusError       error;
        const gchar    *address; 	

	dbus_error_init (&error);

	/* NOTE: We are not using dbus_bus_get here, for the reason that need to
	 * get our own private dbus connection to avoid threading problems with
	 * other libraries or applications that use this module and dbus (the
	 * vfs daemon in particular).
	 */
        address = g_getenv ("DBUS_SYSTEM_BUS_ADDRESS");
       
        /* Provide a fallback to the default location. */
        if (!address) {
		address = "unix:path=/var/run/dbus/system_bus_socket";
                /*address = "unix:path=/opt/gnome2/var/run/dbus/system_bus_socket";*/
        }

        if (!address) {
		g_warning ("Couldn't get the address for the system bus.");
                return NULL;
        }
	
	dbus_error_init (&error);
        dbus_conn = dbus_connection_open (address, &error);	
	
	if (!dbus_conn) {
		g_warning ("Failed to connect to the D-BUS daemon: %s", error.message);
		
		dbus_error_free (&error);
		return NULL;
	}

	if (!dbus_bus_register (dbus_conn, &error)) {
                g_warning ("Failed to register with the D-BUS daemon: %s", error.message);
		
                dbus_connection_disconnect (dbus_conn);
                dbus_connection_unref (dbus_conn);

                dbus_error_free (&error);
                return NULL;
        } 
	
	conn = g_new0 (Connection, 1);
	
	conn->context = g_main_context_new ();
	conn->loop = g_main_loop_new (conn->context, FALSE);

	conn->dbus_conn = dbus_conn;

	dbus_connection_setup_with_g_main (dbus_conn, conn->context);
	
	return conn;
}

static void
connection_free (Connection *conn)
{
	dbus_connection_disconnect (conn->dbus_conn);
	dbus_connection_unref (conn->dbus_conn);
	
	g_main_loop_unref (conn->loop);
	g_main_context_unref (conn->context);
	
	g_free (conn);
}

static gchar *
get_dev (Connection  *conn,
	 const gchar *bda,
	 const gchar *profile,
	 gint        *error)
{
	gint         len, i, j;
	gchar        bda_formatted[18];
	DBusMessage *message;
	DBusError    dbus_error;
	DBusMessage *reply;
	gboolean     ret;
	gchar       *str;
	gchar       *dev;

	/* Format the BDA to a 00:00:00:00:00:00 format, but first do some
	 * sanity checking.
	 */
	len = strlen (bda);
	if (len != 12) {
		*error = GW_OBEX_ERROR_NO_FTP;
		return NULL;
	}

	j = 0;
	for (i = 0; i < len; i++) {
		bda_formatted[j++] = bda[i];

		if (i % 2 && j < 17) {
			bda_formatted[j++] = ':';
		}
	}

	/* Terminate the string. */
	bda_formatted[17] = 0;
	
	message = dbus_message_new_method_call (BTCOND_SERVICE,
						BTCOND_REQ_PATH,
						BTCOND_REQ_INTERFACE,
						BTCOND_GET_RFCOMM_DEV_REQ);
	if (!message) {
		g_error ("Out of memory");
	}
	
	if (!dbus_message_append_args (message,
				       DBUS_TYPE_STRING, bda_formatted,
				       DBUS_TYPE_STRING, profile,
				       DBUS_TYPE_INVALID)) {
		g_error ("Out of memory");
	}

	dbus_error_init (&dbus_error);
	reply = dbus_connection_send_with_reply_and_block (conn->dbus_conn, message, -1, &dbus_error);
	
	dbus_message_unref (message);

	if (dbus_error_is_set (&dbus_error)) {
		if (strcmp (dbus_error.name, BTCOND_ERROR_INVALID_SVC) == 0) {
			d(g_print ("Invalid SDP profile.\n"));
		}
		if (strcmp (dbus_error.name, BTCOND_ERROR_INVALID_DEV) == 0) {
			d(g_print ("Invalid BDA.\n"));
		}
		else if (strcmp (dbus_error.name, BTCOND_ERROR_DISCONNECTED) == 0) {
			d(g_print ("GW not connected.\n"));
		}
		else if (strcmp (dbus_error.name, BTCOND_ERROR_NO_DEV_INFO) == 0) {
			d(g_print ("No dev info.\n"));
		}
		else if (strcmp (dbus_error.name, "org.freedesktop.DBus.Error.ServiceDoesNotExist") == 0) {
			d(g_print ("Gwcond not running.\n"));
		} else {
			d(g_print ("Error: %s, %s\n", dbus_error.name, dbus_error.message));
		}

		/* Can we do better here with the error mapping? */
		*error = GW_OBEX_ERROR_NO_FTP;

		dbus_error_free (&dbus_error);
		return NULL;
	}
	
	if (!reply) {
		*error = GW_OBEX_ERROR_NO_FTP;
		return NULL;
	}

	ret = dbus_message_get_args (reply, NULL,
				     DBUS_TYPE_STRING, &str,
				     DBUS_TYPE_INVALID);

	dbus_message_unref (reply);
	
	if (!ret) {
		*error = GW_OBEX_ERROR_NO_FTP;
		return NULL;
	}
	
	dev = g_strdup (str);
	dbus_free (str);
	
	*error = 0;
	return dev;
}

gchar *
om_dbus_get_dev (const gchar *bda, gint *error)
{
	Connection *conn;
	gchar      *dev;

	conn = get_gwcond_connection ();

	/* Try NFTP first, which appears to be some vendor specific profile. */
	dev = get_dev (conn, bda, "NFTP", error);

	if (!dev) {
		*error = 0;
		dev = get_dev (conn, bda, "FTP", error);
	}
	
	connection_free (conn);
	
	return dev;
}

