/**************************************************************************************
Project:                OSSO

File Name:              osso_case_in.c

Version:                1.0

Date:                   28/04/2004

Author / Copyright:     NOKIA

**************************************************************************************/

#include "stdio.h"
#include "string.h"

#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnomevfs/gnome-vfs-init.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <glib.h>

#include "osso_case_in.h"
#define DEBUG
#define BUFFSIZE 8192

/**
  This function retrives the actual existing and most matching file path. 
  It does so in a case insensitive manner. The input paramter path is 
  updated with the existing path.

  @param path 	Path of input file name. The updated file name with existing 
  		character case is updated in path itself.
  @return 	0 on successful identification of match, 1 on failure.
*/
int osso_get_existing_path (gchar* path)
{
	
	gchar* path_ptr = path;
	gchar* directory = (gchar*) g_new0 (gchar, MAX_LEN);
	gchar* uri = (gchar*) g_new0 (gchar, MAX_LEN);
	gchar* tempuri = uri;
	gint   return_val = OSSO_CASE_PATH_DOESNOT_EXIST;

	if (g_utf8_validate (DELIMITERSTR, -1, NULL) == FALSE){
		return_val = OSSO_CASE_DELIMITERSTR_ERROR; //invalid utf8 Delimiterstr
	}
	if (g_utf8_validate (CURRENT_DIR, -1, NULL) == FALSE){
		return_val = OSSO_CASE_CURRENTDIR_ERROR; //invalid utf8 current directory
	}
	
	if(*path == DELIMITER){
		debug_print("beginning: path==DELIMITER\n");
		g_utf8_strncpy (directory, DELIMITERSTR, g_utf8_strlen (DELIMITERSTR, -1));
		path = g_utf8_offset_to_pointer (path, g_utf8_strlen(DELIMITERSTR, -1));
	}
	else{
		g_utf8_strncpy (directory, CURRENT_DIR, g_utf8_strlen (CURRENT_DIR, -1));
	}
	
	debug_print("before while: path=%s,directory=%s,tempuri=%s,uri=%s\n", path, directory,tempuri,uri);
	while ((*tempuri = *path) != 0){
		path++;
		tempuri++;
		if(*path == DELIMITER){
			*tempuri = '\0';
			debug_print("path==DELIMITER-1: path=%s,directory=%s,tempuri=%s,uri=%s\n", path, directory,tempuri,uri);
			if(osso_get_existing_file (uri, directory)){
				g_utf8_strncpy (directory, "", 
						g_utf8_strlen ("", -1));
				break;
			}
			strcat(directory, uri);
			strcat(directory, DELIMITERSTR);
			tempuri = uri;
			path++;
			debug_print("path==DELIMITER-2: path=%s,directory=%s,tempuri=%s,uri=%s\n", path, directory,tempuri,uri);
		}
		else if (*path == '\0'){
			*tempuri = '\0';
			debug_print("path==0-1: path=%s,directory=%s,tempuri=%s,uri=%s\n", path, directory,tempuri,uri);
			if(osso_get_existing_file (uri, directory)){
				g_utf8_strncpy (directory, "", 
						g_utf8_strlen ("", -1));
				break;
			}
			strcat(directory,uri);
			tempuri = uri;
			debug_print("path==0-2: path=%s,directory=%s,tempuri=%s,uri=%s\n", path, directory,tempuri,uri);
		}
	}
	
	path = path_ptr;

	if(g_utf8_collate (directory, "")){
		g_utf8_strncpy (path, directory, 
				g_utf8_strlen (directory, -1));
		return_val = OSSO_CASE_PATH_EXIST;
	}
	
	debug_print("exiting: path=%s, directory=%s\n", path, directory);
	
	g_free (directory);
	g_free (uri);
	
	return return_val;
}

/**
  This function retrives the actual existing and most matching file path. 
  It does so in a case insensitive manner. The input paramter path is updated 
  with the existing path.
  @param path 		Path of input file name. The updated file name with 
			existing charecter case is updated in path itself.
  @param existing_path 	if the path doesnot exist, then the existing path till the
			match existed is returned else NULL is returned. The caller
			method must allocate and free memory for this parameter. 
  @param offset		the span of matching path in 'path' 	
  @return 		0 on successful identification of match, 1 on failure.
*/
int osso_get_existing_path_offset (gchar* path, gchar *existing_path, int *offset){
	
	gchar* path_ptr = path;
	gchar* directory = (gchar*) g_new0 (gchar, MAX_LEN);
	gchar* uri = (gchar*) g_new0 (gchar, MAX_LEN);
	gchar* tempuri = uri;
	gint   return_val = OSSO_CASE_PATH_DOESNOT_EXIST;
	gboolean found_match = TRUE;
	gchar* remaining_path = (gchar*) g_new0 (gchar, MAX_LEN);
	
	if (g_utf8_validate (DELIMITERSTR, -1, NULL) == FALSE){
		return_val = OSSO_CASE_DELIMITERSTR_ERROR; //invalid utf8 Delimiterstr
	}
	if (g_utf8_validate (CURRENT_DIR, -1, NULL) == FALSE){
		return_val = OSSO_CASE_CURRENTDIR_ERROR; //invalid utf8 current directory
	}
	
	if(*path == DELIMITER){
		debug_print("beginning: path==DELIMITER\n");
		g_utf8_strncpy (directory, DELIMITERSTR, g_utf8_strlen (DELIMITERSTR, -1));
		path = g_utf8_offset_to_pointer (path, g_utf8_strlen(DELIMITERSTR, -1));
	}
	else{
		g_utf8_strncpy (directory, CURRENT_DIR, g_utf8_strlen (CURRENT_DIR, -1));
	}
	
	debug_print("before while: path=%s,directory=%s,tempuri=%s,uri=%s\n", path, directory,tempuri,uri);
	while ((*tempuri = *path) != 0){
		path++;
		tempuri++;
		if(*path == DELIMITER){
			*tempuri = '\0';
			debug_print("path==DELIMITER-1: path=%s,directory=%s,tempuri=%s,uri=%s\n", path, directory,tempuri,uri);
			if(osso_get_existing_file (uri, directory)){
				//set the directory to empty string when the uri doesnot
				//exist in the directory and exit
				found_match = FALSE;
				break;
			}
			strcat(directory, uri);
			strcat(directory, DELIMITERSTR);
			tempuri = uri;
			path++;
			debug_print("path==DELIMITER-2: path=%s,directory=%s,tempuri=%s,uri=%s\n", path, directory,tempuri,uri);
		}
		else if (*path == '\0'){
			*tempuri = '\0';
			debug_print("path==0-1: path=%s,directory=%s,tempuri=%s,uri=%s\n", path, directory,tempuri,uri);
			if(osso_get_existing_file (uri, directory)){
				//set the directory to empty string when the uri doesnot
				//exist in the directory and exit
				found_match = FALSE;
				break;
			}
			strcat(directory,uri);
			tempuri = uri;
			debug_print("path==0-2: path=%s,directory=%s,tempuri=%s,uri=%s\n", path, directory,tempuri,uri);
		}
	}
	
	path = path_ptr;
	*offset = 0; 
		
	
	//copy the matching URI (case insensitive) if the match was found 
	if(found_match){
		debug_print("found match");
		g_utf8_strncpy (path, directory, 
				g_utf8_strlen (directory, -1));
		debug_print("match: directory=%s, offset=%d\n",directory,*offset);
		return_val = OSSO_CASE_PATH_EXIST;
	}else{
		//measure the offset till match, this is the length of directory
		//that was being inspected for uri
		debug_print("found mismatch");
		*offset = g_utf8_strlen(directory, -1);
		
		debug_print("before doing strncpy - directory: %s,offset value is: %d",directory,*offset);
		g_utf8_strncpy (existing_path, directory, *offset);
		get_substring(path,remaining_path,*offset,g_utf8_strlen(path, -1));
		existing_path = g_strconcat(existing_path, remaining_path, NULL);
		debug_print("mismatch: directory=%s, existing_path=%s, offset=%d, remaining_path=%s\n",directory,existing_path,*offset, remaining_path);
	}
		
	debug_print("exiting: path=%s, directory=%s\n", path, directory);
	
	g_free (directory);
	g_free (uri);
	g_free(remaining_path);
		
	return return_val;
	
}


/**
  This Function verifies for a list of file names against a directory entry.
  
  @param filelist 	the input file list
  @param directorypath  the target directory path
  @return		0 if filename was found, else a value > 0
*/
int osso_file_list_validate (const GList *filelist, const gchar *directorypath)
{
    GList *dirlist = NULL;
    GList *templist = NULL;
    GnomeVFSResult result = 0;
    gchar *listname = NULL;
    gchar *dirname = NULL;

    gchar *caselistname = NULL;
    gchar *casedirname = NULL;
    
    int errorCount = 0 ;
    
	/*Read directory entries*/
    result = gnome_vfs_directory_list_load (&dirlist, directorypath, 1 << 3 );
    if (result != GNOME_VFS_OK)
    {
        return -(result);
    }
    
	/*For each file in the file list*/
    while(filelist != NULL){
        listname = ((GnomeVFSFileInfo*)(filelist->data))->name;
        if(g_utf8_collate (listname, CURRENT_DIR) && g_utf8_collate (listname, PARENT_DIR)){
            templist = dirlist;
			/*For each file in the directory*/
            while(templist != NULL){
                dirname = ((GnomeVFSFileInfo*)(templist->data))->name;
                if(g_utf8_collate (dirname, CURRENT_DIR) && g_utf8_collate (dirname, PARENT_DIR)){
                    caselistname = g_utf8_casefold (listname, -1);
		    casedirname = g_utf8_casefold (dirname, -1);
                    if (g_utf8_collate (caselistname, casedirname) == 0){
                        errorCount = errorCount + 1;
                    }
		    if(caselistname != NULL){
			    g_free (caselistname);
			    caselistname = NULL;
		    }
		    if(casedirname != NULL){
			    g_free (casedirname);
			    casedirname = NULL;
		    }
                }
                templist = templist->next;
            }
        }
        filelist = filelist->next;
    }
	/*Error count represent the number of file names present in the target directory.
	If the Error count is zero, no file exist with the name in the target folder.
	*/

    return errorCount;
}

                                                                                                                          


/**
  Method to check if the specified path exists.
  
  @param path 	path of file/folder
  @return true 	if file exist
*/
gboolean
osso_is_path_exists (const gchar *file, const gchar* directorypath)
{
  GnomeVFSURI *uri = NULL;
  gboolean retval = FALSE;
  gchar *path = NULL;

 // printf ("file : %s  directory : %s \n", file, directorypath);
  path = g_strdup_printf ("%s%s",directorypath, file);


  if(path == NULL){
    return FALSE;
  }

  uri = gnome_vfs_uri_new (path);
  g_free (path);
  
  if(uri == NULL){
    return FALSE;
  }

  retval = gnome_vfs_uri_exists(uri);

  if(uri != NULL){
    gnome_vfs_uri_unref(uri);
  }
 
  return retval;
  
}



/**
  This function gets the actual existing file name in the directory.
  This checks the target folder in case insensitive manner and retrives the 
  matching name.

  @param file		Is the file name to be verified.
  @param directorypath	Target directory path.
  @return 		0 if succesfully found the path, else negative for 
  			directory erros, positive for the error count.
*/
int osso_get_existing_file (gchar *file, const gchar *directorypath)
{
	GList *dirlist = NULL;
	GnomeVFSResult result = 0;
	gchar *name = (gchar*) g_new0 (gchar, g_utf8_strlen(file, -1) + 1);
	gchar *dirname = NULL;
	
	gchar *casefile = NULL;
	gchar *casedirname = NULL;
	
	int errorCount = 0 ;

/*	if(osso_is_path_exists (file, directorypath)){
		return 0;
	}*/
		
	/*Reading the entries in the target folder.*/
	result = gnome_vfs_directory_list_load (&dirlist, directorypath, 1 << 3 );
	if (result != GNOME_VFS_OK)
	{
		return -(result);
	}
	casefile = g_utf8_casefold (file, -1);
	/*For each file in the target folder.*/
	while(dirlist != NULL){
		dirname = ((GnomeVFSFileInfo*)(dirlist->data))->name;
		
		/*If case sensitive name is not available, check for case insensitive match*/
		casedirname = g_utf8_casefold (dirname, -1);
	
		if (g_utf8_collate (casefile, casedirname) == 0){
			errorCount = errorCount + 1;
			g_utf8_strncpy (name, dirname, 
					g_utf8_strlen (dirname, -1));
			break;
		}
			
		g_free (casedirname);
		dirlist = dirlist->next;
	}
	g_free (casefile);
	
	if(errorCount == 1){
		g_utf8_strncpy (file, name, 
				g_utf8_strlen (name, -1));
		g_free (name);
		return 0;
	}
	g_free (name);
	return 1;
}






/**
  This method prints a string in the given format to stdout.
  
  @param format 	the format of printable string
*/
void debug_print(const gchar *format, ...)
{
	return;
#ifdef DEBUG
	va_list args;
	gchar buf[BUFFSIZE]; 
	
	va_start(args, format); 
	g_vsnprintf(buf, sizeof(buf), format, args);
	va_end(args);

	fputs(buf, stdout);
	//printf(buf);
#endif

}

void get_substring(gchar *source, gchar *destination, gint index, gint count)
{
	memcpy(destination, &source[index],count);
	destination[count] = '\0';
}

/**
  This function gets the actual existing file path in the directory if it 
  exists or its case-different-path exsits. If they don't exist, the 
  function will get the actual parent directory + basename. 
  This checks the target folder in case insensitive manner and retrives the 
  matching path or the path combined by the matching part and the basename.

  @param path		Is the given path name to be verified. The value of 
  			this parameter is changed after the function runs.
  			It is either the matching path or the path combined 
			by the matching part and the basename.
			
  @return 		0 if successfully found the path, 
			else offset, the span of matching path in 'path' 	
*/
int osso_fetch_path(gchar* path)
{
	gchar *match_dir =(gchar*) g_new0 (gchar, MAX_LEN);
	int offset = 0;
	
	if ( ! osso_get_existing_path_offset (path, match_dir, &offset)) {
		g_free(match_dir);
		return 0;
	}
	
	match_dir = strcat(match_dir, g_utf8_offset_to_pointer(path, offset));
 
	g_utf8_strncpy (path, match_dir, g_utf8_strlen (match_dir, -1));
	
	g_free(match_dir);
	
	return offset;	
}
