/**
  @file xml.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

*/
#include <stdlib.h>
#include <stdio.h>
#include <strings.h>

#include <glib.h>

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xmlmemory.h>

#define STRCEQ(s1, s2) (g_ascii_strcasecmp((s1), (s2)) == 0)

#include "xml.h"

static void show_file(xmlNode *node, gboolean dir) {
    char *name, *size;

#if 0
    char *created, *modified, *type;

    created  = xmlGetProp(node, "created");
    modified = xmlGetProp(node, "modified");
    if (!dir)
        type = xmlGetProp(node, "type");
    else
        type = NULL;
#endif

    name     = xmlGetProp(node, "name");
    size     = xmlGetProp(node, "size");

    printf("%s %8s  %-60s\n",
            dir ? "D" : " ",
            size ? size : " ",
            name ? name : "(no name)");

    if (name)
        xmlFree(name);
    if (size)
        xmlFree(size);
#if 0
    if (created)
        xmlFree(created);
    if (modified)
        xmlFree(modified);
    if (type)
        xmlFree(type);
#endif
}

static void show_parent(xmlNode *node) {
    printf("%s %8s  %-60s\n", "D", " ", "..");
}

static void print_folder(xmlNode * a_node) {
    xmlNode *cur_node = NULL;

    for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
        if (cur_node->type == XML_ELEMENT_NODE) {
            if (STRCEQ(cur_node->name, "folder-listing"))
                print_folder(cur_node->children);
            else if (STRCEQ(cur_node->name, "parent-folder"))
                show_parent(cur_node);
            else if (STRCEQ(cur_node->name, "folder"))
                show_file(cur_node, TRUE);
            else if (STRCEQ(cur_node->name, "file"))
                show_file(cur_node, FALSE);
            else
                printf("ignoring element: %s\n", cur_node->name);
        }
    }
}

static void print_element_names(xmlNode * a_node) {
    xmlNode *cur_node = NULL;

    for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
        switch (cur_node->type) {
            case XML_ELEMENT_NODE:
                printf("XML_ELEMENT_NODE: %s\n", cur_node->name);
                break;

            case XML_ATTRIBUTE_NODE:
                printf("XML_ATTRIBUTE_NODE: %s\n", cur_node->name);
                break;

            case XML_TEXT_NODE:
                printf("XML_TEXT_NODE: %s\n", cur_node->name);
                break;

            case XML_CDATA_SECTION_NODE:
                printf("XML_CDATA_SECTION_NODE: %s\n", cur_node->name);
                break;

            default:
                printf("Unknown node (%d): %s\n", cur_node->type, cur_node->name);
                break;
        }

        print_element_names(cur_node->children);
    }
}

static xmlDocPtr parse_folder(const gchar *buf, gint len) {
    xmlDocPtr doc;

    /* get document */
    doc = xmlParseMemory(buf, len);

    if (doc == NULL) {
        printf("xmlParseMemory() failed\n");
        return NULL;
    }

    if (doc->xmlRootNode == NULL || doc->xmlRootNode->name == NULL
            || !STRCEQ(doc->xmlRootNode->name, "folder-listing")) {
        printf("Invalid folder-listing xml document\n");
        xmlFreeDoc(doc);
        return NULL;
    }

    return doc;
}

void print_folder_listing(const gchar *buf, gint len) {
    xmlNodePtr root;
    xmlDocPtr  doc;

    doc = parse_folder(buf, len);
    if (doc == NULL) {
        printf("Unable to parse folder-listing\n");
        xmlCleanupParser();
        return;
    }

    root = xmlDocGetRootElement(doc);

    print_folder(root);

    xmlFreeDoc(doc);

    xmlCleanupParser();
}

void print_capability(const gchar *buf, gint len) {
    xmlNodePtr root;
    xmlDocPtr  doc;

    doc = parse_folder(buf, len);
    if (doc == NULL) {
        printf("Unable to parse capability\n");
        xmlCleanupParser();
        return;
    }

    root = xmlDocGetRootElement(doc);

    print_element_names(root);

    xmlFreeDoc(doc);

    xmlCleanupParser();
}
