/**
  @file obex-test.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

*/
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <signal.h>
#include <glib.h>

#include <openobex/obex.h>

#include <gw-obex.h>

#include "xml.h"
#include "dbus.h"
#include "obex-test.h"

#define STREQ(a, b) (strcasecmp((a), (b)) == 0)

#define CMD_PROMPT "\n> "

extern GMainLoop *event_loop;

static gboolean user_dev;

static gboolean use_ftp = TRUE;

static volatile gboolean do_abort = FALSE;

static char *rfcomm_dev = NULL;

/* stdin GIOChannel info */
static GIOChannel *gio    = NULL;
static guint       gio_id = 0;

struct command {
    const gchar *name;
    gboolean (*cmd_func) (GwObex *ctx, gchar *args, gint *err);
    const gchar *help;
};

static void cmd_reset(GwObex *old_ctx, gboolean connect);

static void progress_cb(GwObex *ctx, gint cmd, gint current, gint target,
                        gpointer data) {
    printf("Progress: ");
    if (target > 0)
        printf("%s %d%% (%d / %d bytes)\r", cmd == OBEX_CMD_PUT ? "put" : "get",
               (int)(((float)current/target)*100), current, target);
    else
        printf("%d bytes\r", current);
}

static void disconnect_cb(GwObex *ctx, gpointer data) {
    printf("\nDisconnected.\n");
    printf("%s", CMD_PROMPT); fflush(stdout);
    cmd_reset(ctx, FALSE);
}

static gboolean cancel_cb(gpointer data) {
    if (do_abort) {
        do_abort = FALSE;
        return TRUE;
    }
    return FALSE;
}

static void abort_sig(int sig) {
    printf("Aborting!\n");
    do_abort = TRUE;
}

static gboolean cmd_cap(GwObex *ctx, gchar *args, gint *err) {
    gchar *cap;
    gint cap_len;
    if (!gw_obex_get_capability(ctx, &cap, &cap_len, err)) {
        printf("Getting capability failed\n");
        return FALSE;
    }
    cap = g_realloc(cap, cap_len + 1);
    cap[cap_len] = '\0';
    printf("Capability:\n%s\n", cap);
#if 0
    printf("\ncap[len - 2] = 0x%02X ('%c'), cap[len - 1] = 0x%02X ('%c')\n",
            cap[cap_len-2], isgraph(cap[cap_len-2]) ? cap[cap_len-2] : ' ',
            cap[cap_len-1], isgraph(cap[cap_len-1]) ? cap[cap_len-1] : ' ');
#endif

    g_free(cap);
    return TRUE;
}

static gboolean cmd_progress(GwObex *ctx, gchar *args, gint *err) {
    if (args) {
        if (STREQ(args, "on")) {
            gw_obex_set_progress_callback(ctx, progress_cb, err);
            printf("Showing progress\n");
            return TRUE;
        }
        else if (STREQ(args, "off")) {
            gw_obex_set_progress_callback(ctx, NULL, NULL);
            printf("Progress info disabled\n");
            return TRUE;
        }
    }
    printf("Argument should be \"on\" or \"off\"\n");
    return FALSE;
}

static gboolean cmd_exit(GwObex *ctx, gchar *args, gint *err) {
    if (ctx)
        gw_obex_close(ctx);
    g_main_loop_quit(event_loop);
    return TRUE;
}

static gboolean cmd_ls(GwObex *ctx, gchar *args, gint *err) {
    gchar *buf;
    gint buf_size;
    gboolean ret;
    ret = gw_obex_read_dir(ctx, args, &buf, &buf_size, err);
    if (ret == TRUE) {
#if 0
        const gchar *invalid;
        buf = g_realloc(buf, buf_size + 1);
        buf[buf_size] = '\0';
        if (!g_utf8_validate(buf, buf_size, &invalid)) {
            printf("Invalid UTF-8 character at offset %d (buf_size=%d): 0x%02X\n",
                    ((int)invalid - (int)buf), buf_size, *invalid);
        }
        printf("Folder-listing (raw XML):\n%s\n", buf);
#endif
        print_folder_listing(buf, buf_size);
        g_free(buf);
    }
    return ret;
}

static gboolean cmd_put(GwObex *ctx, gchar *args, gint *err) {
    if (args == NULL)
        return FALSE;
    return gw_obex_put_file(ctx, args, args, err);
}

static gboolean cmd_get(GwObex *ctx, gchar *args, gint *err) {
    if (args == NULL)
        return FALSE;
    return gw_obex_get_file(ctx, args, args, err);
}

static gboolean cmd_cd(GwObex *ctx, gchar *args, gint *err) {
    return gw_obex_chdir(ctx, args, err);
}

static gboolean cmd_mkdir(GwObex *ctx, gchar *args, gint *err) {
    if (args == NULL)
        return FALSE;
    return gw_obex_mkdir(ctx, args, err);
}

static gboolean cmd_delete(GwObex *ctx, gchar *args, gint *err) {
    return gw_obex_delete(ctx, args, err);
}

static gboolean cmd_copy(GwObex *ctx, gchar *args, gint *err) {
    gchar *src, *dst;

    src = args;
    dst = strchr(args, ' ');
    if (dst == NULL || dst[1] == '\0') {
        return FALSE;
    }
    else {
        dst[0] = '\0';
        dst++;
    }
    return gw_obex_copy(ctx, src, dst, err);
}

static gboolean cmd_move(GwObex *ctx, gchar *args, gint *err) {
    gchar *src, *dst;

    src = args;
    dst = strchr(args, ' ');
    if (dst == NULL || dst[1] == '\0') {
        return FALSE;
    }
    else {
        dst[0] = '\0';
        dst++;
    }
    return gw_obex_move(ctx, src, dst, err);
}

struct command cmd_table[];

static gboolean cmd_help(GwObex *ctx, gchar *args, gint *err) {
    struct command *cmd;
    printf("Available commands:\n");
    for (cmd = cmd_table; cmd->name != NULL; cmd++) {
        if (cmd->help != NULL) { 
            printf(" %-10s %-70s\n", cmd->name, cmd->help);
        }
    }
    printf(" %-10s %-70s\n", "connect", "Create a new connection");
    printf(" %-10s %-70s\n", "disconnect", "Disconnect");
    printf("\n");
    return TRUE;
}

struct command cmd_table[] = {
    { "exit",     cmd_exit,     NULL },
    { "quit",     cmd_exit,     "Exit the program" },
    { "help",     cmd_help,     "Show available commands" },
    { "ls",       cmd_ls,       "Show files in current directory" },
    { "dir",      cmd_ls,       NULL },
    { "put",      cmd_put,      "Send file to GW" },
    { "get",      cmd_get,      "Get file from GW" },
    { "cd",       cmd_cd,       "Change directory (no args for root, .. to go up)" },
    { "delete",   cmd_delete,   NULL },
    { "del",      cmd_delete,   NULL },
    { "rm",       cmd_delete,   "Remove file from GW" },
    { "mkdir",    cmd_mkdir,    "Create a new directory" },
    { "move",     cmd_move,     NULL },
    { "mv",       cmd_move,     "Move/Rename file on GW" },
    { "copy",     cmd_copy,     NULL },
    { "cp",       cmd_copy,     "Copy file on GW" },
    { "progress", cmd_progress, "Enable/Disable progress indication" },
    { "cap",      cmd_cap,      "Get capability object" },
    { NULL }
};

static gboolean input_command(GwObex *ctx, gchar *line, gint *err) {
    gchar *in_cmd, *in_args;
    struct command *cmd;

    in_cmd = line;
    in_args = strchr(line, ' ');
    if (in_args) {
        in_args[0] = '\0';
        if (in_args[1] == '\0')
            in_args = NULL;
        else
            in_args++;
    }

    for (cmd = cmd_table; cmd->name != NULL; cmd++) {
        if (STREQ(in_cmd, cmd->name)) { 
            return cmd->cmd_func(ctx, in_args, err);
        }
    }

    printf("** Unknown command! **\n");
    return TRUE;
}

static gboolean input_cb(GIOChannel *chan, GIOCondition cond, gpointer data) {
    GwObex *ctx = (GwObex *)data;
    GIOStatus status;
    gboolean ret = TRUE;
    gchar *line;
    guint length;

    status = g_io_channel_read_line(chan, &line, &length, NULL, NULL);
    if (status != G_IO_STATUS_NORMAL) {
        fprintf(stderr, "Error reading from stdin\n");
        return FALSE;
    }

    if (length > 1) {
        /* Remove newline */
        line[length - 1] = '\0';

        if (STREQ(line, "connect")) {
            cmd_reset(ctx, TRUE);
            ret = FALSE;
        }
        else if (STREQ(line, "disconnect")) {
            disconnect_obex_dev();
            cmd_reset(ctx, FALSE);
            ret = FALSE;
        }
        else if (ctx == NULL) {
            if (STREQ(line, "exit") || STREQ(line, "quit")) {
                disconnect_obex_dev();
                cmd_exit(ctx, NULL, NULL);
                ret = FALSE;
            }
            else if (STREQ(line, "help")) {
                cmd_help(ctx, NULL, NULL);
                ret = TRUE;
            }
            else {
                printf("** Not connected **\n");
                ret = TRUE;
            }
        }
        else {
            gint err;
            if (!input_command(ctx, line, &err)) {
                printf("** Command failed! (err: %d) **\n", err);
                if (err == GW_OBEX_ERROR_DISCONNECT) {
                    cmd_reset(ctx, FALSE);
                }
            }
            ret = TRUE;
        }
    }

    if (line)
        g_free(line);

    printf("%s", CMD_PROMPT); fflush(stdout);

    return ret;
}

static void cmd_reset(GwObex *old_ctx, gboolean connect) {
    GwObex *new_ctx = NULL;

    if (old_ctx) {
        gw_obex_close(old_ctx);
        old_ctx = NULL;
    }

    if (connect) {
        if (use_ftp) {
            if (!user_dev) {
                printf("Getting rfcomm device from btcond via D-BUS\n");
                g_free(rfcomm_dev);
                rfcomm_dev = get_obex_dev(use_ftp);
                if (rfcomm_dev == NULL) {
                    printf("Unable to get device from btcond\n");
                    exit(EXIT_FAILURE);
                }
                printf("Using device: %s\n", rfcomm_dev);
            }
            new_ctx = gw_obex_setup_dev(rfcomm_dev,
                                        OBEX_FTP_UUID, OBEX_FTP_UUID_LEN,
                                        NULL, NULL);
        }
        else {
            new_ctx = gw_obex_setup_dev(rfcomm_dev, NULL, 0, NULL, NULL);
        }
        if (new_ctx == NULL) {
            printf("OBEX setup failed!\n");
        }
    }

    if (new_ctx) {
        gw_obex_set_progress_callback(new_ctx, progress_cb, NULL);
        gw_obex_set_disconnect_callback(new_ctx, disconnect_cb, NULL);
        gw_obex_set_cancel_callback(new_ctx, cancel_cb, NULL);
    }

    if (gio) {
        g_io_channel_unref(gio);
        g_source_remove(gio_id);
        gio = NULL;
    }

    gio = g_io_channel_unix_new(STDIN_FILENO);
    gio_id = g_io_add_watch(gio, G_IO_IN | G_IO_ERR | G_IO_HUP,
                          input_cb, new_ctx);
} 

void test_setup(char *dev, gboolean using_ftp) {

    if (dev != NULL) {
        rfcomm_dev = dev;
	user_dev = TRUE;
    }
    else {
	user_dev = FALSE;
    }

    signal(SIGINT, abort_sig);
    signal(SIGUSR1, abort_sig);

    use_ftp = using_ftp;

    cmd_reset(NULL, TRUE);

    printf("\nType \"help\" to see available commands\n");

    printf("%s", CMD_PROMPT); fflush(stdout);
}

