/**
  @file main.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

*/
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <glib.h>
#include <glib-object.h>
#include <openobex/obex.h>

#include "dbus.h"
#include "obex-test.h"

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

static char *program_name;

GMainLoop *event_loop = NULL;

/** Startup config options */
static struct {
    char *dev;
    gboolean use_ftp;
} cfg;

/* For getopt */
static struct option const long_options[] = {
    {"help",     no_argument,       0, 'h'},
    {"device",   required_argument, 0, 'd'},
    {"ftp",      no_argument,       0, 'f'},
    {"opp",      no_argument,       0, 'p'},
    {"version",  no_argument,       0, 'V'},
    {NULL, 0, NULL, 0}
};

/* Print usage information and exit with status */
static void usage(int status) {
    printf("%s - GW OBEX Test application %s\n", program_name, VERSION);

    printf("Compilation flags: ");
#ifdef DEBUG
    printf("+DEBUG ");
#else
    printf("-DEBUG ");
#endif

    printf(
     "\nUsage: %s [OPTION]...\n"
     "Options:\n"
     "-h, --help                 Display this help and exit\n"
     "-V, --version              Output version information and exit\n"
     "-d, --device               Specify which device node to use\n"
     "-f, --ftp                  Use OBEX File Transfer Profile (default)\n"
     "-p, --opp                  Use OBEX Object Push Profile\n"
     "\n", program_name);

    exit(status);
}

/* Process commandline options. Returns index of first
 * non-option argument */
static int decode_switches(int argc, char *argv[])
{
    int c;

    /* Set defaults */
    cfg.use_ftp = TRUE;
    cfg.dev     = NULL;

    while ((c = getopt_long(argc, argv,
                    "h"   /* help      */
                    "V"   /* version   */
                    "d:"  /* device    */
                    "f"   /* ftp       */
                    "p"   /* opp       */
                    ,long_options, (int *) 0)) != EOF) {
        switch (c) {
            case 'V':
                printf("GW OBEX Test Application %s\n", VERSION);
                exit(EXIT_SUCCESS);

            case 'h':
                usage(EXIT_SUCCESS);

            case 'f':
                cfg.use_ftp = TRUE;
                break;

            case 'p':
                cfg.use_ftp = FALSE;
                break;

            case 'd':
                cfg.dev = g_strdup(optarg);
                break;

            default:
                usage(EXIT_FAILURE);
        }
    }

    return optind;
}

static void exit_cleanup(void) {
    printf("Exiting.\n");
}

int main(int argc, char *argv[]) {
    program_name = argv[0];

    (void) decode_switches(argc, argv);

    atexit(exit_cleanup);

    event_loop = g_main_loop_new(NULL, FALSE);

    g_type_init();

    printf("GW OBEX Test Application\n");

    test_setup(cfg.dev, cfg.use_ftp);

    g_main_loop_run(event_loop);

    g_main_loop_unref(event_loop);

    exit(EXIT_SUCCESS);
}

