/**
  @file gw-obex.h

  OSSO GW OBEX Connectivity Library and API

  The error codes returned by many of the functions refer either to an OBEX
  Protocol error or to a GW OBEX error. If the error code is less that 256, it
  refers to an OBEX error, othervice it refers to a GW_OBEX_ERROR_* error.

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

*/
#ifndef _GW_OBEX_H_
#define _GW_OBEX_H_

#include <glib.h>
#include <openobex/obex.h>

/** Value used if target length for put or get is not known */
#define GW_OBEX_UNKNOWN_LENGTH -1

/** Transport connection was disconnected */
#define GW_OBEX_ERROR_DISCONNECT        256

/** Operation was aborted */
#define GW_OBEX_ERROR_ABORT             257

/** GW OBEX internal error */
#define GW_OBEX_ERROR_INTERNAL          258

/** Unable to connecto to the specified service (UUID) */
#define GW_OBEX_ERROR_NO_SERVICE        259

/** Unable to create connection */
#define GW_OBEX_ERROR_CONNECT_FAILED    260

/** Timeout while waiting for data from GW */
#define GW_OBEX_ERROR_TIMEOUT           261

/** GW returned invalid/corrupted data */
#define GW_OBEX_ERROR_INVALID_DATA      262

/** Invalid parameters given to gwobex */
#define GW_OBEX_ERROR_INVALID_PARAMS    263

/** Local access error (e.g. read/write/open failed for local file) */
#define GW_OBEX_ERROR_LOCAL_ACCESS      264

typedef struct gw_obex GwObex;

#define OBEX_FTP_UUID \
    "\xF9\xEC\x7B\xC4\x95\x3C\x11\xD2\x98\x4E\x52\x54\x00\xDC\x9E\x09"
#define OBEX_FTP_UUID_LEN 16

/** Callback type for transport connection loss
 * @param ctx  GwObex pointer for the connection
 * @param data Optional pointer to user data
 */
typedef void (*gw_obex_disconnect_cb_t) (GwObex *ctx,
                                         gpointer data);

/** Callback type for progress information
 * @param ctx       GwObex pointer for the connection
 * @param obex_cmd  eg. OBEX_CMD_PUT
 * @param current   Bytes transfered
 * @param target    Total length (or GW_OBEX_UNKNOWN_LENGTH) 
 * @param data      Optional pointer to user data
 */
typedef void (*gw_obex_progress_cb_t) (GwObex *ctx, gint obex_cmd,
                                       gint current, gint target,
                                       gpointer data);

/** Callback type for checking if the operation should be canceled.
 * In the GNOME VFS case the callback function should be
 * gnome_vfs_cancellation_check().
 * @param data Optional pointer to user data
 * @returns TRUE if the operation should be canceled, FALSE othervice
 */
typedef gboolean (*gw_obex_cancel_cb_t) (gpointer data);


/** Open connection using a local device node and setup parameters.
 * This function should be called before calling any other functions. The
 * pointer returned by this function should be passed to the other functions.
 *
 * @param device   The local device which should be opened for the connection
 * @param uuid     UUID of service to connect to. NULL for the default service
 *                 (INBOX).
 * @param uuid_len Length (in bytes) of UUID
 * @param context  GMainContext to attach to (or NULL for the default one)
 * @param error    Place to store error code on failure (NULL if not interested)
 *
 * @returns A pointer, NULL on failure
 *  This pointer should be passed to the other obex_* functions.
 **/
GwObex *gw_obex_setup_dev(const gchar *device,
                          const gchar *uuid,
                          gint uuid_len,
                          GMainContext *context,
                          gint *error);


/** Setup OBEX connection using an opened filedescriptor
 * This function should be called before calling any other functions. The
 * pointer returned by this function should be passed to the other functions.
 *
 * @param fd       Opened filedescriptor to use for the connection
 * @param uuid     UUID of service to connect to. NULL for the default service
 *                 (INBOX).
 * @param uuid_len Length (in bytes) of UUID
 * @param context  GMainContext to attach to (or NULL for the default one)
 * @param error    Place to store error code on failure (NULL if not interested)
 *
 * @returns A pointer, NULL on failure
 *  This pointer should be passed to the other obex_* functions.
 **/
GwObex *gw_obex_setup_fd(int fd,
                         const gchar *uuid,
                         gint uuid_len,
                         GMainContext *context,
                         gint *error);


/** Close GW OBEX connection and free all memory associated with it.
 *
 * @param ctx Pointer returned by gw_obex_setup().
 *  Cannot be used anymore after this calling this function.
 */
void gw_obex_close(GwObex *ctx);


/** Get the capability object from the connected GW.
 *
 * @param ctx     Pointer returned by gw_obex_setup()
 * @param cap     Place to store the fetched object.
 *                 g_free() when not needed anymore.
 * @param cap_len Place to store the size of the fetched object
 * @param error   Place to store a possible error code
 *   (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_get_capability(GwObex *ctx,
                                gchar **cap,
                                gint *cap_len,
                                gint *error);

/** Set function to be called when a disconnection happens.
 *  You may (and probably should) call gw_obex_close() if this function is
 *  called.
 * @param ctx      Pointer returned by gw_obex_setup()
 * @param callback Function to call
 * @param data     Optional data to pass to the callback function
 */
void gw_obex_set_disconnect_callback(GwObex *ctx,
                                     gw_obex_disconnect_cb_t callback,
                                     gpointer data);


/** Set function to be called when progress for a put or get operation happens.
 *
 * @param ctx      Pointer returned by gw_obex_setup()
 * @param callback Function to call
 * @param data     Optional data to pass to the callback function
 */
void gw_obex_set_progress_callback(GwObex *ctx,
                                   gw_obex_progress_cb_t callback,
                                   gpointer data);


/** Set function to be called to check if the current operation should be
 * canceled. In the GNOME VFS case the callback function should be
 * gnome_vfs_cancellation_check(). The callback function should return TRUE if
 * the operation should be canceled and FALSE othervice.
 *
 * @param ctx      Pointer returned by gw_obex_setup()
 * @param callback Function to call
 * @param data     Pointer to pass to the callback function
 */
void gw_obex_set_cancel_callback(GwObex *ctx,
                                 gw_obex_cancel_cb_t callback,
                                 gpointer data);


/** Get a file from the GW.
 *
 * @param ctx    Pointer returned by gw_obex_setup()
 * @param local  Local filename (null terminated UTF-8)
 * @param remote Remote filename (null terminated UTF-8)
 * @param error  Place to store error code on failure
 *               (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_get_file(GwObex *ctx,
                          const gchar *local,
                          const gchar *remote,
                          gint *error);


/** Send a file to the GW.
 *
 * @param ctx    Pointer returned by gw_obex_setup()
 * @param local  Local filename (null terminated UTF-8)
 * @param remote Remote filename (null terminated UTF-8)
 * @param error  Place to store error code on failure
 *               (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_put_file(GwObex *ctx,
                          const gchar *local,
                          const gchar *remote,
                          gint *error);


/** Get a file from the GW and write it to a file descriptor
 *
 * @param ctx    Pointer returned by gw_obex_setup()
 * @param fd     File descriptor to write the file into
 * @param remote Remote filename (null terminated UTF-8)
 * @param error  Place to store error code on failure
 *               (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_get_fd(GwObex *ctx, gint fd, const gchar *remote, gint *error);

/** Read data from a file descriptor and send it to the GW
 *
 * @param ctx    Pointer returned by gw_obex_setup()
 * @param fd     File descriptor to read the data from 
 * @param remote Remote filename (null terminated UTF-8)
 * @param error  Place to store error code on failure
 *               (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_put_fd(GwObex *ctx, gint fd, const gchar *remote, gint *error);

/** Get an object from the GW and store it in a memory buffer.
 * Either remote filename or type must be supplied (or both).
 *
 * @param ctx      Pointer returned by gw_obex_setup()
 * @param remote   Remote filename (null terminated UTF-8)
 * @param type     MIME-type of the object
 * @param buf      Buffer to store the object in.
 *                  g_free() when not needed anymore.
 * @param buf_size Place to store length of fetched object
 * @param error    Place to store error code on failure
 *                 (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_get_buf(GwObex *ctx, const gchar *remote, const gchar *type,
                         gchar **buf, gint *buf_size, gint *error);


/** Send a object located in a memory buffer to the GW.
 * Either remote filename or type must be supplied (or both)
 *
 * @param ctx      Pointer returned by gw_obex_setup()
 * @param remote   Remote filename (null terminated UTF-8)
 * @param type     MIME-type of the object
 * @param buf      Buffer containing the object
 * @param buf_size Buffer (object) size
 * @param error    Place to store error code on failure
 *                 (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_put_buf(GwObex *ctx, const gchar *remote, const gchar *type,
                         const gchar *buf, gint buf_size, gint *error);


/** Change directory (relative to the current one).
 *
 * @param ctx   Pointer returned by gw_obex_setup()
 * @param dir   New directory to change to (null terminated UTF-8),
 *              ".." to go up, NULL to go to the root folder
 * @param error Place to store error code on failure (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_chdir(GwObex *ctx, const gchar *dir, gint *error);


/** Create a new directory. 
 *
 * @param ctx   Pointer returned by gw_obex_setup()
 * @param dir   Directory to create (null terminated UTF-8)
 * @param error Place to store error code on failure (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_mkdir(GwObex *ctx, const gchar *dir, gint *error);


/** Get folder listing for the specified directory.
 *
 * @param ctx      Pointer returned by gw_obex_setup()
 * @param dir      Directory to list (null terminated UTF-8),
 *                 NULL to list current directory
 * @param buf      Place to store the folder-listing object
 * @param buf_size Place to store the size for the retrieved object
 * @param error    Place to store error code on failure (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_read_dir(GwObex *ctx, const gchar *dir,
                          gchar **buf, gint *buf_size, gint *error);


/** Remove a file from the GW.
 *
 * @param ctx   Pointer returned by gw_obex_setup()
 * @param name  Filename to remove (null terminated UTF-8)
 * @param error Place to store error code on failure (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_delete(GwObex *ctx, const gchar *name, gint *error);


/** Move/Rename a file on the GW.
 *
 * @param ctx   Pointer returned by gw_obex_setup()
 * @param src   Source filename (null terminated UTF-8)
 * @param dst   Destination filename (null terminated UTF-8)
 * @param error Place to store error code on failure (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_move(GwObex *ctx, const gchar *src, const gchar *dst,
                      gint *error);


/** Copy a file on the GW.
 *
 * @param ctx   Pointer returned by gw_obex_setup()
 * @param src   Source filename (null terminated UTF-8)
 * @param dst   Destination filename (null terminated UTF-8)
 * @param error Place to store error code on failure (NULL if not interested)
 *
 * @returns TRUE on success, FALSE on failure
 */
gboolean gw_obex_copy(GwObex *ctx, const gchar *src, const gchar *dst,
                      gint *error);


#endif /* _GW_OBEX_H_ */

