/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2004, 2005 Nokia. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <ctype.h>
#include <string.h>
#include <gw-obex.h>
#include <bt-dbus.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus-glib-lowlevel.h>

#include "om-dbus.h"

#define d(x) 

/* Gwcond communication. Talks to gwcond to get the device.
 *
 * This API can be accessed from any thread, since it creates it's own context
 * and uses that for the dbus connection.
 */

typedef struct {
	DBusConnection *dbus_conn;
	GMainContext   *context;
	GMainLoop      *loop;
} Connection;

static Connection *
get_gwcond_connection (void)
{
	DBusConnection *dbus_conn;
	Connection     *conn;
	DBusError       error;
        const gchar    *address; 	

	dbus_error_init (&error);

	/* NOTE: We are not using dbus_bus_get here, for the reason that need to
	 * get our own private dbus connection to avoid threading problems with
	 * other libraries or applications that use this module and dbus (the
	 * vfs daemon in particular).
	 */
        address = g_getenv ("DBUS_SYSTEM_BUS_ADDRESS");

        /* Provide a fallback to the default location. */
        if (!address) {
		address = "unix:path=" LOCALSTATEDIR "/run/dbus/system_bus_socket";
        }

        if (!address) {
		g_warning ("Couldn't get the address for the system bus.");
                return NULL;
        }
	
	dbus_error_init (&error);
        dbus_conn = dbus_connection_open (address, &error);	
	
	if (!dbus_conn) {
		g_warning ("Failed to connect to the D-BUS daemon: %s", error.message);
		
		dbus_error_free (&error);
		return NULL;
	}

	if (!dbus_bus_register (dbus_conn, &error)) {
                g_warning ("Failed to register with the D-BUS daemon: %s", error.message);
		
                dbus_connection_disconnect (dbus_conn);
                dbus_connection_unref (dbus_conn);

                dbus_error_free (&error);
                return NULL;
        } 
	
	conn = g_new0 (Connection, 1);
	
	conn->context = g_main_context_new ();
	conn->loop = g_main_loop_new (conn->context, FALSE);

	conn->dbus_conn = dbus_conn;

	dbus_connection_setup_with_g_main (dbus_conn, conn->context);
	
	return conn;
}

static void
connection_free (Connection *conn)
{
	dbus_connection_disconnect (conn->dbus_conn);
	dbus_connection_unref (conn->dbus_conn);
	
	g_main_loop_unref (conn->loop);
	g_main_context_unref (conn->context);
	
	g_free (conn);
}

static gboolean
check_bda (const gchar *bda)
{
	gint len, i;

	if (!bda) {
		return FALSE;
	}

	len = strlen (bda);
	if (len != 17) {
		return FALSE;
	}
	
	for (i = 0; i < 17; i += 3) {
		if (!isxdigit (bda[i])) {
			return FALSE;
		}
		if (!isxdigit (bda[i+1])) {
			return FALSE;
		}
		if (i < 15 && bda[i+2] != ':') {
			return FALSE;
		}
	}
	
	return TRUE;
}

static gchar *
get_dev (Connection     *conn,
	 const gchar    *bda,
	 const gchar    *profile,
	 GnomeVFSResult *result)
{
	DBusMessage *message;
	DBusError    dbus_error;
	DBusMessage *reply;
	gboolean     ret;
	gchar       *str;
	gchar       *dev;

	if (!check_bda (bda)) {
		*result = GNOME_VFS_ERROR_INVALID_URI;
		return NULL;
	}
	
	message = dbus_message_new_method_call (BTCOND_SERVICE,
						BTCOND_REQ_PATH,
						BTCOND_REQ_INTERFACE,
						BTCOND_RFCOMM_CONNECT_REQ);
	if (!message) {
		g_error ("Out of memory");
	}
	
	if (!dbus_message_append_args (message,
				       DBUS_TYPE_STRING, bda,
				       DBUS_TYPE_STRING, profile,
				       DBUS_TYPE_BOOLEAN, (dbus_bool_t) FALSE,
				       DBUS_TYPE_INVALID)) {
		g_error ("Out of memory");
	}

	dbus_error_init (&dbus_error);
	reply = dbus_connection_send_with_reply_and_block (conn->dbus_conn, message, -1, &dbus_error);
	
	dbus_message_unref (message);

	if (dbus_error_is_set (&dbus_error)) {
		if (strcmp (dbus_error.name, BTCOND_ERROR_INVALID_SVC) == 0) {
			d(g_printerr ("obex: Invalid SDP profile.\n"));
			*result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
		}
		else if (strcmp (dbus_error.name, BTCOND_ERROR_INVALID_DEV) == 0) {
			d(g_printerr ("obex: Invalid BDA.\n"));
			*result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
		}
		else if (strcmp (dbus_error.name, BTCOND_ERROR_DISCONNECTED) == 0) {
			d(g_printerr ("obex: GW not connected.\n"));
			*result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
		}
		else if (strcmp (dbus_error.name, BTCOND_ERROR_NO_DEV_INFO) == 0) {
			d(g_printerr ("obex: No dev info.\n"));
			*result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
		}
		else if (strcmp (dbus_error.name, "org.freedesktop.DBus.Error.ServiceDoesNotExist") == 0) {
			d(g_printerr ("obex: Btcond not running.\n"));
			*result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
		}
		else if (strcmp (dbus_error.name, "org.freedesktop.DBus.Error.NoReply") == 0) {
			d(g_printerr ("obex: No reply.\n"));
			/* We get this if there is no device, and btcond times
			 * out.
			 */
			*result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
		} else {
			d(g_printerr ("obex generic: %s\n", dbus_error.name));
			*result = GNOME_VFS_ERROR_INTERNAL;
		}

		dbus_error_free (&dbus_error);
		return NULL;
	}
	
	if (!reply) {
		*result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
		return NULL;
	}

	ret = dbus_message_get_args (reply, NULL,
				     DBUS_TYPE_STRING, &str,
				     DBUS_TYPE_INVALID);

	dbus_message_unref (reply);

	if (!ret) {
		*result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
		return NULL;
	}
	
	dev = g_strdup (str);
	dbus_free (str);

	*result = GNOME_VFS_OK;
	return dev;
}

gchar *
om_dbus_get_dev (const gchar *bda, GnomeVFSResult *result)
{
	Connection *conn;
	gchar      *dev;

	conn = get_gwcond_connection ();
	if (!conn) {
		*result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
		return NULL;
	}
	
	*result = GNOME_VFS_OK;
	
	/* Try NFTP first, which appears to be some vendor specific profile. */
	dev = get_dev (conn, bda, "NFTP", result);
	if (!dev) {
		*result = GNOME_VFS_OK;
		dev = get_dev (conn, bda, "FTP", result);
	}
	
	connection_free (conn);
	
	return dev;
}

void
om_dbus_disconnect_dev (const gchar *dev)
{
	Connection  *conn;
	DBusMessage *message;
	DBusMessage *reply;

	conn = get_gwcond_connection ();
	if (!conn) {
		return;
	}
	
	message = dbus_message_new_method_call (BTCOND_SERVICE,
						BTCOND_REQ_PATH,
						BTCOND_REQ_INTERFACE,
						BTCOND_RFCOMM_DISCONNECT_REQ);
	if (!message) {
		g_error ("Out of memory");
	}
	
	if (!dbus_message_append_args (message,
				       DBUS_TYPE_STRING, dev,
				       DBUS_TYPE_INVALID)) {
		g_error ("Out of memory");
	}

	reply = dbus_connection_send_with_reply_and_block (conn->dbus_conn, message, -1, NULL);
	
	dbus_message_unref (message);

	if (reply) {
		dbus_message_unref (reply);
	}

	connection_free (conn);
}

