/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2004 Nokia. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * @author John Hedberg <johan.hedberg@nokia.com>
 */

#include <config.h>
#include "osso-mime.h"
#include "eggdesktopentries.h"
#include <string.h>

#include <syslog.h>
#define LOG_CLOSE() closelog()
#define DLOG_OPEN(X) openlog(X, LOG_PID | LOG_NDELAY, LOG_DAEMON)
#define DLOG_CRIT(...) syslog(LOG_CRIT | LOG_DAEMON, __VA_ARGS__)
#define DLOG_ERR(...) syslog(LOG_ERR | LOG_DAEMON, __VA_ARGS__)
#define DLOG_WARN(...) syslog(LOG_WARNING | LOG_DAEMON, __VA_ARGS__)
#define DLOG_INFO(...) syslog(LOG_INFO | LOG_DAEMON, __VA_ARGS__)
#ifdef DEBUG
# define DLOG_DEBUG(...) syslog(LOG_DEBUG | LOG_DAEMON, __VA_ARGS__)
#else
# define DLOG_DEBUG(...)
#endif
# define DLOG_CRIT_L(FMT, ARG...) syslog(LOG_CRIT | LOG_DAEMON, __FILE__ \
	":%d: " FMT, __LINE__, ## ARG)
# define DLOG_ERR_L(FMT, ARG...) syslog(LOG_ERR | LOG_DAEMON, __FILE__ \
	":%d: " FMT, __LINE__, ## ARG)
# define DLOG_WARN_L(FMT, ARG...) syslog(LOG_WARNING | LOG_DAEMON, __FILE__ \
	":%d: " FMT, __LINE__, ## ARG)
# define DLOG_INFO_L(FMT, ARG...) syslog(LOG_INFO | LOG_DAEMON, __FILE__ \
	":%d: " FMT, __LINE__, ## ARG)
# ifdef DEBUG
#  define DLOG_DEBUG_L(FMT, ARG...) syslog(LOG_DEBUG | LOG_DAEMON, __FILE__ \
	":%d: " FMT, __LINE__, ## ARG)
# else
#  define DLOG_DEBUG_L(FMT, ARG...) ((void)(0))
# endif
# define DLOG_CRIT_F(FMT, ARG...) syslog(LOG_CRIT | LOG_DAEMON, \
	"%s:%d: " FMT, __FUNCTION__, __LINE__, ## ARG)
# define DLOG_ERR_F(FMT, ARG...) syslog(LOG_ERR | LOG_DAEMON, \
	"%s:%d: " FMT, __FUNCTION__, __LINE__, ## ARG)
# define DLOG_WARN_F(FMT, ARG...) syslog(LOG_WARNING | LOG_DAEMON, \
	"%s:%d: " FMT, __FUNCTION__, __LINE__, ## ARG)
# define DLOG_INFO_F(FMT, ARG...) syslog(LOG_INFO | LOG_DAEMON, \
	"%s:%d: " FMT, __FUNCTION__, __LINE__, ## ARG)
# ifdef DEBUG
#  define DLOG_DEBUG_F(FMT, ARG...) syslog(LOG_DEBUG | LOG_DAEMON, \
	"%s:%d: " FMT, __FUNCTION__, __LINE__, ## ARG)
# else
#  define DLOG_DEBUG_F(FMT, ARG...) ((void)(0))
# endif

# ifdef DEBUG
#  include <stdio.h> /* for fprintf */
#  define dprint(f, a...) fprintf(stderr, "%s:%d %s(): "f"\n", __FILE__, \
				     __LINE__, __func__, ##a)
# else
#  define dprint(f, a...)
# endif /* DEBUG */


#define X_OSSO_SERVICE "X-Osso-Service"

static void _launch  (gpointer key,
		      gpointer value,
		      gpointer data);
static void _add_arg (gpointer value,
		      gpointer data);


typedef struct {
	GSList *files;
} AppEntry;


static void
app_entry_free (AppEntry *entry)
{
	g_slist_free (entry->files);
	g_free (entry);
}

static gchar *
desktop_file_get_service_name (const char *id)
{
	OmoDesktopEntries *entries;
	gchar             *filename;
	gchar             *service_name;

	service_name = NULL;

	filename = g_build_filename ("applications", id, NULL);

	entries = omo_desktop_entries_new_from_file (
		NULL,
		OMO_DESKTOP_ENTRIES_GENERATE_LOOKUP_MAP |
		OMO_DESKTOP_ENTRIES_DISCARD_COMMENTS,
		filename,
		NULL);

	g_free (filename);
	
	if (entries == NULL) {
		return NULL;
	}

	service_name = omo_desktop_entries_get_string (
		entries,
		omo_desktop_entries_get_start_group (entries),
		X_OSSO_SERVICE, NULL);

	omo_desktop_entries_free (entries);

	return service_name;
}

/* Returns the dbus service name for the default application of the specified
 * mime type.
 */
static gchar *
get_default_service_name (const char *mime_type)
{
	gchar *default_id;
	GList *list;
	gchar *service_name = NULL;

	default_id = gnome_vfs_mime_get_default_desktop_entry (mime_type);
	if (default_id != NULL && default_id[0] != '\0') {
		service_name = desktop_file_get_service_name (default_id); 
		g_free (default_id);

		return service_name;
	}

	/* Failing that, try something from the complete list */
	list = gnome_vfs_mime_get_all_desktop_entries (mime_type);
	if (list) {
		service_name = desktop_file_get_service_name (list->data); 

		g_list_foreach (list, (GFunc) g_free, NULL);
		g_list_free (list);
	}

	return service_name;
}

gint
osso_mime_open (DBusConnection *con, const gchar *file, ...)
{
	va_list     files;
	GHashTable *apps = NULL;
	gchar      *next;
	gint        num_apps;

	if (con == NULL) {
		DLOG_OPEN("libossomime");
		DLOG_ERR_F("error: no D-BUS connection!");
		LOG_CLOSE();
		return 0;
	}

	if (file == NULL) {
		DLOG_OPEN("libossomime");
		DLOG_ERR_F("error: no files to open!");
		LOG_CLOSE();
		return 0;
	}

	apps = g_hash_table_new_full (g_str_hash, g_str_equal,
				      g_free, (GDestroyNotify) app_entry_free);
    
	gnome_vfs_init (); /* make sure that gnome vfs is initialized */
	va_start (files, file);
	next = (gchar *) file;

	do {
		AppEntry *entry;
		gchar    *mime_type;
		gchar    *service_name;
	
		mime_type = gnome_vfs_get_mime_type (next);
		if (mime_type == NULL) {
			dprint ("No mime type for '%s'\n", next);
			DLOG_OPEN("libossomime");
			DLOG_ERR_F("error: Unable to determine MIME-type "
				   "of file '%s'",file);
			LOG_CLOSE();
			g_hash_table_destroy (apps);
			va_end (files);
			return 0;
		}

		service_name = get_default_service_name (mime_type);
		g_free (mime_type);

		if (service_name) {
			dprint ("Got service_name '%s' for file '%s'",
				service_name, next);

			entry = g_hash_table_lookup (apps, service_name);
			if (!entry) {
				entry = g_new0 (AppEntry, 1);
				g_hash_table_insert (apps, service_name, entry);
			}
		
			entry->files = g_slist_prepend (entry->files, next);
		} else {
			dprint ("No service name for file '%s'", next);
		}			
	
		next = va_arg (files, gchar *);
	} while (next != NULL);

	g_hash_table_foreach (apps, _launch, con);
    
	num_apps = g_hash_table_size (apps);
    
	g_hash_table_destroy (apps);

	va_end (files);
    
	/* If we didn't find an application to launch, it's an error. */
	if (num_apps == 0) {
		return 0;
	} else {
		return 1;
	}
}

gint
osso_mime_open_file (DBusConnection *con, const gchar *file)
{
	AppEntry *entry;
	gchar    *mime_type;
	gchar    *service_name;

	if (con == NULL) {
		DLOG_OPEN("libossomime");
		DLOG_ERR_F("error: no D-BUS connection!");
		LOG_CLOSE();
		return 0;
	}

	if (file == NULL) {
		DLOG_OPEN("libossomime");
		DLOG_ERR_F("error: no file to open!");
		LOG_CLOSE();
		return 0;
	}

	gnome_vfs_init (); /* make sure that gnome vfs is initialized */

	mime_type = gnome_vfs_get_mime_type (file);
	if (mime_type == NULL) {
		dprint ("No mime type for '%s'\n", file);
		DLOG_OPEN("libossomime");
		DLOG_ERR_F("error: Unable to determine MIME-type "
			   "of file '%s'", file);
		LOG_CLOSE();
		return 0;
	}

	service_name = get_default_service_name (mime_type);
	g_free (mime_type);

	if (!service_name) {
		dprint ("No service name for file '%s'", file);
		return 0;
	}

	entry = g_new0 (AppEntry, 1);
	entry->files = g_slist_prepend (NULL, (gpointer) file);

	_launch (service_name, entry, con);

	g_free (service_name);
	g_free (entry);

	return 1;
}

gint
osso_mime_open_file_list (DBusConnection *con, GSList *files)
{
	GHashTable *apps = NULL;
	GSList     *l;
	gint        num_apps;

	if (con == NULL) {
		DLOG_OPEN("libossomime");
		DLOG_ERR_F("error: no D-BUS connection!");
		LOG_CLOSE();
		return 0;
	}

	if (files == NULL) {
		DLOG_OPEN("libossomime");
		DLOG_ERR_F("error: no files to open!");
		LOG_CLOSE();
		return 0;
	}

	apps = g_hash_table_new_full (g_str_hash, g_str_equal,
				      g_free, (GDestroyNotify) app_entry_free);
    
	gnome_vfs_init (); /* make sure that gnome vfs is initialized */

	for (l = files; l; l = l->next) {
		AppEntry    *entry;
		const gchar *file;
		gchar       *mime_type;
		gchar       *service_name;

		file = (const gchar *) l->data;
	
		mime_type = gnome_vfs_get_mime_type (file);
		if (mime_type == NULL) {
			dprint ("No mime type for '%s'\n", file);
			DLOG_OPEN("libossomime");
			DLOG_ERR_F("error: Unable to determine MIME-type "
				   "of file '%s'", file);
			LOG_CLOSE();
			g_hash_table_destroy (apps);
			return 0;
		}

		service_name = get_default_service_name (mime_type);
		g_free (mime_type);

		if (service_name) {
			dprint ("Got service_name '%s' for file '%s'",
				service_name, file);

			entry = g_hash_table_lookup (apps, service_name);
			if (!entry) {
				entry = g_new0 (AppEntry, 1);
				g_hash_table_insert (apps, service_name, entry);
			}
		
			entry->files = g_slist_prepend (entry->files, 
							(gpointer) file);
		} else {
			dprint ("No service name for file '%s'", next);
		}			
	}	

	g_hash_table_foreach (apps, _launch, con);
    
	num_apps = g_hash_table_size (apps);
    
	g_hash_table_destroy (apps);

	/* If we didn't find an application to launch, it's an error. */
	if (num_apps == 0) {
		return 0;
	} else {
		return 1;
	}
}

static void _launch (gpointer key, gpointer value, gpointer data)
{
	DBusMessage     *msg;
	DBusMessageIter  iter;
	AppEntry        *entry = value;
	gchar           *k = key;
	DBusConnection  *con = data;
	gchar            service[256] = {0};
	gchar            object_path[256] = {0};
	gchar            interface[256] = {0};

	/* If the service name has a '.', treat it as a full name, otherwise
	 * prepend com.nokia.
	 */
	if (strchr (k, '.')) {
		g_snprintf (service, 255, "%s", k);
		g_snprintf (object_path, 255, "/%s", k);
		g_snprintf (interface, 255, "%s", k);

		g_strdelimit (object_path, ".", '/');
	} else {
		g_snprintf (service, 255, "%s.%s", "com.nokia",  k);
		g_snprintf (object_path, 255, "%s/%s", "/com/nokia", k);
		g_snprintf (interface, 255, "%s", service);
	}
	
	dprint ("Activating service: %s\n", service);

	msg = dbus_message_new_method_call (service, object_path,
					   interface, "mime_open");
	if(msg == NULL) {
		return;
	}

	dbus_message_set_auto_activation (msg, TRUE);
	dbus_message_set_no_reply (msg, TRUE);
    
	dbus_message_append_iter_init (msg, &iter);
    
	g_slist_foreach (entry->files, _add_arg, &iter);

	dbus_connection_send (con, msg, NULL);
	dbus_connection_flush (con);
	
	dbus_message_unref (msg);
}

static void _add_arg (gpointer value, gpointer data)
{
	DBusMessageIter *iter = data;
	gchar           *uri_string = value;
	GnomeVFSURI     *uri;
	const gchar     *scheme;
	gchar           *escaped;
	
	/* Turn the string into a URI which helps validate and escape it. A bit
	 * messy but needed if we are going to do the escaping here.
	 */
	uri = gnome_vfs_uri_new (uri_string);
	if (!uri) {
		return;
	}

	scheme = gnome_vfs_uri_get_scheme (uri);

	/* Check that the scheme was in the original argument. */
	if (strstr (uri_string, scheme) == uri_string) {
		gchar *tmp;
		
		tmp = gnome_vfs_escape_path_string (uri_string + strlen (scheme) + 1);
		escaped = g_strconcat (scheme, ":", tmp, NULL);
		g_free (tmp);
	} else {
		escaped = gnome_vfs_escape_path_string (uri_string);
	}

	gnome_vfs_uri_unref (uri);

	dprint ("  %s\n", escaped);
	
	dbus_message_iter_append_string (iter, escaped);
	g_free (escaped);
}
