/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2004 Nokia. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus-glib-lowlevel.h>

#include "dfm-dbus.h"


/* Monitor notification. The monitor dbus connection sits on the main thread's
 * main loop and listens for signals from other module instances and sends out
 * notification when we change something.
 *
 * This API should only be accessed from the main loop except for
 * dfm_dbus_emit_notify() which can be accessed from any thread.
 */

static DBusConnection *monitor_dbus_conn = NULL;

#define NOTIFY_SIGNAL_RULE "type='signal',interface='" VFS_MONITOR_INTERFACE "'"

typedef struct {
	GnomeVFSURI              *uri;
	GnomeVFSMonitorEventType  event_type;
} EventData;

static DBusConnection *
get_monitor_connection (gboolean create)
{
	const gchar *address;
	DBusError    error;

	if (monitor_dbus_conn) {
		return monitor_dbus_conn;
	}

	if (!create) {
		return NULL;
	}
	
	address = g_getenv ("DBUS_SESSION_BUS_ADDRESS");
	if (!address) {
		return NULL;
	}

	dbus_error_init (&error);

	monitor_dbus_conn = dbus_connection_open (address, &error);
	if (!monitor_dbus_conn) {
		g_warning ("Failed to connect to the D-BUS daemon: %s", error.message);
		
		dbus_error_free (&error);
		return NULL;
	}
	
	if (!dbus_bus_register (monitor_dbus_conn, &error)) {
		g_warning ("Failed to register with the D-BUS daemon: %s", error.message);
		
		dbus_connection_disconnect (monitor_dbus_conn);
		dbus_connection_unref (monitor_dbus_conn);

		monitor_dbus_conn = NULL;
		
		dbus_error_free (&error);
		return NULL;
	}
	
	dbus_connection_setup_with_g_main (monitor_dbus_conn, NULL);
	
	return monitor_dbus_conn;
}

static DBusHandlerResult
notify_message_filter (DBusConnection *dbus_conn,
		       DBusMessage    *message,
		       gpointer        user_data)
{
	MonitorNotifyFunc         func;
	GnomeVFSURI              *uri;
	gchar                    *str;
	const gchar              *sender, *base;
	GnomeVFSMonitorEventType  event_type;

	func = (MonitorNotifyFunc) user_data;
	
	if (dbus_message_is_signal (message,
				    VFS_MONITOR_INTERFACE,
				    VFS_MONITOR_SIGNAL_CREATED)) {
		event_type = GNOME_VFS_MONITOR_EVENT_CREATED;
	}
	else if (dbus_message_is_signal (message,
					 VFS_MONITOR_INTERFACE,
					 VFS_MONITOR_SIGNAL_DELETED)) {
		event_type = GNOME_VFS_MONITOR_EVENT_DELETED;
	}
	else if (dbus_message_is_signal (message,
					 VFS_MONITOR_INTERFACE,
					 VFS_MONITOR_SIGNAL_CHANGED)) {
		event_type = GNOME_VFS_MONITOR_EVENT_CHANGED;
	} else {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
	
	sender = dbus_message_get_sender (message);
	base = dbus_bus_get_base_service (dbus_conn);

	if (!sender) {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	/* Don't handle messages from ourselves. */
	if (strcmp (dbus_message_get_sender (message),
		    dbus_bus_get_base_service (dbus_conn)) == 0) {
		/*g_print ("message to self, skip\n");*/
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
	
	if (!dbus_message_get_args (message, NULL,
				    DBUS_TYPE_STRING, &str,
				    DBUS_TYPE_INVALID)) {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
	
	uri = gnome_vfs_uri_new (str);
	if (!uri) {
		dbus_free (str);
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	if (func) {
		func (uri, event_type);
	}
	
	dbus_free (str);
	gnome_vfs_uri_unref (uri);
	
	return DBUS_HANDLER_RESULT_HANDLED;
}

static void
event_data_free (EventData *data)
{
	gnome_vfs_uri_unref (data->uri);
	g_free (data);
}

static gboolean
emit_notify_idle_cb (EventData *data)
{
	DBusConnection *dbus_conn;
	DBusMessage    *message;
	gchar          *path;
	const gchar    *event_type_signal;

	dbus_conn = get_monitor_connection (TRUE);
	if (!dbus_conn) {
		event_data_free (data);
		return FALSE;
	}

	switch (data->event_type) {
	case GNOME_VFS_MONITOR_EVENT_CHANGED:
		event_type_signal = VFS_MONITOR_SIGNAL_CHANGED;
		break;

	case GNOME_VFS_MONITOR_EVENT_CREATED:
		event_type_signal = VFS_MONITOR_SIGNAL_CREATED;
		break;

	case GNOME_VFS_MONITOR_EVENT_DELETED:
		event_type_signal = VFS_MONITOR_SIGNAL_DELETED;
		break;

	default:
		event_type_signal = NULL;
		event_data_free (data);
		return FALSE;
	}

	path = gnome_vfs_uri_to_string (data->uri, GNOME_VFS_URI_HIDE_NONE);
	if (!path) {
		event_data_free (data);
		return FALSE;
	}

	/* We rely on that the caller has created valid URIs, i.e. escaped
	 * them. We check for UTF-8 here so the bus doesn't disconnect us.
	 */
	if (!g_utf8_validate (path, -1, NULL)) {
		g_warning ("Trying to send notification on non-utf8 URI.");

		g_free (path);
		event_data_free (data);
		return FALSE;
	}		
	
	message = dbus_message_new_signal (VFS_MONITOR_OBJECT,
					   VFS_MONITOR_INTERFACE,
					   event_type_signal);
	if (!message) {
		g_error ("Out of memory");
	}

	if (!dbus_message_append_args (message,
				       DBUS_TYPE_STRING, path,
				       DBUS_TYPE_INVALID)) {
		g_error ("Out of memory");
	}

	g_free (path);
	
	dbus_connection_send (dbus_conn, message, NULL);
	dbus_message_unref (message);

	event_data_free (data);
	
	return FALSE;
}

void
dfm_dbus_emit_notify (GnomeVFSURI              *uri,
		     GnomeVFSMonitorEventType  event_type)
{
	EventData *data;
	
	data = g_new0 (EventData, 1);
	data->uri = gnome_vfs_uri_dup (uri);
	data->event_type = event_type;

	g_idle_add ((GSourceFunc) emit_notify_idle_cb, data);
}

void
dfm_dbus_init_monitor (MonitorNotifyFunc monitor_func)
{
	DBusConnection *dbus_conn;

	dbus_conn = get_monitor_connection (TRUE);
	if (!dbus_conn) {
		return;
	}
	
	dbus_bus_add_match (dbus_conn, NOTIFY_SIGNAL_RULE, NULL);
	dbus_connection_add_filter (dbus_conn, notify_message_filter, 
				    monitor_func, NULL);
}

void
dfm_dbus_shutdown_monitor (void)
{
	if (!monitor_dbus_conn) {
		return;
	}
	
	dbus_bus_remove_match (monitor_dbus_conn, NOTIFY_SIGNAL_RULE, NULL);

	dbus_connection_disconnect (monitor_dbus_conn);
	dbus_connection_unref (monitor_dbus_conn);

	monitor_dbus_conn = NULL;
}
