/*
* DSP audio backend for ESD
*
*
* Copyright (c) 2005 Nokia Corporation
*
* @contact: Makoto Sugano <makoto.sugano@nokia.com>
*
* ESD is free software; you can redistribute it and/or modify it under the terms of
* the GNU General Public License as published by the Free Software Foundation;
* either version 2, or (at your option) any later version.
*
* ESD is distributed in the hope that it will be useful, but WITHOUT ANY
* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
* FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along with 
* ESD; see the file COPYING.  If not, write to the Free Software Foundation, 
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*
*/

#ifdef HAVE_MACHINE_SOUNDCARD_H
#  include <machine/soundcard.h>
#else
#  ifdef HAVE_SOUNDCARD_H
#    include <soundcard.h>
#  else
#    include <sys/soundcard.h>
#  endif
#endif

#include <dsp/pcm_socket_node.h>
#include <dsp/interface_common.h>
#include <dsp/audio_socket_node.h>

#include <sys/mman.h>
#include <sys/poll.h>

#define DSP_MODE_ERROR 0xE0

#define ARCH_esd_audio_devices
const char *esd_audio_devices()
{
    return "/dev/dsptask/pcm1";
}

int mode = O_RDWR;
AUDIO_STATUS_INFO status_info;
int pcm_drv;
short int pcm_bridge_buffer[BRIDGE_BUFFER_SIZE];
AUDIO_INIT_STATUS pcm_init_status;
short int *pcm_mmap_buf_addr;
int buf;
AUDIO_PARAMS_DATA pcm_params_data;
VOLUME_DATA pcm_volume_data;
PANNING_DATA pcm_panning_data;
short int pcm_state=STATE_UNINITIALISED;                            

int dsp_hard_reset(void);

int
dsp_check_read()
{
  struct pollfd pfd;

  pfd.fd = pcm_drv;
  pfd.events = POLLIN;

  if(poll(&pfd,1,0)>0) return 1;
  else return 0;
}

void dsp_read_junk(void) {
  while( dsp_check_read() )
    read(pcm_drv,&pcm_bridge_buffer[0],sizeof(short int));    
}



#define ARCH_esd_audio_open
int esd_audio_open()
{
  const char *device;
  int afd = -1, value = 0, test = 0;
  
  AUDIO_INIT_STATUS init_status;
  AUDIO_STATUS_INFO statInfo;
  short int tmp;
  int len;

  DSP_CMD_STATUS cmd;
  
  printf("Initializing DSP...\n");
  // open the sound device
  //device = esd_audio_device ? esd_audio_device : "/dev/dsptask/pcm";
  device = "/dev/dsptask/pcm1";
  
  if ((afd = open(device, mode, 0)) == -1) {
      // Opening device failed
      perror(device);
      return( -1 );
  }
    
  pcm_drv=afd;

  dsp_read_junk();

  statInfo.dsp_cmd = DSP_CMD_STATE;
  len = write(pcm_drv, &statInfo, sizeof(short int));
  if(len == -1) {
    pcm_state = DSP_MODE_ERROR;
    printf("failed to query dsp state\n");
    return -1;
  }

  len = read(pcm_drv, &statInfo, sizeof(AUDIO_STATUS_INFO));
  if(len == -1) {
    pcm_state = DSP_MODE_ERROR;
    printf("failed to read dsp return value\n");
    return -1;
  }

  if(statInfo.status == STATE_UNINITIALISED) {

    cmd.dsp_cmd = DSP_CMD_INIT;
    len = write(pcm_drv, &cmd, sizeof(short int));
    if(len == -1) {
      pcm_state = DSP_MODE_ERROR;
      return -1;
    }
    
    len = read(pcm_drv, &init_status, sizeof(AUDIO_INIT_STATUS));
    if(len == -1) {
      printf("Error reading status from DSP\n");
      return -1;
    }
    else if(init_status.init_status != DSP_OK) {
      printf("INIT_STATUS FAIL: %d\n", init_status.init_status);
      return -1;
    }

    pcm_init_status.stream_ID = init_status.stream_ID;
    pcm_init_status.bridge_buffer_size = init_status.bridge_buffer_size;
    pcm_init_status.mmap_buffer_size = init_status.mmap_buffer_size;
  }
  else {

    pcm_init_status.stream_ID = statInfo.stream_ID;
    pcm_init_status.bridge_buffer_size = statInfo.bridge_buffer_size;
    pcm_init_status.mmap_buffer_size = statInfo.mmap_buffer_size;

    tmp = DSP_CMD_STOP;
    len = write(pcm_drv, &tmp, sizeof(short int));
    if(len == -1) {
      pcm_state = DSP_MODE_ERROR;
      return -1;
    }
  
    // Read the "answer"
    read(pcm_drv, &cmd, sizeof(DSP_CMD_STATUS));
  }

  printf("PCM mmap buf size: %d\n", pcm_init_status.mmap_buffer_size);
  // mmap                                                                                                                           
  pcm_mmap_buf_addr = (char *) mmap(0,
      pcm_init_status.mmap_buffer_size*sizeof(short int),
      PROT_READ | PROT_WRITE, MAP_SHARED, pcm_drv, 0);

  if(pcm_mmap_buf_addr == NULL) {
    printf("Cannot mmap data buffer");
    return -1;
  }
    
  // set the sound driver number playback rate
  buf = esd_audio_rate;
  switch(buf)
  {
    case 48000: pcm_params_data.sample_rate = SAMPLE_RATE_48KHZ; break;
    case 44100: pcm_params_data.sample_rate = SAMPLE_RATE_44_1KHZ; break;
    case 32000: pcm_params_data.sample_rate = SAMPLE_RATE_32KHZ; break;
    case 24000: pcm_params_data.sample_rate = SAMPLE_RATE_24KHZ; break;
    case 22050: pcm_params_data.sample_rate = SAMPLE_RATE_22_05KHZ; break;
    case 16000: pcm_params_data.sample_rate = SAMPLE_RATE_16KHZ; break;
    case 12000: pcm_params_data.sample_rate = SAMPLE_RATE_12KHZ; break;
    case 11025: pcm_params_data.sample_rate = SAMPLE_RATE_11_025KHZ; break;
    case  8000: pcm_params_data.sample_rate = SAMPLE_RATE_8KHZ; break;
    default: printf("\nUnsupported sample rate.\n\n");getchar();
    return -1;
  }
     

  //while(!dsp_check_read()) read(pcm_drv,&pcm_bridge_buffer[0],sizeof(short int));
  dsp_read_junk();

  pcm_params_data.dsp_cmd = DSP_CMD_SET_PARAMS;
  pcm_params_data.stream_priority = 0;
  
  // set the sound driver audio format for playback
  value = test = ( (esd_audio_format & ESD_MASK_BITS) == ESD_BITS16 )
                   ? DSP_AFMT_S16_LE : DSP_AFMT_U8;
  pcm_params_data.audio_fmt = value;
  
  // set the sound driver number of channels for playback
  value = test = ( ( ( esd_audio_format & ESD_MASK_CHAN) == ESD_STEREO )
        ? 2 : 1 );
  pcm_params_data.number_channels = value;
  
  pcm_params_data.ds_stream_ID = 0;
  write(pcm_drv,&pcm_params_data,sizeof(pcm_params_data));
  read(pcm_drv,pcm_bridge_buffer,sizeof(DSP_CMD_STATUS));

  if(((DSP_CMD_STATUS *)pcm_bridge_buffer)->status != DSP_OK) {
    switch(((DSP_CMD_STATUS *)pcm_bridge_buffer)->status) {
      case DSP_ERROR_STATE: printf("\nError setting parameters: DSP_ERROR_STATE\n\n");break;
      case DSP_ERROR_RATE: printf("\nError setting parameters: DSP_ERROR_RATE\n\n");break;
      case DSP_ERROR_CHANNELS: printf("\nError setting parameters: DSP_ERROR_CHANNELS\n\n");break;
      case DSP_ERROR_DS_ID: printf("\nError setting parameters: DSP_ERROR_DS_ID\n\n");break;
      case DSP_ERROR_GENERAL: printf("\nError setting parameters: DSP_ERROR_GENERAL\n\n");break;
      case DSP_ERROR_FMT: printf("\nError setting parameters: DSP_ERROR_FMT\n\n");break;
      default:  printf("\nDSP init: Unknown error\n\n");break;
    }
    return -1;
  }

  pcm_volume_data.dsp_cmd = DSP_CMD_SET_VOLUME;
  pcm_volume_data.scale = 0x0C00; // 0x7FFF;
  pcm_volume_data.power2 = -1;
  write(pcm_drv,&pcm_volume_data,sizeof(pcm_volume_data));
  read(pcm_drv,pcm_bridge_buffer,sizeof(DSP_CMD_STATUS));
  if(((DSP_CMD_STATUS *)pcm_bridge_buffer)->status != DSP_OK) printf("\nError setting up volume.\n");
                                                                                                                      
  pcm_panning_data.dsp_cmd = DSP_CMD_SET_PANNING;
  pcm_panning_data.left_gain = 0x4000;
  pcm_panning_data.right_gain = 0x4000;
  pcm_panning_data.steps = 300;
  write(pcm_drv,&pcm_panning_data,sizeof(pcm_panning_data));
  read(pcm_drv,pcm_bridge_buffer,sizeof(DSP_CMD_STATUS));
  if(((DSP_CMD_STATUS *)pcm_bridge_buffer)->status != DSP_OK) printf("\nError setting up panning.\n");

  pcm_bridge_buffer[0] = DSP_CMD_PLAY;
  write(pcm_drv,pcm_bridge_buffer,sizeof(short int));
  pcm_state = DSP_CMD_PLAY;
  printf("\nPCM play ...");

  esd_audio_fd = pcm_drv;
  return pcm_drv;
}


#define ARCH_esd_audio_flush
void esd_audio_flush()
{
//    printf("Flush called\n");
    return;
}


#define ARCH_esd_audio_close
void esd_audio_close()
{

  unsigned short tmp;
  DSP_CMD_STATUS cmd;
  int len;

  printf("Close called\n");
  if(pcm_mmap_buf_addr != NULL) {
    munmap(pcm_mmap_buf_addr,
           pcm_init_status.mmap_buffer_size * sizeof(short int));
    pcm_mmap_buf_addr = NULL;
  }

  if(pcm_drv != -1) {
    
    // Read the junk
    dsp_read_junk();
      
    cmd.dsp_cmd = DSP_CMD_CLOSE;
    printf("  Writing DSP_CMD_CLOSE\n");
    len = write(pcm_drv, &cmd, sizeof(short int));
    if(len != -1) {
      printf("  Reading response to DSP_CMD_CLOSE\n");
      read(pcm_drv, &cmd, sizeof(DSP_CMD_STATUS));
    }
    printf("  Got response\n");
    
    close(pcm_drv);
    
    pcm_state = STATE_UNINITIALISED;
    pcm_drv = -1;
  }
}

#define ARCH_esd_audio_pause
void esd_audio_pause()
{
  if ( pcm_state == DSP_MODE_ERROR ) return;
  if ( pcm_state == DSP_CMD_PAUSE ) return;
  
  read(pcm_drv,&pcm_bridge_buffer[0],sizeof(short int));
  
  if (pcm_bridge_buffer[0]==DSP_CMD_DATA_WRITE) {
    read(pcm_drv,&pcm_bridge_buffer[1],sizeof(short int));
    
    if(pcm_bridge_buffer[1]==DSP_ERROR_STATE) {
      printf("DSP_CMD_DATA_WRITE: Wrong pcm node state.\n");
      
      if ( ! dsp_hard_reset() ) {
        printf("Critical problem with DSP, giving up\n");
        pcm_state = DSP_MODE_ERROR;
        return;
      }
    }
    read(pcm_drv,&pcm_bridge_buffer[2],sizeof(short int));
  }
  
  pcm_bridge_buffer[0] = DSP_CMD_PAUSE;
  write(pcm_drv,pcm_bridge_buffer,sizeof(short int));
  
  read(pcm_drv,&pcm_bridge_buffer[0],sizeof(short int));
  
  if (pcm_bridge_buffer[0]==DSP_CMD_PAUSE) {
    read(pcm_drv,&pcm_bridge_buffer[1],sizeof(short int));
    
    if (pcm_bridge_buffer[1]!=DSP_OK) {
      printf("\nError pausing stream: %d\n", pcm_bridge_buffer[1]);
    } else {
      pcm_state = DSP_CMD_PAUSE;
      printf("\nStream paused.\n");
    }
  }

  return;
}

                                                                                                                

#define ARCH_esd_audio_write
int esd_audio_write( void *buffer, int buf_size )
{
  if ( ( buf_size == 0 )  ||
       ( buffer == NULL ) ||
       ( pcm_state == DSP_MODE_ERROR ) ) {
    printf("Empty buffer received\n");
    return 0;
  }
// Test purposes wait
//  usleep(10000);


  if (pcm_state==DSP_CMD_PAUSE) {
    pcm_bridge_buffer[0] = DSP_CMD_PLAY;
    write(pcm_drv,pcm_bridge_buffer,sizeof(short int));
    pcm_state = DSP_CMD_PLAY;
    printf("\nPCM play ...");
  }

  read(pcm_drv,&pcm_bridge_buffer[0],sizeof(short int));
  
  if (pcm_bridge_buffer[0]==DSP_CMD_DATA_WRITE) {
    read(pcm_drv,&pcm_bridge_buffer[1],sizeof(short int));
    if(pcm_bridge_buffer[1]==DSP_ERROR_STATE) {
      printf("\nDSP_CMD_DATA_WRITE: Wrong pcm node state.\n");
      if ( ! dsp_hard_reset() ) {
        printf("Critical problem with DSP, giving up\n");
        pcm_state = DSP_MODE_ERROR;
        return -1;
      }
    }
    
    read(pcm_drv,&pcm_bridge_buffer[2],sizeof(short int));
    if(pcm_state == DSP_CMD_PLAY) {
      pcm_bridge_buffer[0] = DSP_CMD_DATA_WRITE;
      //printf("\nReceived: %d\n", buf_size);
      memcpy(pcm_mmap_buf_addr, (short int*)(buffer), buf_size);
      pcm_bridge_buffer[1] = buf_size/2;
      write(pcm_drv,pcm_bridge_buffer,2*sizeof(short int));
    }
  }

  return buf_size;
}

/* This method does the dsp node hard reinit.
 * Node is closed and then opened and initalized again.
 *
 * @return if hard reset went ok, 1 is returned. Otherwise 0
 */
 
int dsp_hard_reset(void)
{
  printf("Commencing hard reset, closing node...\n");
  esd_audio_close();
  
  // Sleep for a while so DSP can settle down
  usleep(50000);
  
  printf("Reopening and initalizing node...\n");
  if ( esd_audio_open() < 0 ) {
    printf("Hard init failed. Try resetting the DSP.\n");
    return 0;
  }
  else {
    printf("Hard init finished ok.\n");
  }
  return 1;
}
