/**
  @file io.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <sys/select.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#include "io.h"

#define TIMEOUT 2

static int rselect(int fd, int secs) {
    struct timeval tv;
    fd_set fds;
    FD_ZERO(&fds);
    FD_SET(fd, &fds);
    tv.tv_sec = secs;
    tv.tv_usec = 0;
    return select(fd+1, &fds, NULL, NULL, &tv); 
}

ssize_t io_write(int fd, const void *buf, int size) {
    ssize_t ret, sent = 0;

    while (sent < size) {
        ret = write(fd, ((char*)buf)+sent, size-sent);
        if (ret > 0) {
            sent += ret;
        }
        else if (ret < 0) {
            return ret;
        }
    }

    return sent;
}

/* Return number of bytes read, -1 on error or timeout */
ssize_t io_read(int fd, void *buf, int size) {
    ssize_t ret, rcvd = 0;

    while (rcvd < size) {
        if (rselect(fd, TIMEOUT) <= 0) {
            return -1;
        }
        ret = read(fd, ((char*)buf)+rcvd, size-rcvd); 
        if (ret > 0) {
            rcvd += ret;
        }
        else if (ret == 0) {
            break;
        }
        else if (errno != EAGAIN || errno != EINTR) {
            return -1;
        }
    }

    return rcvd;
}

