/**
  @file dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "log.h"
#include "dbus.h"

static DBusConnection *_dbus_connection = NULL;

DBusConnection *get_dbus_connection(void) {
    return _dbus_connection;
}

gboolean setup_dbus_connection(const char *service,
                          void (*handler_init)(DBusConnection *connection)) {
    DBusError derr;

    g_assert(_dbus_connection == NULL);

    dbus_error_init(&derr);
    _dbus_connection = dbus_bus_get(DBUS_BUS_SYSTEM, &derr);
    if (_dbus_connection == NULL) {
        error("System DBus connection failed: %s", derr.message);
        dbus_error_free(&derr);
        return FALSE;
    }

    dbus_connection_setup_with_g_main(_dbus_connection, NULL);

    if (service) {
        dbus_bus_acquire_service(_dbus_connection,
                                 service,
                                 DBUS_SERVICE_FLAG_PROHIBIT_REPLACEMENT,
                                 &derr);
        if (dbus_error_is_set(&derr)) {
            error("Could not aquire DBUS service '%s'", service);
            debug("%s: %s", derr.name, derr.message);
            dbus_error_free(&derr);
            return FALSE;
        }
    }

    if (handler_init)
        handler_init(_dbus_connection);

    return TRUE;
}

void close_dbus_connection(void) {
    g_assert(_dbus_connection != NULL);
    dbus_connection_unref(_dbus_connection);
    _dbus_connection = NULL;
    dbus_shutdown();
}

