/**
  @file btpin.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <getopt.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <glib.h>

#include "log.h"
#include "dbus.h"
#include "daemon.h"
#include "bttools-dbus.h"
#include "btpin-dbus.h"

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#define PIDFILE "/var/run/btpin.pid"

static char *program_name;

static int signal_pipe[2];

/** Startup config options */
static struct {
    gboolean daemon;     /**< fork and detatch into background */
    gboolean use_syslog; /**< Use syslog (instead of stdout and stderr) */
} cfg;

/* FIXME: Is this global variable necessary? */
GMainLoop *event_loop = NULL;

/* For getopt */
static struct option const long_options[] = {
    {"help",     no_argument,       0, 'h'},
    {"daemon",   no_argument,       0, 'd'},
    {"syslog",   no_argument,       0, 'l'},
    {"version",  no_argument,       0, 'V'},
    {NULL, 0, NULL, 0}
};

/* Print usage information and exit with status */
static void usage(int status) {
    printf("%s - BT PIN Application %s\n", program_name, VERSION);

    printf("Compilation flags: ");
#ifdef DEBUG
    printf("+DEBUG ");
#else
    printf("-DEBUG ");
#endif

    printf(
     "\nUsage: %s [OPTION]...\n"
     "Options:\n"
     "-h, --help                 Display this help and exit\n"
     "-V, --version              Output version information and exit\n"
     "-l, --syslog               Use syslog for loging (instead of stdio)\n"
     "-d, --daemon               Fork to background\n"
     "\n", program_name);

    exit(status);
}

static void exit_cleanup(void) {
    (void) remove_pid(PIDFILE);
    report("Exiting.");
}

/* Process commandline options. Returns index of first
 * non-option argument */
static int decode_switches(int argc, char *argv[])
{
    int c;

    /* Set default for logging */
    cfg.daemon     = FALSE;
    cfg.use_syslog = FALSE;

    while ((c = getopt_long(argc, argv,
                    "h"   /* help      */
                    "V"   /* version   */
                    "d"   /* daemon    */
                    "l"   /* syslog    */
                    ,long_options, (int *) 0)) != EOF) {
        switch (c) {
            case 'V':
                printf("GW SDP Application %s\n", VERSION);
                exit(EXIT_SUCCESS);

            case 'h':
                usage(EXIT_SUCCESS);

            case 'l':
                cfg.use_syslog = TRUE;
                break;

            case 'd':
                cfg.daemon = TRUE;
                cfg.use_syslog = TRUE;
                break;

            default:
                usage(EXIT_FAILURE);
        }
    }

    return optind;
}

static void handle_signal(int sig) {
    report("Signal %d received: %s.", sig, strsignal(sig));
    g_main_loop_quit(event_loop);
}

static void signal_handler(int sig) {   
        send(signal_pipe[1], &sig, sizeof(sig), MSG_DONTWAIT);
}   

static gboolean signal_cb(GIOChannel *chan, GIOCondition cond, gpointer data) {
    int fd;

    if (cond != G_IO_IN) {
        error("signal_cb: cond != G_IO_IN");
        handle_signal(SIGTERM);
        g_io_channel_unref(chan);
        return FALSE;
    }

    fd = g_io_channel_unix_get_fd(chan);
    g_assert(fd >= 0);

    for (;;) {
        int sig;

        if (recv(fd, &sig, sizeof(sig), MSG_DONTWAIT) != sizeof(sig))
            break;

        handle_signal(sig);
    }

    return TRUE;
}

static void bind_unix_signals(void) {
    GIOChannel *gio;

    if (socketpair(AF_UNIX, SOCK_STREAM, 0, signal_pipe) < 0) {
        die("socketpair: %s", strerror(errno));
    }

    gio = g_io_channel_unix_new(signal_pipe[0]);
    g_io_add_watch(gio, G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP,
                   signal_cb, NULL);

    if (signal(SIGINT, signal_handler) == SIG_ERR) {
        die("signal(SIGINT) failed");
    }
    if (signal(SIGTERM, signal_handler) == SIG_ERR) {
        die("signal(SIGTERM) failed");
    }

}

int main(int argc, char *argv[]) {
    gint i, old_pid;

    program_name = argv[0];
    i = decode_switches(argc, argv);

    open_log("btpin", cfg.use_syslog);

    old_pid = check_pid(PIDFILE);
    if (old_pid) {
        die("Unable to run: another instance running (PID %d)", old_pid);
    }

    if (cfg.daemon) {
        daemonize();
    }

    write_pid(PIDFILE);
    atexit(exit_cleanup);

    bind_unix_signals();

    event_loop = g_main_loop_new(NULL, FALSE);

    if (!setup_dbus_connection(PINAGENT_SERVICE, init_dbus_handlers)) {
        die("D-BUS connection setup failed!");
    }

    report("BT PIN %s started.", VERSION);

    /* Enter main loop */
    g_main_loop_run(event_loop);

    close_dbus_connection();
    g_main_loop_unref(event_loop);

    exit(EXIT_SUCCESS);
}

