/**
  @file btpin-stack.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "log.h"
#include "btpin-stack.h"

typedef struct {
    gchar *service;
    gchar *path;
    gchar *bda;
} PinHandler;

static GSList *stack = NULL;

#ifdef DEBUG
static void stack_show(void) {
    GSList *cur;

    debug("Stack contents (service, path, bda):");

    for (cur = stack; cur != NULL; cur = cur->next) {
        PinHandler *handler;
        handler = cur->data;
        g_assert(handler != NULL);
        debug("  (%s, %s, %s)", handler->service, handler->path,
                handler->bda ? handler->bda : "any");
    }
}
#else
# define stack_show(...) (void)(0)
#endif

static PinHandler *pin_handler_new(const gchar *service,
                                   const gchar *path,
                                   const gchar *bda) {
    PinHandler *handler;

    handler = g_new0(PinHandler, 1);

    if (service != NULL) {
        handler->service = g_strdup(service);
    }
    if (path != NULL) {
        handler->path = g_strdup(path);
    }
    if (bda != NULL) {
        handler->bda = g_strdup(bda);
    }
    
    return handler;
}

static void pin_handler_free(PinHandler *handler) {
    g_free(handler->service);
    g_free(handler->path);
    g_free(handler->bda);
    g_free(handler);
}

static gint pin_handler_cmp(PinHandler *h1, PinHandler *h2) {
    gint ret;

    /* If h2->service is NULL, we are only looking for matching BDA */
    if (h2->service == NULL) {
        /* Return match for handler which takes any (NULL) BDA */
        if (h1->bda == NULL) {
            return 0;
        }
        if (h2->bda != NULL) {
            return g_ascii_strcasecmp(h1->bda, h2->bda);
        }
        return 1;
    }

    ret = strcmp(h1->service, h2->service);
    /* If h2->path is not specified, we just want to match the service */
    if (ret != 0 || h2->path == NULL) {
        return ret;
    }

    ret = strcmp(h1->path, h2->path);
    if (ret != 0) {
        return ret;
    }
    
    if (h1->bda == NULL && h2->bda == NULL) {
        return 0;
    }

    if (h1->bda != NULL && h2->bda != NULL) {
        return g_ascii_strcasecmp(h1->bda, h2->bda);
    }

    return 1;
}

static PinHandler *stack_find_handler(const gchar *service,
                                      const gchar *path,
                                      const gchar *bda) {
    PinHandler *needle;
    GSList *match;

    needle = pin_handler_new(service, path, bda);

    match = g_slist_find_custom(stack, needle,
                                (GCompareFunc)pin_handler_cmp);

    pin_handler_free(needle);

    if (match != NULL) {
        return match->data;
    }

    return NULL;
}

static void stack_add(PinHandler *handler) {
    stack = g_slist_prepend(stack, handler);
}

static void stack_remove(PinHandler *handler) {
    GSList *match;

    match = g_slist_find(stack, handler);

    /* stack_remove should not be called unless we know there's a match */
    g_assert(match != NULL);

    pin_handler_free(match->data);
    
    stack = g_slist_delete_link(stack, match);
}

gboolean btpin_stack_register(const gchar *service,
                              const gchar *path,
                              const gchar *bda) {
    PinHandler *handler;

    g_assert(service != NULL);
    g_assert(path != NULL);

    debug("Registering (%s, %s, %s)", service,
            path ? path : "any",
            bda  ? bda  : "any");

    if (stack_find_handler(service, path, bda)) {
        error("Not adding duplicate handler");
        return FALSE;
    }

    handler = pin_handler_new(service, path, bda);
    stack_add(handler);

    stack_show();

    return TRUE;
}

gboolean btpin_stack_unregister(const gchar *service,
                                const gchar *path,
                                const gchar *bda) {
    gint i = 0;

    g_assert(service != NULL);

    do {
        PinHandler *handler;

        handler = stack_find_handler(service, path, bda);

        if (handler != NULL) {
            stack_remove(handler);
            i++;
        }
        else {
            break;
        }

    } while (TRUE);

    if (i > 0) {
        debug("Removed %i handlers", i);
        stack_show();
        return TRUE;
    }
    else {
        return FALSE;
    }
}

gboolean btpin_stack_get_handler(const gchar *bda,
                                 gchar **service,
                                 gchar **path) {
    PinHandler *handler;

    handler = stack_find_handler(NULL, NULL, bda);
    if (handler == NULL) {
        error("Could not find handler for %s", bda);
        return FALSE;
    }

    *service = g_strdup(handler->service);
    *path    = g_strdup(handler->path);

    return TRUE;
}

