/**
  @file btpair.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>
#include <getopt.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <glib.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "bttools-dbus.h"
#include "dbus.h"
#include "btpair-dbus.h"
#include "btpair-signal.h"
#include "btpair-engine.h"
#include "bt-utils.h"

GMainLoop *mainloop;

static char *program_name;

/** Command line config options */
typedef struct {
    gboolean user_handle;
    gboolean encrypt;
    gboolean use_syslog;
    gboolean use_dbus;
    uint16_t clock;
} CmdOpt;

/* For getopt */
static struct option const long_options[] = {
    {"help",     no_argument,       0, 'h'},
    {"version",  no_argument,       0, 'V'},
    {"stdio",    no_argument,       0, 's'},
    {"handle",   no_argument,       0, 'H'},
    {"encrypt",  no_argument,       0, 'e'},
    {"dbus",     no_argument,       0, 'd'},
    {"clock",    required_argument, 0, 'c'},
    {NULL, 0, NULL, 0}
};

/** Print usage information
 * @param status exit with status
 */
static void usage(int status) {
    printf("%s - Bluetooth Pairing Tool %s\n", program_name, VERSION);

    printf("Compilation flags: ");
#ifdef DEBUG
    printf("+DEBUG ");
#else
    printf("-DEBUG ");
#endif

#ifdef BUGGY_BT_HW
    printf("+BUGGY_BT_HW ");
#else
    printf("-BUGGY_BT_HW ");
#endif

    printf(
     "\nUsage: %s [OPTION] <BDA or handle>\n"
     "Options:\n"
     "-h, --help         Display this help and exit\n"
     "-V, --version      Output version information and exit\n"
     "-e, --encrypt      Also request encryption for the connection\n"
     "-d, --dbus         Use D-BUS inteface\n"
     "-s, --stdio        Log to stdout instead of syslog\n"
     "-H, --handle       Use existing connection handle\n"
     "-c, --clock        Clock offset (for HCI_Create_Connection)\n"
     "\n", program_name);

    exit(status);
}

/** Process commandline options.
 * @param argc Parameter given to main()
 * @param argv Parameter given to main()
 * @returns Index of first non-option argument
 */
static int decode_switches(int argc, char *argv[], CmdOpt *cfg)
{
    int c;

    memset(cfg, 0 , sizeof(*cfg));
    cfg->use_syslog = TRUE;
    if (argc < 2) {
        cfg->use_dbus = TRUE;
    }

    while ((c = getopt_long(argc, argv,
                    "h"   /* help    */
                    "V"   /* version */
                    "s"   /* stdio   */
                    "e"   /* encrypt */
                    "c:"  /* clock   */
                    "d"   /* dbus   */
                    "H"   /* handle  */
                    ,long_options, NULL)) != EOF) {
        switch (c) {
            case 'V':
                printf("Bluetooth Pairing Tool %s\n", VERSION);
                exit(EXIT_SUCCESS);

            case 'h':
                usage(EXIT_SUCCESS);

            case 's':
                cfg->use_syslog = FALSE;
                break;

            case 'e':
                cfg->encrypt = TRUE;
                break;

            case 'd':
                cfg->use_dbus = TRUE;
                break;

            case 'c':
                cfg->clock = (uint16_t) strtol(optarg, NULL, 0);
                break;

            case 'H':
                cfg->user_handle = TRUE;
                break;

            default:
                usage(EXIT_FAILURE);
        }
    }

    return optind;
}

static void cli_cb(GError *err, int *ret) {
    if (err) {
        error("btpair failed: %s", err->message);
        *ret = EXIT_FAILURE;
    }
    else {
        debug("pairing succeeded");
        *ret = EXIT_SUCCESS;
    }

    g_main_loop_quit(mainloop);
}

int main(int argc, char *argv[]) {
    int ret, i;
    CmdOpt cfg;

    program_name = argv[0];

    i = decode_switches(argc, argv, &cfg);
    open_log("btpair", cfg.use_syslog);

    if (cfg.use_dbus) {
        debug("Using D-BUS");

        if (!setup_dbus_connection(BTPAIR_SERVICE, init_dbus_handlers)) {
            die("D-BUS setup failed");
        }

        ret = EXIT_SUCCESS;
    }
    else {
        PairingContext *ctx;
        GError *err = NULL;

        if (argv[i] == NULL) {
            usage(EXIT_FAILURE);
        }

        if (!cfg.user_handle && !bda_ok(argv[i])) {
            die("Invalid BDA: %s", argv[i]);
        }

        debug("Pairing requested for %s %s",
                cfg.user_handle ? "handle" : "BDA", argv[i]);

        if (cfg.user_handle) {
            int handle;
            handle = strtol(argv[i], NULL, 0);
            ctx = pair_handle(handle, cfg.encrypt, (btpair_cb)cli_cb, &ret, &err);
        }
        else {
            ctx = pair_bda(argv[i], cfg.clock, cfg.encrypt, (btpair_cb)cli_cb, &ret, &err);
        }

        if (err) {
            error("btpair failed: %s", err->message);
            g_error_free(err);
            exit(EXIT_FAILURE);
        }

        bind_unix_signals(ctx);

        /* set default exit status */
        ret = EXIT_FAILURE;
    }

    debug("Entering mainloop");
    mainloop = g_main_loop_new(NULL, FALSE);
    g_main_loop_run(mainloop);

    g_main_loop_unref(mainloop);

    exit(ret);
}

