/**
  @file btpair-error.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdio.h>
#include <string.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib.h>

#include "log.h"
#include "bttools-dbus.h"
#include "btpair-error.h"

typedef struct {
    gint code;
    const char *derr;
} BtPairErrorMap;

static BtPairErrorMap btpair_map[] = {
    { BT_PAIR_ERROR_CONNECT_FAILED,     BTPAIR_ERROR_CONNECT_FAILED },
    { BT_PAIR_ERROR_DISCONNECTED,       BTPAIR_ERROR_DISCONNECTED   },
    { BT_PAIR_ERROR_TIMEOUT,            BTPAIR_ERROR_TIMEOUT        },
    { BT_PAIR_ERROR_REFUSED,            BTPAIR_ERROR_REFUSED        },
    { BT_PAIR_ERROR_AUTH_FAILED,        BTPAIR_ERROR_AUTH_FAILED    },
    { BT_PAIR_ERROR_TOO_FREQUENT,       BTPAIR_ERROR_TOO_FREQUENT   },
    { BT_PAIR_ERROR_RESOURCE_LIMIT,     BTPAIR_ERROR_RESOURCE_LIMIT },
    { BT_PAIR_ERROR_CANCELED,           BTPAIR_ERROR_CANCELED       },
    { BT_PAIR_ERROR_IN_PROGRESS,        BTPAIR_ERROR_IN_PROGRESS    },
    { BT_PAIR_ERROR_FAILED,             BTPAIR_ERROR_FAILED         },
    { 0, NULL }
};

static const char *btpair_dbus_error(GError *err) {
    BtPairErrorMap *map;

    for (map = btpair_map; map->derr != NULL; map++) {
        if (err->code == map->code)
            return map->derr;
    }

    error("Unknown btpair error %d", err->code);
    return BTPAIR_ERROR_FAILED;
}

#if 0
static const char *gerror_dbus_error(GError *err) {
    static char name[128];
    if (sscanf(err->message, "D-BUS error %127s:", name) != 1) {
        error("Unrecognized gerror message: %s", err->message);
        return "com.nokia.bt.error.unknown";
    } 

    debug("Detected non-BT error: %s", name);

    return name;
}
#endif

static void btpair_set_g_error(GError **err, DBusError *derr) {
    BtPairErrorMap *map;

    for (map = btpair_map; map->derr != NULL; map++) {
        if (strcmp(map->derr, derr->name) == 0) {
            g_set_error(err, BT_PAIR_ERROR,
                        map->code,
                        "%s", derr->message);
            return;
        }
    }

    error("Unknown btpair D-BUS error %s", derr->name);
    g_set_error(err, BT_PAIR_ERROR,
                BT_PAIR_ERROR_FAILED,
                "%s", derr->name);
}

const char *bt_pair_dbus_error(GError *err) {
    if (err->domain == BT_PAIR_ERROR)
        return btpair_dbus_error(err);

    error("Unknown error domain \"%s\" given to bt_dbus_error",
            g_quark_to_string(err->domain));
    return NULL;
}

gint bt_pair_bt_error(uint8_t err, gint fallback) {
    switch (err) {
        case 0x00:
            g_assert_not_reached();
            break;

        case 0x04: /* page timeout */
        case 0x08: /* connection timeout */
        case 0x10: /* connection accept timeout */
        case 0x22: /* LMP response timeout */
        case 0x28: /* instant passed - is this a timeout? */
            return BT_PAIR_ERROR_TIMEOUT;
            
        case 0x05: /* authentication failure */
        case 0x06: /* pin missing */
        case 0x0E: /* rejected due to security reasons - is this auth failure? */
        case 0x25: /* encryption mode not acceptable - is this auth failure? */
        case 0x26: /* link key cannot be changed - is this auth failure? */
        case 0x29: /* pairing with unit key unsupported - is this auth failure? */
        case 0x2f: /* insufficient security - is this auth failure? */
            return BT_PAIR_ERROR_AUTH_FAILED;

        case 0x17: /* too frequent pairing attempts */
            return BT_PAIR_ERROR_TOO_FREQUENT;

        case 0x18: /* pairing not allowed (e.g. gw rejected attempt) */
            return BT_PAIR_ERROR_REFUSED;

        case 0x07: /* memory capacity */
        case 0x09: /* connection limit */
        case 0x0a: /* synchronous connection limit */
        case 0x0d: /* limited resources */
        case 0x14: /* terminated due to low resources */
            return BT_PAIR_ERROR_RESOURCE_LIMIT;

        default:
            debug("Unmapped Bluetooth error: 0x%02X", err);
            return fallback;
    }
}

gboolean bt_pair_set_g_error(GError **err, DBusError *derr) {
    if (g_str_has_prefix(derr->name, BTPAIR_ERROR))
        btpair_set_g_error(err, derr);
    else
        return FALSE;

    return TRUE;
}

GQuark bt_pair_error_quark(void) {
    static GQuark q = 0;
    if (q == 0)
        q = g_quark_from_static_string("bt-pair-error-quark");

    return q;
}

