/**
  @file btpair-engine.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>
#include <glib.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "btpair-error.h"
#include "btpair-engine.h"

typedef enum {
    STATE_DISCONNECTED,
    STATE_CONNECTING,
    STATE_CONNECTED,
    STATE_AUTHENTICATING,
    STATE_AUTHENTICATED,
    STATE_INVALID
} State;

struct pairing_context {
    uint16_t    handle;
    uint16_t    clock;
    GIOChannel *io;
    guint       io_id;
    gboolean    user_handle;
    gboolean    connected;
    gboolean    encrypted;
    State       state;
    bdaddr_t    bdaddr;
    gboolean    encrypt;
    btpair_cb   func;
    gpointer    user_data;
};

#define CTX_SOCK(ctx) (g_io_channel_unix_get_fd((ctx)->io))

#ifndef OCF_CC_CANCEL
#define OCF_CC_CANCEL 0x0008
#endif

#ifdef BUGGY_BT_HW
# define AUTH_WAIT 500
#endif

#define HCI_DEV_ID      0
#define CC_TIMEOUT      30000
#define DC_TIMEOUT      100
#define CANCEL_TIMEOUT  100
#define AUTH_TIMEOUT    30000
#define ROLE_SWITCH     0x01

static int event_socket(int *events) {
    struct sockaddr_hci addr;
    struct hci_filter flt;
    int sock, i;

    /* Create HCI socket */
    sock = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_HCI);
    if (sock < 0) {
        die("Can't create HCI socket: %s (%d)",
                g_strerror(errno), errno);
    }

    /* Setup filter */
    hci_filter_clear(&flt);
    hci_filter_set_ptype(HCI_EVENT_PKT, &flt);
    for (i = 0; events[i] != 0; i++)
        hci_filter_set_event(events[i], &flt);
    if (setsockopt(sock, SOL_HCI, HCI_FILTER, &flt, sizeof(flt)) < 0) {
        die("Can't set HCI filter: %s (%d)",
                g_strerror(errno), errno);
    }

    /* Bind socket to the HCI device */
    addr.hci_family = AF_BLUETOOTH;
    addr.hci_dev = HCI_DEV_ID;
    if (bind(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
        die("Can't attach to device hci%d. %s (%d)",
                HCI_DEV_ID, g_strerror(errno), errno);
    }
    return sock;
}

static void get_bda_handle(bdaddr_t *ba, int *handle) {
    struct hci_conn_info_req *cr;
    int dd;

    dd = hci_open_dev(HCI_DEV_ID);
    if (dd < 0) {
        error("hci_open_dev: %s", g_strerror(errno));
        return;
    }   

    cr = g_malloc0((sizeof(*cr) + sizeof(struct hci_conn_info)));
    bacpy(&cr->bdaddr, ba);
    cr->type = ACL_LINK;

    if (ioctl(dd, HCIGETCONNINFO, (unsigned long)cr) < 0) {
        close(dd);
        g_free(cr);
        return;
    }

    close(dd);

    *handle = cr->conn_info->handle;
}       

static gboolean cancel_connect(struct pairing_context *ctx) {
   status_bdaddr_rp rp; 
   struct hci_request rq;

   memset(&rq, 0, sizeof(rq));
   rq.ogf    = OGF_LINK_CTL;
   rq.ocf    = OCF_CC_CANCEL;
   rq.cparam = &ctx->bdaddr;
   rq.clen   = sizeof(ctx->bdaddr);
   rq.rparam = &rp;
   rq.rlen   = STATUS_BDADDR_RP_SIZE;

   if (hci_send_req(CTX_SOCK(ctx), &rq, CANCEL_TIMEOUT) < 0) {
       error("Sending cancel request failed: %s", g_strerror(errno));
       return FALSE;
   }

   if (rp.status) {
       error("Connect cancel returned status 0x%02X\n", rp.status);
       return FALSE;
   }

   return TRUE;
}

/* Mostly from hcitool */
static gboolean create_connection(struct pairing_context *ctx) {
    uint16_t ptype;
    create_conn_cp cp;

    g_assert(!ctx->connected && ctx->state == STATE_DISCONNECTED);

    memset(&cp, 0, sizeof(cp));

    bacpy(&cp.bdaddr, &ctx->bdaddr);

    ptype = HCI_DM1 | HCI_DM3 | HCI_DM5 | HCI_DH1 | HCI_DH3 | HCI_DH5;
    cp.pkt_type       = htobs(ptype);
    cp.pscan_rep_mode = 0x02;
    cp.clock_offset   = ctx->clock;
    cp.role_switch    = ROLE_SWITCH;

    if (hci_send_cmd(CTX_SOCK(ctx), OGF_LINK_CTL, OCF_CREATE_CONN,
                     CREATE_CONN_CP_SIZE, &cp) < 0) {
        error("hci_send_cmd: %s", g_strerror(errno));
        return FALSE;
    }

    ctx->state = STATE_CONNECTING;

    return TRUE;
}

static gboolean disconnect(struct pairing_context *ctx) {

    if (hci_disconnect(CTX_SOCK(ctx), ctx->handle, HCI_OE_USER_ENDED_CONNECTION, DC_TIMEOUT) < 0) {
        return FALSE;
    }

    return TRUE;
}

static void destroy_context(struct pairing_context *ctx) {
    if (ctx->io) {
        g_io_channel_unref(ctx->io);
    }
    if (ctx->io_id > 0) {
        g_source_remove(ctx->io_id);
    }
    g_free(ctx);
}

static void pair_finalize(struct pairing_context *ctx, GError *err) {
    if (ctx->state == STATE_CONNECTING) {
        (void) cancel_connect(ctx);
    }

    if (ctx->state != STATE_AUTHENTICATED) {
        g_assert(err != NULL);
    }

    if (ctx->connected && !ctx->user_handle && !disconnect(ctx)) {
        error("Disconnect failed.");
    }

    ctx->func(err, ctx->user_data);

    if (err) {
        g_error_free(err);
    }

    destroy_context(ctx);
}

static gboolean request_auth(struct pairing_context *ctx) {
    auth_requested_cp cp;

    g_assert(ctx->state == STATE_CONNECTED);

    cp.handle = ctx->handle;

    if (hci_send_cmd(CTX_SOCK(ctx), OGF_LINK_CTL, OCF_AUTH_REQUESTED,
                     AUTH_REQUESTED_CP_SIZE, &cp) < 0) {
        return FALSE;
    }

    ctx->state = STATE_AUTHENTICATING;

    return TRUE;
}

static gboolean set_encrypt(struct pairing_context *ctx) {
    set_conn_encrypt_cp cp;

    cp.handle = ctx->handle;
    cp.encrypt = 0x01;

    if (hci_send_cmd(CTX_SOCK(ctx), OGF_LINK_CTL, OCF_SET_CONN_ENCRYPT,
                      SET_CONN_ENCRYPT_CP_SIZE, &cp) < 0) {
        return FALSE;
    }

    return TRUE;
}

#ifdef BUGGY_BT_HW
static gboolean auth_wait_cb(struct pairing_context *ctx) {
    if (!request_auth(ctx)) {
        GError *err;
        g_set_error(&err, BT_PAIR_ERROR,
                    BT_PAIR_ERROR_AUTH_FAILED,
                    "Sending HCI_Request_Authentication failed!");
        pair_finalize(ctx, err);
    }

    return FALSE;
}
#endif

static int handle_conn_complete(void *data, struct pairing_context *ctx, GError **err) {
    evt_conn_complete *ev = data;

    if (ev->status != 0) {
        ctx->state = STATE_DISCONNECTED;
        g_set_error(err, BT_PAIR_ERROR,
                    bt_pair_bt_error(ev->status, BT_PAIR_ERROR_CONNECT_FAILED),
                    "Create Connection failed with 0x%02X", ev->status);
        return 1;
    }

    ctx->state     = STATE_CONNECTED;
    ctx->connected = TRUE;
    ctx->handle    = ev->handle;

    debug("Connected. Handle = %u", btohs(ev->handle));

#ifdef BUGGY_BT_HW
    debug("Waiting %u milliseconds before HCI_Authentication_Requested",
            AUTH_WAIT);
    (void) g_timeout_add(AUTH_WAIT,
                         (GSourceFunc)auth_wait_cb,
                         ctx);
#else
    if (!request_auth(ctx)) {
        g_set_error(err, BT_PAIR_ERROR,
                    BT_PAIR_ERROR_AUTH_FAILED,
                    "Sending HCI_Request_Authentication failed!");
        return 1;
    }
#endif

    return 0;
}

static int handle_disconn_complete(void *data, struct pairing_context *ctx, GError **err) {
    evt_disconn_complete *ev = data;

    if (ev->handle != ctx->handle) {
        debug("Disconnection Complete for unknown handle %u", btohs(ev->handle));
        return 0;
    }

    if (ev->status != 0) {
        error("Disconnection failed with 0x%02X", ev->status);
        return 1;
    }

    debug("Disconnected. Reason: 0x%02X", ev->reason);

    if (ctx->state == STATE_AUTHENTICATING) {
        debug("Buggy BT: disconnected before receiving auth_complete");
    }

    ctx->connected = FALSE;
    ctx->handle    = 0;

    if (ctx->state != STATE_AUTHENTICATED) {
        g_set_error(err, BT_PAIR_ERROR,
                    BT_PAIR_ERROR_DISCONNECTED,
                    "Disconnected before authentication was completed");
    }

    return 1;
}

static int handle_encrypt_change(void *data, struct pairing_context *ctx, GError **err) {
    evt_encrypt_change *ev = data;

    if (ev->handle != ctx->handle) {
        return 0;
    }

    if (ev->status != 0) {
        error("Encryption change failed with 0x%02X", ev->status);
        return 1;
    }

    if (ev->encrypt == 0x01) {
        debug("Encryption enabled");
        ctx->encrypted = TRUE;
    }
    else {
        /* Should not happen */
        debug("Encryption disabled");
    }

    if (ctx->state == STATE_AUTHENTICATED) {
        return 1;
    }

    return 0;
}

static int handle_auth_complete(void *data, struct pairing_context *ctx, GError **err) {
    evt_auth_complete *ev = data;

    if (ev->handle != ctx->handle) {
        return 0;
    }

    if (ev->status != 0) {
        g_set_error(err, BT_PAIR_ERROR,
                    bt_pair_bt_error(ev->status, BT_PAIR_ERROR_AUTH_FAILED),
                    "Authentication failed with 0x%02X", ev->status);
        return 1;
    }

    debug("Authenticated successfully");

    ctx->state = STATE_AUTHENTICATED;

    if (ctx->encrypt && !ctx->encrypted) {
        debug("Requesting encryption");
        if (!set_encrypt(ctx)) {
            /* Don't fail since HCI_Request_Authentication is the important part */
            error("Sending HCI_Set_Connection_Encryption command failed!");
            return 1;
        }
        return 0;
    }

    return 1;
}

/* Return 1 when finished */
static int process_data(char *data, int data_len, struct pairing_context *ctx, GError **err) {
    uint8_t type = data[0];

    if (type == HCI_EVENT_PKT) {
        hci_event_hdr *hdr = (hci_event_hdr *) &data[1];
        char *body = &data[HCI_EVENT_HDR_SIZE + 1];

        switch (hdr->evt) {
            case EVT_CONN_COMPLETE:
                return handle_conn_complete(body, ctx, err);
            case EVT_AUTH_COMPLETE:
                return handle_auth_complete(body, ctx, err);
            case EVT_ENCRYPT_CHANGE:
                return handle_encrypt_change(body, ctx, err);
            case EVT_DISCONN_COMPLETE :
                return handle_disconn_complete(body, ctx, err);
            case EVT_CMD_STATUS:
                if (body[0] != 0x00) {
                    g_set_error(err, BT_PAIR_ERROR,
                                BT_PAIR_ERROR_FAILED,
                                "HCI Command failed with 0x%02x", body[0]);
                    return 1;
                }
                return 0;
            default:
                debug("Unhandled event (0x%02x)", hdr->evt);
                return 0;
        }
    }
    else {
        g_set_error(err, BT_PAIR_ERROR,
                    BT_PAIR_ERROR_FAILED,
                    "Invalid packet type (0x%02x)", type);
        return 1;
    }
}

#define DATA_LEN HCI_MAX_FRAME_SIZE

static gboolean hci_cb(GIOChannel *chan, GIOCondition cond, struct pairing_context *ctx) {
    char *data;
    GError *err = NULL;
    int data_len, ret, sock;

    if (cond != G_IO_IN) {
        if (ctx->connected) {
            disconnect(ctx);
        }
        g_set_error(&err, BT_PAIR_ERROR,
                    BT_PAIR_ERROR_FAILED,
                    "Error on HCI socket");
        ctx->io_id = 0;
        pair_finalize(ctx, err);
        return FALSE;
    }

    data = g_malloc(DATA_LEN);

    sock = g_io_channel_unix_get_fd(chan);

    data_len = read(sock, data, DATA_LEN);
    if (data_len < 0) {
        if (ctx->connected) {
            disconnect(ctx);
        }
        g_set_error(&err, BT_PAIR_ERROR,
                    BT_PAIR_ERROR_FAILED,
                    "read(hci_sock): %s", g_strerror(errno));
        ctx->io_id = 0;
        pair_finalize(ctx, err);
        return FALSE;
    }

    ret = process_data(data, data_len, ctx, &err);
    g_free(data);

    if (ret != 0) {
        ctx->io_id = 0;
        pair_finalize(ctx, err);
        return FALSE;
    }

    return TRUE;
}

static gboolean hci_setup(struct pairing_context *ctx) {
    int sock;
    int events[] = { EVT_CONN_COMPLETE,
                     EVT_DISCONN_COMPLETE,
                     EVT_AUTH_COMPLETE,
                     EVT_ENCRYPT_CHANGE,
                     EVT_CMD_STATUS,
                     0 };

    sock = event_socket(events);
    if (sock < 0) {
        return FALSE;
    }

    ctx->io = g_io_channel_unix_new(sock);
    g_io_channel_set_close_on_unref(ctx->io, TRUE);
    ctx->io_id = g_io_add_watch(ctx->io, G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
                                (GIOFunc)hci_cb, ctx);

    return TRUE;
}

static struct pairing_context *create_context(const char *bda, uint16_t clock, gboolean encrypt,
                                      btpair_cb cb, gpointer user_data, GError **err) {
    struct pairing_context *ctx;

    ctx = g_new0(struct pairing_context, 1);
    ctx->state     = STATE_DISCONNECTED;
    ctx->clock     = clock;
    ctx->encrypt   = encrypt;
    ctx->func      = cb;
    ctx->user_data = user_data;
    if (bda) {
        str2ba(bda, &ctx->bdaddr);
    }

    if (!hci_setup(ctx)) {
        g_set_error(err, BT_PAIR_ERROR,
                    BT_PAIR_ERROR_FAILED,
                    "Setting up HCI socket failed!");
        g_free(ctx);
        return NULL;
    }

    return ctx;
}

static gboolean pair_start(struct pairing_context *ctx, GError **err) {
    if (ctx->user_handle) {
        ctx->connected = TRUE;
        ctx->state = STATE_CONNECTED;

        debug("Using existing connection handle %d", ctx->handle);

        if (!request_auth(ctx)) {
            g_set_error(err, BT_PAIR_ERROR,
                        BT_PAIR_ERROR_FAILED,
                        "Sending HCI_Request_Authentication failed!");
            return FALSE;
        }
    }
    else {
        if (!create_connection(ctx)) {
            g_set_error(err, BT_PAIR_ERROR,
                        BT_PAIR_ERROR_CONNECT_FAILED,
                        "Sending HCI_Create_Connection failed!");
            return FALSE;
        }
    }

    return TRUE;
}

void pair_cancel(struct pairing_context *ctx) {
    GError *err = NULL;

    g_set_error(&err, BT_PAIR_ERROR,
                BT_PAIR_ERROR_CANCELED,
                "User canceled pairing");

    pair_finalize(ctx, err);
}

struct pairing_context *pair_bda(const char *bda, uint16_t clock, gboolean encrypt,
                                 btpair_cb cb, gpointer user_data, GError **err) {

    int handle = -1;
    struct pairing_context *ctx;

    ctx = create_context(bda, clock, encrypt, cb, user_data, err);

    if (!ctx) {
        return NULL;
    }

    get_bda_handle(&ctx->bdaddr, &handle);
    if (handle >= 0) {
        ctx->handle = (uint16_t)handle;
        ctx->user_handle = TRUE;
    }

    if (!pair_start(ctx, err)) {
        destroy_context(ctx);
        return NULL;
    }

    return ctx;
}

struct pairing_context *pair_handle(uint16_t handle, gboolean encrypt,
                                    btpair_cb cb, gpointer user_data,
                                    GError **err) {
    struct pairing_context *ctx;

    ctx = create_context(NULL, 0, encrypt, cb, user_data, err);

    if (!ctx) {
        return NULL;
    }

    ctx->handle = handle;
    ctx->user_handle = TRUE;

    if (!pair_start(ctx, err)) {
        destroy_context(ctx);
        return NULL;
    }

    return ctx;
}
