/**
  @file btpair-dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "log.h"
#include "dbus.h"
#include "dbus-helper.h"
#include "btpair-signal.h"
#include "btpair-engine.h"
#include "bttools-dbus.h"
#include "btpair-error.h"
#include "btpair-dbus.h"

extern GMainLoop *mainloop;

static gboolean pairing = FALSE;

#define EXIT_TIMEOUT 5000 /* milliseconds */

static GSource *exit_timer = NULL;

static gboolean exit_cb(gpointer user_data) {
    error("btpair running but no requests received.");
    g_main_loop_quit(mainloop);
    return FALSE;
}

static void set_exit_timer(void) {
    if (exit_timer)
        g_source_destroy(exit_timer);
    exit_timer = g_timeout_source_new(EXIT_TIMEOUT);
    g_source_set_callback(exit_timer, exit_cb, NULL, NULL);
    (void) g_source_attach(exit_timer, NULL);
}

static void remove_exit_timer(void) {
    if (exit_timer) {
        g_source_destroy(exit_timer);
        exit_timer = NULL;
    }
}

static void btpair_dbus_cb(GError *err, DBusMessage *msg) {
    DBusMessage *reply;

    if (err) {
        reply = new_dbus_error(msg, bt_pair_dbus_error(err), err->message);
    }
    else {
        reply = new_dbus_method_return(msg);
    }

    dbus_message_unref(msg);

    if (!send_and_unref(get_dbus_connection(), reply)) {
        error("Sending D-BUS reply failed");
    }

    g_main_loop_quit(mainloop);
}

static DBusHandlerResult btpair_req_handler(DBusConnection     *connection,
                                            DBusMessage        *message,
                                            void               *user_data) {
    DBusMessage *reply;
    DBusMessageIter iter;
    GError *err = NULL;
    PairingContext *ctx = NULL;
    dbus_bool_t encrypt = FALSE;

    remove_exit_timer();

    if (pairing) {
        reply = new_dbus_error(message, BTPAIR_ERROR_IN_PROGRESS, NULL);
        if (!send_and_unref(get_dbus_connection(), reply)) {
            error("Sending D-BUS reply failed");
        }
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    if (dbus_message_is_method_call(message,
                                    BTPAIR_INTERFACE,
                                    BTPAIR_REQ_PAIR_BDA)) {
        char *bda = NULL;
        int clock = 0;

        debug("Method call: pair_bda()");

        if (!dbus_message_iter_init(message, &iter) ||
            !get_dbus_iter_args(&iter,
                                DBUS_TYPE_STRING, &bda,
                                DBUS_TYPE_UINT32, &clock,
                                DBUS_TYPE_INVALID)) {
                send_invalid_args(connection, message);
                g_main_loop_quit(mainloop);
                return DBUS_HANDLER_RESULT_HANDLED;
        }

        if (dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_BOOLEAN) {
            get_dbus_iter_args(&iter,
                               DBUS_TYPE_BOOLEAN, &encrypt,
                               DBUS_TYPE_INVALID);
        }

        ctx = pair_bda(bda, clock, encrypt, (btpair_cb)btpair_dbus_cb, message, &err); 
    }

    if (dbus_message_is_method_call(message,
                                    BTPAIR_INTERFACE,
                                    BTPAIR_REQ_PAIR_HANDLE)) {
        int handle;

        if (!dbus_message_iter_init(message, &iter) || 
            !get_dbus_iter_args(&iter,
                                DBUS_TYPE_UINT32, &handle,
                                DBUS_TYPE_INVALID)) {
                send_invalid_args(connection, message);
                g_main_loop_quit(mainloop);
                return DBUS_HANDLER_RESULT_HANDLED;
        }

        if (dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_BOOLEAN) {
            get_dbus_iter_args(&iter,
                               DBUS_TYPE_BOOLEAN, &encrypt,
                               DBUS_TYPE_INVALID);
        }

        ctx = pair_handle(handle, encrypt, (btpair_cb)btpair_dbus_cb, message, &err); 
    }

    if (err) {
        reply = new_dbus_error(message, bt_pair_dbus_error(err), err->message);
        if (!send_and_unref(connection, reply)) {
            error("Sending D-BUS reply failed");
        }
        g_main_loop_quit(mainloop);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    if (ctx) {
        bind_unix_signals(ctx);
        pairing = TRUE;
        dbus_message_ref(message);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    error("Got unknown method call: %s", dbus_message_get_member(message));
    g_main_loop_quit(mainloop);

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusObjectPathVTable btpair_req_vtable = {
    .message_function    = btpair_req_handler,
    .unregister_function = NULL
};

/* Create bindings for D-Bus handlers */
void init_dbus_handlers(DBusConnection *connection) {
    DBusError derr;

    dbus_error_init(&derr);
    dbus_connection_register_object_path(connection,
                                         BTPAIR_PATH,
                                         &btpair_req_vtable,
                                         &derr);
    if (dbus_error_is_set(&derr)) {
        error("register_object_path(%s) failed: %s",
                BTPAIR_PATH, derr.message);
        dbus_error_free(&derr);
    }

    set_exit_timer();
}

void destroy_dbus_handlers(DBusConnection *connection) {
    dbus_connection_unregister_object_path(connection, BTPAIR_PATH);
}

