/**
  @file main.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <glib.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>

#include "log.h"
#include "bt-utils.h"
#include "key.h"

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#define HCI_TIMEOUT 100

#define HCI_DEV_ID 0

#define CMD_INVALID -1
#define CMD_LIST     0
#define CMD_REMOVE   1
#define CMD_CHECK    2

static char *program_name;

/** Command line config options */
static struct {
    char *filename;
    char *sba;
    char *dba;
    int   cmd;
} cfg;

/* For getopt */
static struct option const long_options[] = {
    {"help",     no_argument,       0, 'h'},
    {"version",  no_argument,       0, 'V'},
    {"list",     no_argument,       0, 'l'},
    {"check",    no_argument,       0, 'c'},
    {"remove",   no_argument,       0, 'r'},
#ifdef USE_FILENAME
    {"filename", required_argument, 0, 'f'},
#endif
    {"sba",      required_argument, 0, 's'},
    {"dba",      required_argument, 0, 'd'},
    {NULL, 0, NULL, 0}
};


#ifndef OCF_DELETE_LINK_KEY_REQ
# define OCF_DELETE_LINK_KEY_REQ 0x0012
typedef struct {
    bdaddr_t        bdaddr;
    uint8_t         delete_all;
} __attribute__ ((packed)) delete_link_key_req_cp;
# define DELETE_LINK_KEY_CP_SIZE 7

typedef struct {
    uint8_t         status;
    uint16_t        num_keys_deleted;
} __attribute__ ((packed)) delete_link_key_rp;
# define DELETE_LINK_KEY_RP_SIZE 3
#endif /* OCF_DELETE_LINK_KEY_REQ */


static gboolean hci_delete_link_key(gchar *bda) {
    delete_link_key_rp rp;
    delete_link_key_req_cp cp;
    struct hci_request rq;
    int sock;

    sock = hci_open_dev(HCI_DEV_ID);
    if (sock < 0) {
        error("hci_open_dev(%d): %s", HCI_DEV_ID, strerror(errno));
        return FALSE;
    }

    memset(&cp, 0, sizeof(cp));
    if (bda) {
        str2ba(bda, &cp.bdaddr);
        cp.delete_all = 0x00;
    }
    else {
        cp.delete_all = 0x01;
    }

    memset(&rq, 0, sizeof(rq));

    rq.ogf      = OGF_HOST_CTL;
    rq.ocf      = OCF_DELETE_LINK_KEY_REQ;
    rq.cparam   = &cp;
    rq.clen     = DELETE_LINK_KEY_CP_SIZE;
    rq.rparam   = &rp;
    rq.rlen     = DELETE_LINK_KEY_RP_SIZE;

    if (hci_send_req(sock, &rq, HCI_TIMEOUT) < 0) {
        close(sock);
        return FALSE;
    }

    close(sock);

    if (rp.status != 0x00) {
        error("HCI_Delete_Stored_Link_Key failed with 0x%02X", rp.status);
        return FALSE;
    }

    debug("%u keys deleted from BT controller", btohs(rp.num_keys_deleted));

    return TRUE;
}


#ifdef USE_FILENAME
static void drop_privileges(const char *username) {
    struct passwd *pw;

    if (getuid() != 0 && geteuid() != 0) {
        return;
    }


    pw = getpwnam(username);
    if (pw == NULL) {
        die("Unable to find user \"%s\"", username);
    }

    if (setgid(pw->pw_gid) < 0 || setuid(pw->pw_uid) < 0) {
        die("Failed to switch to user %s", username);
    }
}
#endif

/** Print usage information
 * @param status exit with status
 */
static void usage(int status) {
    printf("%s - Bluetooth Link Key Editor %s\n", program_name, VERSION);

    printf("Compilation flags: ");
#ifdef DEBUG
    printf("+DEBUG ");
#else
    printf("-DEBUG ");
#endif

    printf(
     "\nUsage: %s [OPTION]...\n"
     "Options:\n"
     "-h, --help         Display this help and exit\n"
     "-V, --version      Output version information and exit\n"
     "-l, --list         List stored keys\n"
     "-c, --check        Check if a key exists\n"
     "-r, --remove       Remove a stored key\n"
#ifdef USE_FILENAME
     "-f, --filename     Specify link key file\n"
#endif
     "-s, --sba          Specify source BT address\n"
     "-d, --dba          Specify destination BT address\n"
     "\n", program_name);

    exit(status);
}

/** Process commandline options.
 * @param argc Parameter given to main()
 * @param argv Parameter given to main()
 * @returns Index of first non-option argument
 */
static int decode_switches(int argc, char *argv[])
{
    int c;

    memset(&cfg, 0 , sizeof(cfg));
    cfg.cmd = CMD_INVALID;

    while ((c = getopt_long(argc, argv,
                    "h"   /* help      */
                    "V"   /* version   */
                    "l"   /* list      */
                    "r"   /* remove    */
                    "c"   /* check     */
#ifdef USE_FILENAME
                    "f:"  /* filename  */
#endif
                    "s:"  /* sba       */
                    "d:"  /* dba       */
                    ,long_options, NULL)) != EOF) {
        switch (c) {
            case 'V':
                printf("Bluetooth Link Key Editor %s\n", VERSION);
                exit(EXIT_SUCCESS);

            case 'h':
                usage(EXIT_SUCCESS);

            case 'l':
                cfg.cmd = CMD_LIST;
                break;

            case 'r':
                cfg.cmd = CMD_REMOVE;
                break;

            case 'c':
                cfg.cmd = CMD_CHECK;
                break;

#ifdef USE_FILENAME
            case 'f':
                drop_privileges();
                cfg.filename = g_strdup(optarg);
                break;
#endif

            case 's':
                if (!bda_ok(optarg)) {
                    fprintf(stderr, "Invalid BDA: %s\n", optarg);
                    usage(EXIT_FAILURE);
                }
                cfg.sba = g_strdup(optarg);
                break;

            case 'd':
                if (!bda_ok(optarg)) {
                    fprintf(stderr, "Invalid BDA: %s\n", optarg);
                    usage(EXIT_FAILURE);
                }
                cfg.dba = g_strdup(optarg);
                break;

            default:
                usage(EXIT_FAILURE);
        }
    }

    return optind;
}

int main(int argc, char *argv[]) {
    gboolean ret;
    int i;

    program_name = argv[0];
    i = decode_switches(argc, argv);

    open_log("btlk", TRUE);

    key_file_init(cfg.filename);

    switch (cfg.cmd) {
        case CMD_LIST:
            ret = list_link_keys(cfg.sba, cfg.dba);
            break;

        case CMD_REMOVE:
            ret = remove_link_key(cfg.sba, cfg.dba);
            if (!hci_delete_link_key(cfg.dba)) {
                error("Removing link key from BT controller failed");
            }
            break;

        case CMD_CHECK:
            ret = have_link_key(cfg.sba, cfg.dba);
            break;
            
        case CMD_INVALID:
        default:
            error("Nothing to do!");
            ret = FALSE;
            break;
    }

    if (ret == TRUE) {
        exit(EXIT_SUCCESS);
    } else {
        exit(EXIT_FAILURE);
    }
}

