/**
	@file callbacks.c

	Provides callbacks for the user interface for
	OSSO Application Installer

*/
/* 
 *
 * This file is part of Application Installer
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Marius Vollmer <marius.vollmer@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
#include "callbacks.h"
#include "interface.h"


/* Callback for install button */
void on_button_install_clicked(GtkButton *button, AppUIData *app_ui_data)
{
  gchar *filename = NULL;
  gboolean direct_install = FALSE;

  /* if we have param, lets have that instead of fcd */
  if ( (app_ui_data->param != NULL) &&
       (app_ui_data->param->len > 0) ) {
    filename = app_ui_data->param->str;
    fprintf(stderr, "install: having '%s' instead\n", filename);
    direct_install = TRUE;
  }
  else {
    filename = ui_show_file_chooser(app_ui_data);
  }

  /* Try installing if installation confirmed */
  if (0 != g_strcasecmp(filename, "")) {

    /* Disable buttons while installing */
    fprintf(stderr, "install: disabling buttons\n");
    gtk_widget_set_sensitive(app_ui_data->uninstall_button, FALSE);
    gtk_widget_set_sensitive(app_ui_data->installnew_button, FALSE);
    gtk_widget_set_sensitive(app_ui_data->close_button, FALSE);

    /* Try install */
    if (!install_package(filename, app_ui_data)) {
      ULOG_ERR("Installation of '%s' failed.", filename);
    } else {
      ULOG_INFO("Successfully installed '%s'.", filename);
    }

    if (!direct_install) {
      /* Represent packages and uninstall button state */ 
      represent_packages(app_ui_data);

      /* Enable buttons again, except uninstall, which is set
	 to proper value at represent_package */
      gtk_widget_set_sensitive(app_ui_data->installnew_button, TRUE);
      gtk_widget_set_sensitive(app_ui_data->close_button, TRUE);
    }
  } else {
    ULOG_WARN("No file selected for installing or filename is empty.");
  }

  /* shutting down since direct install */
  if (direct_install)
    gtk_widget_destroy(app_ui_data->main_dialog);
  /*
  g_signal_emit_by_name((gpointer)app_ui_data->main_dialog)
    "delete_event");
  */
}



/* Callback for uninstall button */
void on_button_uninstall_clicked(GtkButton *button, AppUIData *app_ui_data)
{
  gchar *package;
  package = ui_read_selected_package(app_ui_data);

  /* Try uninstalling if package was selected */
  if (0 != g_strcasecmp(package, "")) {
    gchar *dependencies = show_remove_dependencies(package);
    //fprintf(stderr, "Dependencies: '%s'\n", dependencies);

    /* Some other package is dependant on this one, cannot uninstall */
    if (0 != g_strcasecmp(dependencies, "")) {
      represent_dependencies(app_ui_data, _(dependencies));
      return;
    }

    /* Dim button while uninstalling */
    gtk_widget_set_sensitive(app_ui_data->uninstall_button, FALSE);
    gtk_widget_set_sensitive(app_ui_data->installnew_button, FALSE);
    gtk_widget_set_sensitive(app_ui_data->close_button, FALSE);

    /* Trying to uninstall */
    if (!uninstall_package(package, app_ui_data)) {
      ULOG_ERR("Uninstallation of '%s' failed.", package);
    } else {
      ULOG_INFO("Successfully uninstalled '%s'.", package);
    }

    /* Deselect description */
    gtk_text_buffer_set_text(GTK_TEXT_BUFFER(app_ui_data->main_label),
			       MESSAGE_DOUBLECLICK, -1);

    /* Update the package list (and uninstall button state) */
    represent_packages(app_ui_data);

    /* Enable buttons again, except uninstall, which is set
       to proper value at represent_package */
    gtk_widget_set_sensitive(app_ui_data->installnew_button, TRUE);
    gtk_widget_set_sensitive(app_ui_data->close_button, TRUE);

  } else {
    ULOG_WARN("No package selected for uninstall or package is empty.");
  }
}


void on_treeview_activated(GtkTreeView *treeview, GtkTreePath *path, 
			   GtkTreeViewColumn *col, AppUIData *app_ui_data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;

  model = gtk_tree_view_get_model(treeview);
  
  /* If something is selected */
  if (gtk_tree_model_get_iter(model, &iter, path)) {
    gchar *package = NULL;
    gchar *description = NULL;
    gchar *version = NULL;

    gtk_tree_model_get(model, &iter, 
		       COLUMN_NAME, &package, 
		       COLUMN_VERSION, &version,
		       -1);
    
    /* Dig up information, but only if its with version */
    /* No version = No packages installed text */
    if ( (version != NULL) && (strlen(version) > 0) ) {
	   description = show_description(package, DPKG_INFO_INSTALLED);

	   gtk_text_buffer_set_text(GTK_TEXT_BUFFER(app_ui_data->main_label), 
				    description, -1);
    }
    
    /* Free resources */
    g_free(version);
    g_free(package);
    if (description != NULL && 0 != strcmp("", description)) 
      g_free(description);
  }
}


void on_button_cancel_operation(GtkButton *button, gpointer dialog)
{
  terminate_dpkg();
  //gtk_widget_destroy(GTK_WIDGET(dialog));
}



/* progressbar timeout */
gboolean progressbar_timeout_cb(AppUIData *app_ui_data)
{
  fprintf(stderr, "timeout_cb: timeouted\n");
  //app_ui_data = (AppUIData *)app_ui_data;
  g_assert(app_ui_data != NULL);

  app_ui_data->current_progress += PROGRESSBAR_STEP;
  ui_set_progressbar(app_ui_data, app_ui_data->current_progress);

  return TRUE;
}



/* Callback for keypresses */
gboolean key_press(GtkWidget * widget, GdkEventKey * event, gpointer data)
{
  AppUIData *app_ui_data;
  app_ui_data = (AppUIData *) data;
	
  /* what does this do? */
  if (event->state & (GDK_CONTROL_MASK |
		      GDK_SHIFT_MASK |
		      GDK_MOD1_MASK |
		      GDK_MOD3_MASK | GDK_MOD4_MASK | GDK_MOD5_MASK)) {
    fprintf(stderr, "keypress, true\n");
    return TRUE;
  }

  /* What key was it */
  switch (event->keyval) {
    
  case GDK_Up:
    fprintf(stderr, "key up\n");
    break;
    
  case GDK_Left:
    fprintf(stderr, "key left\n");
    break;
    
  case GDK_Down:
    fprintf(stderr, "key down\n");
    break;
    
  case GDK_Right:
    fprintf(stderr, "key right\n");
    break;
  }

  fprintf(stderr, "keypress, ret false\n");
  return FALSE;
}


/*
Navigation keys		Arrow keys	GDK_Left, GDK_Right, GDK_Up, GDK_Down
Cancel (Escape)		Esc		GDK_Escape
Menu key		F4		GDK_F4
Home			F5		GDK_F5
Fullscreen		F6		GDK_F6
Plus (Zoom in)		F7		GDK_F7
Minus (Zoom out)	F8		GDK_F7
*/


/* JP 30.3.2005 non-repeating key handlers */
gboolean key_release(GtkWidget * widget, GdkEventKey * event, gpointer data)
{
  AppUIData *app_ui_data = (AppUIData *) data;
  g_assert(app_ui_data != NULL);
	
  /* whats this? */
  if (event->state & (GDK_CONTROL_MASK | GDK_SHIFT_MASK | GDK_MOD1_MASK |
		      GDK_MOD3_MASK | GDK_MOD4_MASK | GDK_MOD5_MASK)) {
    fprintf(stderr, "state on, returning true\n");
    return TRUE;
  }

  /* whats da key */
  switch (event->keyval) {
  case GDK_Return:
    fprintf(stderr, "key return\n");
    break;
    
    /* F6 = toggle full screen mode */
  case GDK_F6:
    fprintf(stderr, "key fullscreen\n");
    break;
	
    /* F8 = zoom out */
  case GDK_F8:
    fprintf(stderr, "key zoomout\n");
    break;
    
    /* F7 = zoom in */
  case GDK_F7:
    fprintf(stderr, "key zoomin\n");
    break;
    
    /* ESC = if fullscreen: back to normal */
  case GDK_Escape:
    fprintf(stderr, "key esc\n");
    break;

  case GDK_KP_Enter:
    fprintf(stderr, "key kpenter\n");
    break;
	
  case GDK_plus:
    fprintf(stderr, "key plus\n");
    break;
    
  case GDK_KP_Add:
    fprintf(stderr, "key kpadd\n");
    break;
	
  case GDK_KP_Down:
    fprintf(stderr, "key kpdown\n");
    break;
    
  case GDK_KP_Left:
    fprintf(stderr, "key kpleft\n");
    break;
    
  case GDK_KP_Up:
    fprintf(stderr, "key kpup\n");
    break;
	
  case GDK_KP_Right:
    fprintf(stderr, "key kpright\n");
    break;
    
  case GDK_minus:
    fprintf(stderr, "key minus\n");
    break;
    
  case GDK_KP_Subtract:
    fprintf(stderr, "key kpsubstract\n");
    break;
    
  }

  fprintf(stderr, "didnt match anything, return false\n");
  return FALSE;
}

