/**
    @file core.c

    Core functionality of OSSO Application Installer.

*/
/* 
 *
 * This file is part of Application Installer
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Marius Vollmer <marius.vollmer@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
#include "core.h"

/* are we using new g_spawn or old (buggy?) pipe+execvp */
#define USE_SPAWN_TO_LAUNCH 1


/* new version */
#ifdef USE_SPAWN_TO_LAUNCH

DpkgOutput use_dpkg(guint method, gchar **args, AppUIData *app_ui_data)
{
  int i = 0;
  gchar **package_items = NULL;
  DpkgOutput output;
  ListItems list_items;
  GString *cmdline = g_string_new("");
  gchar *cmd_stdout, *cmd_stderr;
  guint timerid = 0;
  gchar **buffer = NULL;

  fprintf(stderr, "use_dpkg: started, method %d\n", method);
  fprintf(stderr, "(1=list, 2=install, 3=uninstall, 4=fields, 5=status)\n");
  if ( (0 != method) && (0 < sizeof(args)) ) {
    g_assert(method);
    g_assert(args);

    output.out = g_string_new("");
    output.err = g_string_new("");
    output.list = g_array_new(FALSE, TRUE, sizeof(ListItems));

    i = 0;
    while (args[i] != NULL) {
      if (i > 0) cmdline = g_string_append(cmdline, " ");
      cmdline = g_string_append(cmdline, args[i++]);
    }

    /* app_ui_data is != null only at install/uninstall */
    if (app_ui_data != NULL) {
      fprintf(stderr, "use_dpkg: adding timeout 150ms\n");
      timerid = g_timeout_add(150, 
			      (gpointer)progressbar_timeout_cb,
			      (gpointer)app_ui_data);
      fprintf(stderr, "use_dpkg: got timerid '%d'\n", timerid);
      sleep(1);
      ui_forcedraw();
    }

    fprintf(stderr, "use_dpkg: cmdline '%s'\n", cmdline->str);
    g_spawn_command_line_sync(cmdline->str, &cmd_stdout, &cmd_stderr, 
			      NULL, NULL);

    /* removing progressbar updating timer after done */
    if (app_ui_data != NULL) {
      fprintf(stderr, "use_dpkg: removing timer\n");
      g_source_remove(timerid);
      ui_forcedraw();
    }

    fprintf(stderr, "cmd: '%s'\n", cmdline->str);
    fprintf(stderr, "out: '%s'\n", cmd_stdout);
    fprintf(stderr, "err: '%s'\n", cmd_stderr);


    /* Were done with spawn, now parsing our outputs */
    fprintf(stderr, "use_dpkg: preparing outputs\n");
    if (cmd_stdout == NULL && cmd_stderr == NULL) {
      fprintf(stderr, "****************************************\n");
      fprintf(stderr, "* FAKEROOT MISSING, UNABLE TO CONTINUE *\n");
      fprintf(stderr, "****************************************\n");
    }
    g_assert(cmd_stdout != NULL);
    g_assert(cmd_stderr != NULL);
    
    buffer = g_strsplit(cmd_stdout, "\n", 0);
    output.out = g_string_new(cmd_stdout);
    output.err = g_string_new(cmd_stderr);

    fprintf(stderr, "use_dpkg: going to stdout read loop\n");
    i = 0;
    while (buffer[i] != NULL) {
      /* break on empty line */
      if (0 == g_strcasecmp(g_strstrip(buffer[i]), "")) { 
	ULOG_INFO("use_dpkg: breaking on empty line at line %d\n", i+1);
	break; 
      }
      
      if (DPKG_METHOD_LIST == method) {
	/* Split out of the string which is formed as below: */
	/* package-name version size */
	package_items = g_strsplit(buffer[i], " ", 0);
	list_items.name = strdup(package_items[0]);
	list_items.version = strdup(package_items[1]);
	list_items.size = strdup(package_items[2]);
	g_array_append_val(output.list, list_items);
      }

      i++;
    }
    
    /* Terminate list with NULL */
    if (DPKG_METHOD_LIST == method) {
      list_items.name = list_items.version = list_items.size = NULL;
      g_array_append_val(output.list, list_items);
    }
    
    g_strfreev(buffer);
  }

  fprintf(stderr, "use_dpkg: finished method '%d'\n", method);
  return output;
}

/* old version */
#else

DpkgOutput use_dpkg(guint method, gchar **args, AppUIData *app_ui_data)
{
  int i;
  FILE *out_fp;
  FILE *err_fp;
  gchar **package_items = NULL;
  gdouble completed = 0.0;
  gdouble progress_step = 0.125;
  gint fork_result = -1;
  gint out_pipe[2];
  gint err_pipe[2];
  DpkgOutput output;
  ListItems list_items;

  fprintf(stderr, "use_dpkg: started, method %d\n", method);
  fprintf(stderr, "(1=list, 2=install, 3=uninstall, 4=fields, 5=status)\n");
  if ( (0 != method) && (0 < sizeof(args)) ) {
    g_assert(method);
    g_assert(args);

    output.out = g_string_new("");
    output.err = g_string_new("");
    output.list = g_array_new(FALSE, TRUE, sizeof(ListItems));

    ULOG_INFO("use_dpkg: piping");
    if ( (0 == pipe(out_pipe)) && (0 == pipe(err_pipe)) ) {
      ULOG_INFO("use_dpkg: forking");
      fork_result = fork();

      /* If forking fails */
      if (-1 == fork_result) {
	ULOG_INFO("use_dpkg: forking failed");

        exit(EXIT_FAILURE);
      /* We're in child process */
      } else if (0 == fork_result) {
	fprintf(stderr, "use_dpkg: in_child\n");

        close(1);
        dup(out_pipe[1]);
        close(out_pipe[0]);
        close(out_pipe[1]);

        close(2);
        dup(err_pipe[1]);
        close(err_pipe[0]);
        close(err_pipe[1]);

        /* Set new PATH variable */
        GString *tmp = g_string_new(g_getenv(PATH_ENV));
        tmp = g_string_append(tmp, PATH_ADD);
        g_setenv(PATH_ENV, tmp->str, TRUE);

	/* Not the same !!*/
	g_unsetenv("LD_PRELOAD");
	//g_setenv("LD_PRELOAD","\"\"", TRUE);
       

        //fprintf(stderr,"use_dpkg: execvp args:\n");
	ULOG_INFO("use_dpkg: execvp");
        i=0;
        while (args[i]!=NULL)
	{
	  //fprintf(stderr, "arg[%i]: %s\n ", i, args[i]);
	  i++;
	}

	fprintf(stderr, "use_dpkg: child at end, doing EXECVP\n");
	if (execvp(args[0], args)) fprintf(stderr, "*****\nOhshit\n*****\n");

      /* We're in parent process */
      } else {
	fprintf(stderr, "use_dpkg: in_parent\n");
	sleep(2);
	//fprintf(stderr, "use_dpkg: going to waitpid(%d)\n", fork_result);
	//waitpid(fork_result);
	//fprintf(stderr, "use_dpkg: returned from waitpid()\n");

        /* Started reading stdout */
	ULOG_INFO("use_dpkg: parent: close and open pipe for reading");
        close(out_pipe[1]);
        out_fp = fdopen(out_pipe[0], "r");

        gchar temp[READ_BUFFER_LENGTH+1];
        gchar *buffer;
        buffer = temp;

	if (DPKG_METHOD_UNINSTALL == method)
	  progress_step = 0.13;
	if (DPKG_METHOD_INSTALL == method)
	  progress_step = 0.16;

	/* Things fail without this sleep(), don't remove until you
	   find cause and fix it */
	ULOG_INFO("going to stdout read loop");
        while (fgets(buffer, READ_BUFFER_LENGTH, out_fp)) {
          /* break on empty line */
          if (0 == g_strcasecmp(g_strstrip(buffer), "")) { break; }

	  //fprintf(stderr, "dpkg: out: %s\n", buffer);

          /* Update progress bar when new status line appears */
	  if (app_ui_data != NULL) {
	    if (g_str_has_prefix(buffer, DPKG_STATUS_PREFIX)) {
	      completed += progress_step;
	      fprintf(stderr, "going to set progessbar to %f", completed);
	      ui_set_progressbar(app_ui_data, completed);
	    }
          }

          if (DPKG_METHOD_LIST == method) {
            /* Split out of the string which is formed as below: */
            /* package-name version size */
            package_items = g_strsplit(buffer, " ", 0);
            list_items.name = strdup(package_items[0]);
            list_items.version = strdup(package_items[1]);
            list_items.size = strdup(package_items[2]);
            g_array_append_val(output.list, list_items);
          }

          output.out = g_string_append(output.out, buffer);
          output.out = g_string_append_c(output.out, '\n');
        }

        /* Terminate list with NULL */
        if (DPKG_METHOD_LIST == method) {
          list_items.name = list_items.version = list_items.size = NULL;
          g_array_append_val(output.list, list_items);
        }

        /* Stopped reading stdout */
	ULOG_INFO("use_dpkg: closing stdout, opening stderr");
        fclose(out_fp);
        close(out_pipe[0]);

        /* Started reading stderr */
	ULOG_INFO("use_dpkg: stderr");
        close(err_pipe[1]);
        err_fp = fdopen(err_pipe[0], "r");

        while (fgets(buffer, READ_BUFFER_LENGTH, err_fp)) {
	  //fprintf(stderr, "dpkg: err: %s", buffer);
          output.err = g_string_append(output.err, buffer);
        }

        /* Stopped reading stderr */
        fclose(err_fp);
        close(err_pipe[0]);

	fprintf(stderr, "use_dpkg: parent FINISHED\n");
      }
    }
  }

  ULOG_INFO("use_dpkg: final\n");
  return output;
}
#endif


void terminate_dpkg() {
  fprintf(stderr, "terminate button clicked, doing nothing\n");
}



GtkTreeModel *list_packages(void)
{
  gint i = 0;
  GtkListStore *store;
  GtkTreeIter iter;
  ListItems *package;
  DpkgOutput output;
  gchar *args[5];

  /* Dpkg query params for needed information */
#ifdef USE_SPAWN_TO_LAUNCH
  args[0] = DPKG_QUERY_BINARY;
  args[1] = "-W";
  args[2] = "--admindir=/var/lib/install/var/lib/dpkg";
  args[3] = "--showformat=\\${Package}\\ \\${Version}\\ \\${Installed-Size}\\\\n";
  args[4] = (gchar *) NULL;

#else

  args[0] = DPKG_QUERY_BINARY;
  args[1] = "-W";
  args[2] = "--admindir=/var/lib/install/var/lib/dpkg";
  args[3] = "--showformat=${Package} ${Version} ${Installed-Size}\n";
  args[4] = (gchar *) NULL;
#endif

  output = use_dpkg(DPKG_METHOD_LIST, args, NULL);

  store = gtk_list_store_new (NUM_COLUMNS,
			      G_TYPE_OBJECT,
                              G_TYPE_STRING,
                              G_TYPE_STRING,
                              G_TYPE_STRING);

  /* No packages installed */
  if ( ((package = &g_array_index(output.list, ListItems, 1)) == NULL) || 
       (package->name == NULL) ) {

    gtk_list_store_append(store, &iter);
    gtk_list_store_set(store, &iter,
		       COLUMN_ICON, NULL,
		       COLUMN_NAME, _("ai_ti_application_installer_nopackages"),
		       COLUMN_VERSION, "",
		       COLUMN_SIZE, "",
		       -1
		       );

    return GTK_TREE_MODEL(store);
  }


  /* Some packages installed */
  for (i = 0; (package = &g_array_index(output.list, ListItems, i)) &&
       (NULL != package->name); i++) {
    if (0 != g_strcasecmp(package->name, META_PACKAGE)) {
      GString *version = g_string_append(g_string_new("v"), package->version);
      GString *size = g_string_append(g_string_new(package->size), "kB");

      /* Load icon */
      GdkPixbuf *icon = NULL;
      icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
				      "qgn_list_gene_default_app",
				      24,
				      0,
				      NULL);

      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter,
			 COLUMN_ICON, icon, 
                         COLUMN_NAME, package->name,
                         COLUMN_VERSION, version->str,
                         COLUMN_SIZE, size->str,
                         -1);

      /* Release icon space */
      g_object_unref(icon);
    }
  }

  return GTK_TREE_MODEL(store);
}



void add_columns(GtkTreeView *treeview)
{
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
 
  /* Icon column */
  renderer = gtk_cell_renderer_pixbuf_new();
  column = gtk_tree_view_column_new_with_attributes("Icon",
						    renderer,
						    "pixbuf",
						    COLUMN_ICON,
						    NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_ICON);
  gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_append_column(treeview, column);

  /* Name column */
  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes("Name",
                                                    renderer,
                                                    "text",
                                                    COLUMN_NAME,
                                                    NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_NAME);
  gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_GROW_ONLY);
  gtk_tree_view_append_column(treeview, column);


  /* Version column */
  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes("Version",
                                                    renderer,
                                                    "text",
                                                    COLUMN_VERSION,
                                                    NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_VERSION);
  gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_append_column(treeview, column);


  /* Size column */
  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes("Size",
                                                    renderer,
                                                    "text",
                                                    COLUMN_SIZE,
                                                    NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_SIZE);
  gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
  gtk_tree_view_append_column(treeview, column);
}



PackageInfo package_info(gchar *package_or_deb, gint installed)
{
  PackageInfo info;
  gchar *dpkg_args[8];
  gchar *name = NULL, *size = NULL, *description = NULL, *version = NULL;
  gchar **row, **namerow, **sizerow, **descriptionrow, **versionrow;
  DpkgOutput output;

  info.name = NULL;
  info.size = NULL;
  info.description = NULL;
  info.version = NULL;

  if (package_or_deb == NULL) {
    fprintf(stderr, "Package_or_deb empty, returning NULL\n");
    return info;
  }

  /*
  fprintf(stderr, "Digging information for %s '%s'\n", 
	  installed == DPKG_INFO_INSTALLED ? 
	  "installed application" : "package",
	  package_or_deb);
  */

  /* Information from .deb file */
  if (installed == DPKG_INFO_PACKAGE) {
    dpkg_args[0] = DPKG_BINARY;
    dpkg_args[1] = "-f";
    dpkg_args[2] = package_or_deb;
    dpkg_args[3] = "Package";
    dpkg_args[4] = "Version";
    dpkg_args[5] = "Installed-Size";
    dpkg_args[6] = "Description";
    dpkg_args[7] = (gchar *)NULL;

    /* Send command to dpkg */
    output = use_dpkg(DPKG_METHOD_FIELDS, dpkg_args, NULL);


    /* If invalid files */
    if ( (0 != g_strrstr(output.err->str, DEB_FORMAT_FAILURE)) ||
	 (0 != g_strrstr(output.err->str, DEB_FORMAT_FAILURE2)) ) {
      fprintf(stderr, "pkg_info: invalid format\n");
      return info;
    }

    /* Split rows from the output */
    row = g_strsplit(output.out->str, "\n", 4);
    /* Split fields from the row */
    namerow = g_strsplit(row[0], ":", 2);
    versionrow = g_strsplit(row[1], ":", 2);
    sizerow = g_strsplit(row[2], ":", 2);
    descriptionrow = g_strsplit(row[3], ":", 2);
    /* Strip empty chars */
    name = g_strstrip(g_strdup(namerow[1]));
    version = g_strstrip(g_strdup(versionrow[1]));
    size = g_strstrip(g_strdup(sizerow[1]));
    description = g_strstrip(g_strdup(descriptionrow[1]));

    /* Free string arrays */
    g_strfreev(row);
    g_strfreev(namerow);
    g_strfreev(versionrow);
    g_strfreev(sizerow);
    g_strfreev(descriptionrow);
  }

  /* Information from installed package */
  else {
    if (0 == strcmp("", package_or_deb)) {
      return info;
    }

    dpkg_args[0] = DPKG_QUERY_BINARY;
    dpkg_args[1] = "--status";
    dpkg_args[2] = "--admindir=/var/lib/install/var/lib/dpkg";
    dpkg_args[3] = package_or_deb;
    dpkg_args[4] = (gchar *)NULL;

    output = use_dpkg(DPKG_METHOD_STATUS, dpkg_args, NULL);
  
    if ( (0 != g_strrstr(output.err->str, DPKG_NO_SUCH_PACKAGE)) ||
	 (0 != g_strrstr(output.out->str, DPKG_NO_SUCH_PACKAGE2)) ) {
      fprintf(stderr, "No such package '%s' installed\n", package_or_deb);
      return info;
    }

    /* Lets dig up pointers to strings we need */
    gchar *version_ptr = g_strrstr(output.out->str, DPKG_VERSION_PREFIX);
    gchar *size_ptr = g_strrstr(output.out->str, DPKG_SIZE_PREFIX);
    gchar *desc_ptr = g_strrstr(output.out->str, DPKG_DESCRIPTION_PREFIX);

    g_assert(version_ptr != NULL);
    g_assert(size_ptr != NULL);
    g_assert(desc_ptr != NULL);

    /* Split them nicely to components we can use */
    versionrow = g_strsplit(version_ptr, ": ", 2);
    sizerow = g_strsplit_set(size_ptr, ":\n", 2);
    descriptionrow = g_strsplit(desc_ptr, ": ", 2);

    /* And take a copy of them for us */
    name = package_or_deb;
    size = g_strstrip(g_strdup(sizerow[1]));
    version = g_strstrip(g_strdup(versionrow[1]));
    description = g_strstrip(g_strdup(descriptionrow[1]));

    /* Free the arrays */
    g_strfreev(sizerow);
    g_strfreev(descriptionrow);
    g_strfreev(versionrow);
  }

  /* Construct the info package */
  info.name = g_string_new(name);
  info.size = g_string_new(size);
  info.version = g_string_new(version);
  info.description = g_string_new(description);

  return info;
}
  


/* Are there any packages installed (except maemo) */
gboolean any_packages_installed(GtkTreeModel *model) 
{
  GtkTreeIter iter;
  gboolean true_if_any = FALSE;
  gchar *size = NULL;

  /* Lets get a list of packages and its first entry */
  if (model == NULL) 
    model = list_packages();
  true_if_any = gtk_tree_model_get_iter_first(model, &iter);

  /* If entry size is empty, its "No Packages" text */
  gtk_tree_model_get(model, &iter, COLUMN_SIZE, &size, -1);
  if (0 == strcmp("", size))
    true_if_any = FALSE;
  g_free(size);

  ULOG_INFO("any_installed: List had items: %s", 
	    (true_if_any ? "true" : "false"));
  return true_if_any;
}



gchar *show_description(gchar *package, gint installed)
{
  PackageInfo info = package_info(package, installed);

  if (info.description != NULL) {
    return info.description->str;
  }

  return "";
}



gchar *show_remove_dependencies(gchar *package)
{
  gchar *dpkg_args[8];
  DpkgOutput output;
  GString *deps = g_string_new("");

  /* We can get dependencies only by --simulating --remove */
  dpkg_args[0] = FAKEROOT_BINARY;
  dpkg_args[1] = DPKG_BINARY;
  dpkg_args[2] = "--simulate";
  dpkg_args[3] = "--force-not-root";
  dpkg_args[4] = "--root=/var/lib/install";
  dpkg_args[5] = "--remove";
  dpkg_args[6] = package;
  dpkg_args[7] = (gchar *) NULL;

  output = use_dpkg(DPKG_METHOD_STATUS, dpkg_args, NULL);
  
  /* If we have erroneous output, it probably means dependency issues */
  if (0 < output.err->len) {

    /* If the whole err has even one "depends on", we have problem */
    if (0 != g_strrstr(output.err->str, DPKG_ERROR_DEPENDENCY)) {
      gint i = 0;
      gchar **errors = g_strsplit_set(output.err->str, " \n", 0);
      GString *prev = NULL;

      /* Checking all lines for dependent packages */
      for (i=0; errors[i] != NULL; i++) {
	GString *this = g_string_new(errors[i]);

	/* Trying to find lines matching to this: */
	/*  example: 
	    file-roller depends on bzip2 (>= 1.0.1). */
	if (this->len > 0 && 0 == g_strcasecmp(this->str, "depends")) {
	  deps = g_string_append(deps, prev->str);
	  deps = g_string_append(deps, " ");

	} else {
	  prev = g_string_new(this->str);
	}
      }
      
      g_strfreev(errors);
    }
  }

  return deps->str;
}



gboolean install_package(gchar *deb, AppUIData *app_ui_data) 
{
  gchar *install_args[12];

  /* Start installation if deb package is given */
  if (0 != g_strcasecmp(deb, "")) {
    ULOG_INFO("Started installing '%s'..", deb);

    /* Install command */
    install_args[0] = FAKEROOT_BINARY;
    install_args[1] = DPKG_BINARY;
    install_args[2] = "--force-not-root";
    install_args[3] = "--root=/var/lib/install";
    install_args[4] = "--refuse-depends";
    install_args[5] = "--status-fd";
    install_args[6] = "1";
    install_args[7] = "--abort-after";
    install_args[8] = "1";
    install_args[9] = "--install";
    install_args[10] = deb;
    install_args[11] = (gchar *) NULL;

    /* If package is already installed, quit & ask to remove it first */
    PackageInfo deb_info = package_info(deb, DPKG_INFO_PACKAGE);
    if (NULL != deb_info.name) {
      PackageInfo pkg_info = package_info(deb_info.name->str, 
					  DPKG_INFO_INSTALLED);
      
      if (NULL != pkg_info.name) {
	represent_error(app_ui_data, pkg_info.name->str,
			_("ai_error_alreadyinstalled"));
	return FALSE;
      }
    }

    /* Confirm installation from user */
    ULOG_INFO("Trying to confirm install..");
    if (!represent_confirmation(app_ui_data, DIALOG_CONFIRM_INSTALL, deb)) {
      ULOG_INFO("User cancelled installation.");
      return FALSE;
    }

    /* Use dpkg to install package and get output */
    ui_create_progressbar_dialog(app_ui_data,
				 _("ai_ti_installing_installing"),
				 _("ai_bd_installing_cancel"));
    DpkgOutput output = use_dpkg(DPKG_METHOD_INSTALL, install_args,
				 app_ui_data);
    ui_cleanup_progressbar_dialog(app_ui_data);


    /* If the file is not a deb package */
    if (NULL != g_strrstr(output.err->str, DEB_FORMAT_FAILURE) ||
        NULL != g_strrstr(output.err->str, DEB_FORMAT_FAILURE2)) {
      ULOG_INFO("Tried to install a non-deb file: '%s'.", deb);
     
      /* Show notification */
      represent_error(app_ui_data, deb, _("ai_error_corrupted"));
      return FALSE;
    }

    /* If installation succeeds */
    if (NULL != g_strrstr(output.out->str, DPKG_STATUS_SUFFIX_INSTALL)) {
      ULOG_INFO("Installation of '%s' succeeded.", deb);

      /* Get package's name from deb package and show notification */
      PackageInfo info = package_info(deb, DPKG_INFO_PACKAGE);

      /* Installation succeeded */
      represent_notification(app_ui_data,
			     info.name->str,
			     _("ai_ti_application_installed_text%s"),
			     _("ai_bd_application_installed_ok"),
			     info.name->str);

      gtk_text_buffer_set_text(GTK_TEXT_BUFFER(app_ui_data->main_label),
			       MESSAGE_DOUBLECLICK, -1);

      return TRUE;

    /* If installation fails */
    } else {
      ULOG_INFO("Installation of '%s' failed.", deb);

      PackageInfo info = package_info(deb, DPKG_INFO_PACKAGE);
      gchar *err = g_strdup_printf(_("ai_ti_installation_failed_text%s"),
				   info.name->str);

      /* Installation failed, do you wanna see details */
      if (represent_confirmation(app_ui_data, DIALOG_SHOW_DETAILS, err)) {

	/* Show error to user */
	gchar *errormsg = verbalize_error(output.err->str);
	represent_error(app_ui_data, info.name->str, errormsg);
      }

      /* Remove any remains if any left from that, many packages
	 install as non-configured, if they fail deps */
      gchar *uninstall_args[9];
      uninstall_args[0] = FAKEROOT_BINARY;
      uninstall_args[1] = DPKG_BINARY; 
      uninstall_args[2] = "--force-not-root";
      uninstall_args[3] = "--root=/var/lib/install";
      uninstall_args[4] = "--status-fd";
      uninstall_args[5] = "1";
      uninstall_args[6] = "--purge";
      uninstall_args[7] = info.name->str;
      uninstall_args[8] = (gchar *) NULL;
      
      use_dpkg(DPKG_METHOD_UNINSTALL, uninstall_args, NULL);

      g_free(err);
      return FALSE;
    }
  }

  return FALSE;
}



gboolean uninstall_package(gchar *package, AppUIData *app_ui_data)
{
  gchar *uninstall_args[9];

  /* Arg array for dpkg uninstall */
  uninstall_args[0] = FAKEROOT_BINARY;
  uninstall_args[1] = DPKG_BINARY; 
  uninstall_args[2] = "--force-not-root";
  uninstall_args[3] = "--root=/var/lib/install";
  uninstall_args[4] = "--status-fd";
  uninstall_args[5] = "1";
  uninstall_args[6] = "--purge";
  uninstall_args[7] = package;
  uninstall_args[8] = (gchar *) NULL;


  /* Start uninstallation if package is given */
  if (0 != g_strcasecmp(package, "")) {
    ULOG_INFO("Started uninstalling '%s'..", package);
    gchar *deps = show_remove_dependencies(package);

    /* Some dependencies found */
    if (0 != g_strcasecmp(deps, "")) {
      represent_dependencies(app_ui_data, deps);
      return FALSE;
    }

    /* Confirm uninstallation from user */
    if (!represent_confirmation(app_ui_data, DIALOG_CONFIRM_UNINSTALL, 
				package)) {
      ULOG_INFO("User cancelled uninstallation.");
      return FALSE;
    }


    /* Use dpkg to install package and get output */
    ui_create_progressbar_dialog(app_ui_data,
				 _("ai_ti_uninstall_progress_uninstalling"),
				 _("ai_bd_installing_cancel"));
    DpkgOutput output = use_dpkg(DPKG_METHOD_UNINSTALL, uninstall_args,
				 app_ui_data);
    ui_cleanup_progressbar_dialog(app_ui_data);
    
    //fprintf(stderr, "out: '%s'\n", output.out->str);
    //fprintf(stderr, "err: '%s'\n", output.err->str);
    
    /* Uninstallation succeeds */
    if (0 == g_strrstr(output.err->str, DPKG_STATUS_UNINSTALL_ERROR)) {
      ULOG_INFO("Uninstallation of '%s' succeeded.", package);
      
      /* Show notification to user */
      represent_notification(app_ui_data,
			     package,
			     _("ai_ti_application_uninstalled_text%s"), 
			     _("ai_bd_application_uninstalled_ok"),
			     package);
      
      return TRUE;
      
      /* If uninstallation fails */
    } else {
      ULOG_INFO("Uninstallation of '%s' failed.", package);
      
      represent_notification(app_ui_data, 
			     package,
			     _("ai_ti_uninstallation_failed_text%s"), 
			     _("ai_ti_uninstallation_failed_ok"),
			     package);

      return FALSE;
    }
  }

  return FALSE;
}




gchar *verbalize_error(gchar *dpkg_err) 
{
  if (0 != g_strrstr(dpkg_err, DEB_FORMAT_FAILURE))
    return _("ai_error_corrupted");

  if (0 != g_strrstr(dpkg_err, DEB_FORMAT_FAILURE2))
    return _("ai_error_corrupted");

  if (0 != g_strrstr(dpkg_err, DPKG_ERROR_LOCKED))
    return "FATAL: Something went wrong, dpkg database area is locked. "
      "Reboot machine to unlock.";

  if (0 != g_strrstr(dpkg_err, DPKG_ERROR_INCOMPATIBLE))
    return _("ai_error_incompatible");

  if (0 != g_strrstr(dpkg_err, DPKG_FIELD_ERROR)) {
    GString *err = g_string_new("");
    dpkg_err = g_strstr_len(dpkg_err, 1024, DPKG_ERROR_INSTALL_DEPENDS);

    while ( (dpkg_err != NULL) && strlen(dpkg_err) > 0) {
      gchar **row = g_strsplit(dpkg_err, " ", 5);
      gchar *dep = g_strdup_printf(_("ai_error_componentmissing%s"), row[3]);

      //fprintf(stderr, "found dep: '%s'\n", row[3]);
      err = g_string_append(err, dep);
      err = g_string_append(err, "\n");
      dpkg_err = (char *)g_strstr_len((char *)(dpkg_err+sizeof(char *)), 1024, 
				      DPKG_ERROR_INSTALL_DEPENDS);

      g_free(dep);
      g_strfreev(row);
    }

    return err->str;
  }

  return dpkg_err;
}

