/**
 * @file osso-hw.c
 * This file implements the device signal handler functions.
 * 
 * Copyright (C) 2005  Nokia
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/* Ideally, these definitions should come from elsewhere,
   but it is not possible at the moment... (bug #10866) */

#define MCE_SERVICE                    "com.nokia.mce"
#define MCE_REQUEST_PATH               "/com/nokia/mce/request"
#define MCE_REQUEST_IF                 "com.nokia.mce.request"
#define MCE_SIGNAL_PATH                        "/com/nokia/mce/signal"
#define MCE_SIGNAL_IF                  "com.nokia.mce.signal"
#define MCE_DISPLAY_ON_REQ             "req_display_state_on"
#define MCE_PREVENT_BLANK_REQ          "req_display_blanking_pause"
#define MCE_DEVICE_MODE_SIG            "sig_device_mode_ind"

#define MCE_NORMAL_MODE                "normal"        /* normal mode */
#define MCE_FLIGHT_MODE                "flight"        /* flight mode */
#define MCE_OFFLINE_MODE       "offline"       /* offline mode; unsupported! */
#define MCE_INVALID_MODE       "invalid"       /* should *never* occur! */
#define MCE_SHUTDOWN_SIG           "shutdown_ind"

#include "osso-internal.h"
#include "osso-hw.h"

osso_return_t osso_display_state_on(osso_context_t *osso)
{
  DBusMessage *msg = NULL;
  dbus_bool_t b;
  
  if (osso->sys_conn == NULL) {
    DLOG_OPEN("libosso");
    DLOG_ERR_F("error: no sys D-BUS connection!");
    LOG_CLOSE();
    return OSSO_ERROR;
  }
  msg = dbus_message_new_method_call(MCE_SERVICE, MCE_REQUEST_PATH,
				     MCE_REQUEST_IF, MCE_DISPLAY_ON_REQ);
  if (msg == NULL)
    {
      return OSSO_ERROR;
    }

  b = dbus_connection_send(osso->sys_conn, msg, NULL);
  if (!b)
    {
      return OSSO_ERROR;
    }
  dbus_connection_flush(osso->sys_conn);
  dbus_message_unref(msg);
  return OSSO_OK;
}

osso_return_t osso_display_blanking_pause(osso_context_t *osso)
{
  DBusMessage *msg = NULL;
  dbus_bool_t b;
  
  if (osso->sys_conn == NULL) {
    DLOG_OPEN("libosso");
    DLOG_ERR_F("error: no sys D-BUS connection!");
    LOG_CLOSE();
    return OSSO_ERROR;
  }
  msg = dbus_message_new_method_call(MCE_SERVICE, MCE_REQUEST_PATH,
				     MCE_REQUEST_IF, MCE_PREVENT_BLANK_REQ);
  if (msg == NULL)
    {
      return OSSO_ERROR;
    }

  b = dbus_connection_send(osso->sys_conn, msg, NULL);
  if (!b)
    {
      return OSSO_ERROR;
    }
  dbus_connection_flush(osso->sys_conn);
  dbus_message_unref(msg);
  return OSSO_OK;
}

osso_return_t osso_hw_set_event_cb(osso_context_t *osso,
				   osso_hw_state_t *state,
				   osso_hw_cb_f *cb, gpointer data)
{
    osso_hw_state_t temp = {TRUE,TRUE,TRUE,TRUE,'\1'};
    if( (osso == NULL) || (cb == NULL) )
	return OSSO_INVALID;
    if (osso->sys_conn == NULL) {
	DLOG_OPEN("libosso");
	DLOG_ERR_F("error: no D-BUS connection!");
	LOG_CLOSE();
	return OSSO_INVALID;
    }

    dprint("state = %p", state);
    if(state == NULL)
	state = &temp;
    dprint("state = %p", state);

    osso->hw_state = _osso_hw_state_get();

    /* Now set the HW state / add monitoring if requested */
    
    _set_state_cb(shutdown_ind, cb, data);
    _set_state_cb(memory_low_ind, cb, data);
    _set_state_cb(save_unsaved_data_ind, cb, data);
    _set_state_cb(system_inactivity_ind, cb, data);
    _set_state_cb(sig_device_mode_ind, cb, data);
    _msg_handler_set_cb_f(osso, MCE_SIGNAL_IF, _hw_handler, state,
			  FALSE);

    return OSSO_OK;    
}

osso_return_t osso_hw_unset_event_cb(osso_context_t *osso,
				     osso_hw_state_t *state)
{
    osso_hw_state_t temp = {TRUE,TRUE,TRUE,TRUE,'\1'};
    if( osso == NULL )
	return OSSO_INVALID;
    if (osso->sys_conn == NULL) {
	DLOG_OPEN("libosso");
	DLOG_ERR_F("error: no D-BUS connection!");
	LOG_CLOSE();
	return OSSO_INVALID;
    }

    dprint("state = %p", state);
    if(state == NULL)
	state = &temp;
    dprint("state = %p", state);

    _unset_state_cb(shutdown_ind);
    _unset_state_cb(memory_low_ind);
    _unset_state_cb(save_unsaved_data_ind);
    _unset_state_cb(system_inactivity_ind);
    _unset_state_cb(sig_device_mode_ind);

    if(_state_is_unset())
	_msg_handler_rm_cb_f(osso, MCE_SIGNAL_IF,
			     _hw_handler, FALSE);
    
    return OSSO_OK;    
}

static osso_hw_state_t * _osso_hw_state_get(void)
{
    FILE *f;
    gchar storedstate[STORED_LEN];
    osso_hw_state_t *state = NULL;
    
    state = (osso_hw_state_t*)malloc(sizeof(osso_hw_state_t));
    if(state == NULL) return NULL;
    
    f = fopen(OSSO_DEVSTATE_SHUTDOWN_FILE, "r");
    if(f != NULL) {
	fclose(f);
 	state->shutdown_ind = TRUE;
    }
    else {
 	state->shutdown_ind = FALSE;
    }

    f = fopen(OSSO_DEVSTATE_MODE_FILE, "r");
    if(f != NULL) {
      if (fgets(storedstate, STORED_LEN, f) != NULL) {
	
	  if (strcmp(storedstate, MCE_NORMAL_MODE) == 0) {
	    state->sig_device_mode_ind = g_strdup(MCE_NORMAL_MODE);
	  }
	  else if (strcmp(storedstate, MCE_FLIGHT_MODE) == 0) {
	    state->sig_device_mode_ind = g_strdup(MCE_FLIGHT_MODE);
	  }
	  else if (strcmp(storedstate, MCE_OFFLINE_MODE) == 0) {
	    state->sig_device_mode_ind = g_strdup(MCE_OFFLINE_MODE);
	  }
	  else if (strcmp(storedstate, MCE_INVALID_MODE) == 0) {
	    state->sig_device_mode_ind = g_strdup(MCE_INVALID_MODE);
	  }
      }
      fclose(f);
    }
    else
      state->sig_device_mode_ind = NULL;
    
    return state;
}

static DBusHandlerResult _hw_handler(osso_context_t *osso,
				     DBusMessage *msg,
				     gpointer data)
{
    const gchar *signal = NULL, *mode_str = NULL;
    gboolean state = FALSE;
    DBusMessageIter iter;
    int type;
    
    dprint("");
    signal = dbus_message_get_member(msg);
    dbus_message_iter_init(msg, &iter);

    type = dbus_message_iter_get_arg_type(&iter);

    if (signal == NULL)
      {
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
      }
    
    dprint("argument type = '%c'",type);

    /* We need to get the argument types for shutdown and mode
       signals, others have no arguments specified */

    if (strcmp(signal, MCE_SHUTDOWN_SIG) == 0 && type != DBUS_TYPE_BOOLEAN)
      {
	DLOG_OPEN("libosso");
	DLOG_WARN_F("warning: invalid arguments in HW signal '%s'", signal);
	LOG_CLOSE();	
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
      }
    else if (strcmp(signal, MCE_DEVICE_MODE_SIG) == 0 &&
	     type != DBUS_TYPE_STRING)
      {
	DLOG_OPEN("libosso");
	DLOG_WARN_F("warning: invalid arguments in HW signal '%s'", signal);
	LOG_CLOSE();
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
      }
    if (strcmp(signal, MCE_SHUTDOWN_SIG) == 0)
      {
	state = dbus_message_iter_get_boolean(&iter);
      }
    else if (strcmp(signal, MCE_DEVICE_MODE_SIG) == 0)
      {
	mode_str = dbus_message_iter_get_string(&iter);
      }
    dprint("Signal is '%s' in state '%s'",signal,state?"TRUE":"FALSE");

     /* Unsetting the non-stateful signals before possible call should
	keep the backward compability without side effects */
     
     osso->hw_state->memory_low_ind =  FALSE;
     osso->hw_state->save_unsaved_data_ind = FALSE;
     osso->hw_state->system_inactivity_ind = FALSE;

     _set_state(signal, shutdown_ind, state);

     /* To keep things single for identification of signals that do not
	have a state on the callback function, just set their state to TRUE.
     */
     _set_state(signal, memory_low_ind, TRUE);
     _set_state(signal, save_unsaved_data_ind, TRUE);
     _set_state(signal, system_inactivity_ind, TRUE);

     _set_state(signal, sig_device_mode_ind, mode_str);

     _call_state_cb(shutdown_ind);
     _call_state_cb(memory_low_ind);
     _call_state_cb(save_unsaved_data_ind);
     _call_state_cb(system_inactivity_ind);
     _call_state_cb(sig_device_mode_ind);

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}
