/**
 * @file libosso.h
 * This file is the API for LibOSSO.
 *
 * Copyright (C) 2005  Nokia
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#ifndef LIBOSSO_H_
# define LIBOSSO_H_

# include <glib.h>
# include <time.h>
# include <syslog.h>

# include <dbus/dbus-protocol.h>

G_BEGIN_DECLS


/**
 * This enum represents the returnvalues that most LibOSSO
 * functions use.*/
typedef enum {
    OSSO_OK = 0, /**<The function executed normally. */
	OSSO_ERROR = -1, /**<Some kind of an error occured. */
	OSSO_INVALID = -2, /**<At least one parameter is invalid. */
        OSSO_RPC_ERROR = -4, /**< Osso RPC method returned an error. */
	OSSO_ERROR_NAME = -3,
	OSSO_ERROR_NO_STATE = -5, /**< No state file found to read. */
	OSSO_ERROR_STATE_SIZE = -6 /**< The size of the given structure is
				    * different from the saved size */
}
osso_return_t;



/**
 * \defgroup Init Initialization
 *
 * @{
 *
 * This is the return type for the initialize function.
 */
typedef struct osso_af_context_t osso_context_t;

/**
 * This function initializes the library and connects to the D-BUS session
 * and system busses. and integrates with the GLib main loop, and
 * initializes the library for use. #osso_initialize should only be called
 * once for each application.
 * @param application The name that this application is known as. This is
 * not the name that a user sees, but rather the name that other
 * applications use as an address to communicate with this program, AKA
 * the "D-BUS name".
 * @param version The version string of the application. It mus be simply
 * comparable with strcmp().
 * @param activation If this is TRUE, then the library assumes that the
 * application binary has been launched by the D-BUS daemon, and thus will
 * connect to the D-BUS activation bus (the D-BUS bus where the D-BUS message
 * that resulted in program activation originated from).
 * @param context The GLib main-loop context to connect to, use NULL for
 * the default context.
 * @return A context to use in later calls to this library. NULL is
 * returned if there is an error.
 */
osso_context_t * osso_initialize(const gchar *application,
				 const gchar* version,
				 gboolean activation,
				 GMainContext *context);
/**
 * This function will close the message bus connection and free all memory
 * allocated by the library. This function renders the library in an
 * unusable state, #osso_initialize has to be called again to reinitialize
 * the library.
 * @param osso The library reference as returned by #osso_initialize.
 */
void osso_deinitialize(osso_context_t *osso);

/* @} */
/************************************************************************/
/**
 * \defgroup RPC Generic Remote Process Communication (RPC)
 *
 * These functions provide a wrapper for DBUS messages.
 */
/* @{*/

/**
 * The argument element used in the Generic RPC functions is a GArray of 
 * these #osso_rpc_t structures.
 */
typedef struct {
    int type; /**<The type of argument */
    union {
	guint32 u; /**<Type is DBUS_TYPE_UINT32 */
	gint32 i; /**<Type is DBUS_TYPE_INT32 */
	gboolean b; /**<Type is DBUS_TYPE_BOOLEAN */
	gdouble d; /**<Type is DBUS_TYPE_DOUBLE */
	const gchar *s; /**<Type is DBUS_TYPE_STRING */
    } value; /**<The way the param is interpeted depends on the #type field.*/
}
osso_rpc_t;

/**
 * This is the type for the generic rpc handler. This function is called
 * every time a method is requested on thr registered interface.
 * @param interface The interface that the method is called on.
 * @param method The method that is called.
 * @param arguments A GArray of #osso_rpc_t structures.
 * @param retval The return value of the method. This should be set to
 * DBUS_TYPE_INVALID for no reply.
 * @param data An application specific pointer.
 * @return #OSSO_OK if the function executed successfully. retval is set to
 * DBUS_TYPE_INVALID for no reply. #OSSO_ERROR, if an error occured, a
 * dbus_error will be returned, and the retval should be of type
 * DBUS_TYPE_STRING with an error message string as value.
 */
typedef gint (osso_rpc_cb_f)(const gchar *interface, const gchar *method,
			     GArray *arguments, gpointer data,
			     osso_rpc_t *retval);

/**
 * This is the type for the asyncronous rpc return handler. This function
 * is called when the asynchronous function returns.
 * @param interface The interface of the called method.
 * @param method The method that was called.
 * @param retval.The value that was returned.
 * @param data An application specific pointer.
 */
typedef void (osso_rpc_async_f)(const gchar *interface,
				const gchar *method,
				osso_rpc_t *retval, gpointer data);

/**
 * This function sends a message to an other application, using D-BUS
 * with syncronous return value if needed. This function will try to
 * activate the application that has registered the service with the D-BUS
 * daemon, by using the D-BUS auto-activation feature.
 * 
 * The variable arguments work in a type - value pairs. The type defines the
 * value. If the type is G_TYPE_STRING, then the value is a pointer to a
 * string. The list must end in a DBUS_TYPE_INVALID. The supported types
 * are:
 *  - DBUS_TYPE_BOOLEAN
 *    - The value is a gboolean.
 *  - DBUS_TYPE_INT, DBUS_TYPE_UINT
 *    - The value is an int or an unsigned int.
 *  - DBUS_TYPE_DOUBLE
 *    - The value is a float.
 *  - DBUS_TYPE_STRING
 *    - The value is a pointer to a string.
 *    - If the pointer is NULL it is translated to a DBUS_TYPE_NIL.
 *  - DBUS_TYPE_NIL
 *    - No value, this represents a NULL type.
 * @param osso The library context as returned by #osso_initialize.
 * @param service The service name of the target.
 * e.g. com.nokia.application
 * @param object_path The object path of the target object.
 * @param interface The interface that the method belongs to.
 * @param method The method to call
 * @param retval A pointer to a structure where the return value can be
 * stored. If a reply is not interesting, or not desired, this can be set
 * to NULL. When this is non null the call blocks for a while to wait for
 * the return value. If the expected reply doesn't come within the
 * timeout value set with the #osso_rpc_set_timeout function, #OSSO_ERROR
 * will be returned and the retval variable will be set to #OSSO_RPC_ERROR.
 * If NULL is given the function returns when the message is sent.
 * @param argument_type The type of the first argument.
 * @param ... The first argument value, and then a list of other arguments.
 * This list must end in a DBUS_TYPE_INVALID type.
 * @return #OSSO_OK if the message was sent. #OSSO_INVALID if a parameter
 * is invalid. If the remote method returns an error, or does not return
 * anything, then #OSSO_ERROR is returned, and retval is set to an error
 * message. #OSSO_ERROR is also returned if any other kind of error
 * occures, like IO error.
 *
 */
osso_return_t osso_rpc_run (osso_context_t * osso, const gchar * service,
                            const gchar * object_path,
                            const gchar * interface, const gchar * method,
                            osso_rpc_t * retval, int argument_type, ...);

/**
 * This function is a wrapper for #osso_rpc_run. It calls a remote method
 * on the destination. The service is defaulted to
 * 'com.nokia.{application}', the object path is set to
 * '/com/nokia/{application}' and the interface is defaulted to
 * 'com.nokia.{application}'. This function will try to activate the
 * application, by using the D-BUS auto-activation feature.
 * 
 * @param osso The library context as returned by #osso_initialize.
 * @param application The name of the application to call the method on
 * @param method The method to call
 * @param retval A pointer to a structure where the return value can be
 * stored. If a reply is not interesting, or not desired, this can be set
 * to NULL. When this is non null the call blocks for a while to wait for
 * the return value. If the expected reply doesn't come within the
 * timeout value set with the #osso_rpc_set_timeout function, #OSSO_ERROR
 * will be returned and the retval variable will be set to #OSSO_RPC_ERROR.
 * @param argument_type The type of the first argument.
 * @param ... The first argument value, and then a list of other arguments.
 *        See #osso_rpc_run for more information.
 * @return #OSSO_OK if the message was sent. #OSSO_INVALID if a parameter
 * is invalid. If the remote method returns an error, or does not return
 * anything, then #OSSO_ERROR is returned, and retval is set to an error
 * message. #OSSO_ERROR is also returned if any other kind of error
 * occures, like IO error.
 */
osso_return_t osso_rpc_run_with_defaults (osso_context_t * osso,
                                          const gchar * application,
                                          const gchar * method,
                                          osso_rpc_t * retval,
                                          int argument_type, ...);

/**
 * This function sends a message to an other application, using D-BUS
 * with asyncronous return value if needed. This function will try to
 * activate the application that has registered the service with the D-BUS
 * daemon, by using the D-BUS auto-activation feature.
 *
 * @param osso The library context as returned by #osso_initialize.
 * @param service The service name of the target.
 * e.g. com.nokia.application
 * @param object_path The object path of the target object.
 * @param interface The interface that the method belongs to.
 * @param method The method to call
 * @param async_cb A pointer to a function to be called when the method
 * returns. If the call timeouts async_cb will be called with an error,
 * generated by the D-BUS library. If this is NULL this function behaves
 * just like #osso_rpc_run, with the argument retval set to NULL.
 * with the argument retval set to NULL.
 * @param data An application specific pointer that is given to the cb
 * function.
 * @param argument_type The type of the first argument.
 * @param ... The first argument value, and then a list of other arguments.
 *        See #osso_rpc_run for more information.
 * @return #OSSO_OK if the message was sent. #OSSO_INVALID if a parameter
 * is invalid, and #OSSO_ERROR if an error occurs (like failure to send
 * the message)
 */
osso_return_t osso_rpc_async_run (osso_context_t * osso,
                                  const gchar * service,
                                  const gchar * object_path,
                                  const gchar * interface,
                                  const gchar * method,
                                  osso_rpc_async_f * async_cb, gpointer data,
                                  int argument_type, ...);

/**
 * This function sends a message to an other application, using D-BUS
 * with asyncronous return value if needed. The service is defaulted to
 * 'com.nokia.{application}', the object path is set to
 * '/com/nokia/{application}' and the interface is defaulted to
 * 'com.nokia.{application}'. This function will try to activate the
 * application, by using the D-BUS auto-activation feature.
 *
 * @param osso The library context as returned by #osso_initialize.
 * @param application The name of the application to call the method on
 * @param method The method to call
 * @param async_cb A pointer to a function to be called when the method
 * returns. If the call timeouts async_cb will be called with an error,
 * generated by the D-BUS library. If this is NULL this function behaves
 * just like #osso_rpc_run_with_defaults, with the argument retval set to
 * NULL.
 * @param data An application specific pointer that is given to the cb
 * function.
 * @param argument_type The type of the first argument.
 * @param ... The first argument value, and then a list of other arguments.
 *        See #osso_rpc_run for more information.
 * @return #OSSO_OK if the message was sent. #OSSO_INVALID if a parameter
 * is invalid, and #OSSO_ERROR if an error occurs (like failure to send
 * the message)
 */
osso_return_t osso_rpc_async_run_with_defaults (osso_context_t * osso,
                                                const gchar * application,
                                                const gchar * method,
                                                osso_rpc_async_f * async_cb,
                                                gpointer data,
                                                int argument_type, ...);

/**
 * This function registers a function to handle specific method calls with
 * the given interface reachable by the give object path and service name.
 * @param osso The library context as returned by #osso_initialize.
 * @param service The service name to set up.
 * e.g. com.nokia.application
 * @param object_path The object path that this object has.
 * @param interface The interface that this object implements.
 * @param cb The function that handles the method calls.
 * @param data An application specific pointer that is given to the cb
 * function.
 * @return #OSSO_OK if the service could be set up. #OSSO_INVALID if a parameter is
 * invalid, and #OSSO_ERROR if an error occurs (like IO)
 */
osso_return_t osso_rpc_set_cb_f (osso_context_t * osso, const gchar * service,
                                 const gchar * object_path,
                                 const gchar * interface, osso_rpc_cb_f * cb,
                                 gpointer data);

/**
 * This function registers a function to handle specific method calls to the
 * default service of 'com.nokia.{application}'.
 * @param osso The library context as returned by #osso_initialize.
 * @param cb The function that handles the method calls.
 * @param data An application specific pointer that is given to the cb
 * function.
 * @return #OSSO_OK if the new service was set up. #OSSO_INVALID if a
 * parameter is invalid, and #OSSO_ERROR if an error occurs (like IO)
 */
osso_return_t osso_rpc_set_default_cb_f (osso_context_t * osso,
                                         osso_rpc_cb_f * cb, gpointer data);

/**
 * This function unregisters a function to handle specific method calls.
 * @param osso The library context as returned by #osso_initialize.
 * @param service The service name to unregister.
 * e.g. com.nokia.application
 * @param object_path The object path that this object has.
 * @param interface The interface that this object implements.
 * @param cb The function that was registered.
 * @param data The same pointer that was used with the
 * #osso_rpc_set_default_cb_f call
 * @return #OSSO_OK if the new service was set up. #OSSO_INVALID if a
 * parameter is invalid, and #OSSO_ERROR if an error occurs (like IO)
 */
osso_return_t osso_rpc_unset_cb_f (osso_context_t * osso,
                                   const gchar * service,
                                   const gchar * object_path,
                                   const gchar * interface,
                                   osso_rpc_cb_f * cb, gpointer data);

/**
 * This function unregisters a function to handle specific method calls.
  * @param osso The library context as returned by #osso_initialize.
 * @param cb The function that was registered.
 * @param data The same pointer that was used with the
 * #osso_rpc_set_default_cb_f call
 * @return #OSSO_OK if the new service was set up. #OSSO_INVALID if a
 * parameter is invalid, and #OSSO_ERROR if an error occurs (like IO)
 */
osso_return_t osso_rpc_unset_default_cb_f (osso_context_t * osso,
                                           osso_rpc_cb_f * cb, gpointer data);

/**
 * Sets the timeout value used by rpc functions.
 * @param osso The library context as returned by #osso_initialize.
 * @param timeout The new timeout value.
 * @return #OSSO_OK on success. #OSSO_INVALID if osso is invalid
 */
osso_return_t osso_rpc_get_timeout (osso_context_t * osso, gint * timeout);

/**
 * Returns the current timeout value.
 * @param osso The library context as returned by #osso_initialize.
 * @param timeout A pointer where to return the timeout value.
 * @return #OSSO_OK on success. #OSSO_INVALID if osso is invalid
 */
osso_return_t osso_rpc_set_timeout(osso_context_t * osso, gint timeout);

/* @}*/
/**********************************************************************/
/**
 * \defgroup Apps Applications
 *
 * @{
 */

/**
 * This typedef is the application function that is registered via a call
 * to #osso_application_set_exit_cb.
 * @param die_now This boolean is set to TRUE if the application should
 * save its state now and exit. if it is set to FALSE.
 * @param data The data given to #osso_application_set_exit_cb.
 */
typedef void (osso_application_exit_cb)(gboolean die_now, gpointer data);

/**
 * This function will set */
osso_return_t osso_application_set_exit_cb(osso_context_t *osso,
					   osso_application_exit_cb *cb,
					   gpointer data);

/**
 * This function topps an application. If the application is not running
 * D-Bus will launch it via the Auto activation mechanism. By using this
 * function only one instance of the application will be running. If the
 * application is already running, this function will only "top" it.
 * 
 * @param osso The library context as returned by #osso_initialize.
 * @param application The name of the application to top.
 * @param arguments This string lists some extra parameters for the
 * application. It can be used to give the names of the files to open, etc.
 * If no arguments are to be passed, set this to NULL.
 * @return #OSSO_OK if the application launch request was successfully sent
 * to the dbus-daemon, #OSSO_ERROR if not.
 *
 * <h2>Example</h2>
 * This is an example on how to parse the argument from a top message. The
 * format is the same as a commandline with long options, i.e.
 * --param=value... So that the same arguments can be easily implemented
 * as commandline arguments.
 * @code
# include <unistd.h>
# define _GNU_SOURCE
# include <getopt.h>

void top_handler(const gchar *arguments, gpointer data);
{
    gchar **argv;
    gint argc;
    GError *err

    // this will split the arguments string into something similar to
    // what one would get in main.
    g_shell_parse_argv(arguments, &argc, &argv, &err);

    // This forces getopt to start from the first argument every time
    optind=0;
    optarg=argv;

    //call the generic commandline parser
    parse_commandline(argc, argv);

    // Free memory allocated previously.
    g_strfreev(argv);
}

void parse_commandline(int argc, char *argv[])
{
    static struct option long_options[] = {
	{"to", 1, 0, 't'}, // to requires an argument
	{"files", 1, 0, 'f'},
	{"url", 1, 0, 'u'},
	{0, 0, 0, 0}
    };
    gint index = 0, r;

    while(1) {
	//This function will parse and return the next argument,
	//see getopt(3) for more info.

	r = getopt_long(argc, argv, "t:f:u:", long_options, &index);

	// No more arguments
	if(r == -1) break;

	switch(r) {
	  case 't':
	    // do something with the argument, optarg is a global, external
	    // variable that will point to the parameter value
	    handle_to(optarg);
	    break;
	    // And so on
	}
    }
}
 * @endcode
 */
osso_return_t osso_application_top(osso_context_t *osso,
				   const gchar *application,
				   const gchar *arguments);

/**
 * This is the type for the top callback function.
 * @deprecated { This feature will be removed from LibOSSO soon, use
 * osso_rpc functions instead}
 * 
 * @param arguments This string lists some extra parameters for the
 * application. It can be used to give the names of the files to open, etc.
 * @param data The data that was set with the #osso_application_set_top_cb
 * function.
 * 
 */
typedef void(osso_application_top_cb_f)(const gchar *arguments,
					gpointer data);

/**
 * This function registers a top-event handler.
 * @deprecated { This feature will be removed from LibOSSO soon, use
 * osso_rpc functions instead}
 * 
 * @param osso The library context as returned by #osso_initialize.
 * @param cb The callback function.
 * @param data An arbitrary pointer to some application specific data.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_application_set_top_cb(osso_context_t *osso,
					  osso_application_top_cb_f *cb,
					  gpointer data);

/**
 * This function unregisters a top-event handler.
 * @deprecated { This feature will be removed from LibOSSO soon, use
 * osso_rpc functions instead}
 * 
 * @param osso The library context as returned by #osso_initialize.
 * @param cb The callback function.
 * @param data The same pointer that was used with the call to
 * #osso_application_set_top_cb
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_application_unset_top_cb(osso_context_t *osso,
					    osso_application_top_cb_f *cb,
					    gpointer data);

/**********************************************************************/
/* @}*/
/**
 * \defgroup Autosave Autosaving
 *
 * How to use autosave:<br />
 * The application tells LibOSSO that its user data state has changed,
 * by calling #osso_application_userdata_changed.
 * LibOSSO will then call the user data state save callback (along with UI
 * state save) when:
 * -# a timer set in LibOSSO (when the application informed that it's state
 *    has changed) expires (this is currently hardcoded to 2 minutes). 
 *    This timer is defined in
 *    <libosso source root>/src/osso-application-autosave.c
 * -# application goes to background (is untopped)
 *      - Since LibOSSO cannot detect this the application should use the
 *        #osso_application_autosave_force function
 * -# LibOSSO gets message from the system that state should be saved
 *    (e.g. at shutdown) [NOT IMPLEMENTED]
 * .
 * After the data/UI state save callbacks are called, the timer is removed
 * and the application needs to tell again that it has a "dirty" user data
 * state.
 * 
 * It is up to the application to optimize whether it calls the "dirty
 * state" function each time each user data changes or only the first time
 * after the state has been loaded or saved.
 * 
 * @todo Add code that will react to the shutdown system signal, to call
 * the autosave callback functions.
 */

/*@{*/

/**
 * This is the type for the autosave callback function.
 *
 * @param data The data that was set with the
 * #osso_application_set_autosave_cb function.
 */
typedef void(osso_application_autosave_cb_f)(gpointer data);

/**
 * This function registers an autosave user data event handler.
 *
 * @param osso The library context as returned by #osso_initialize.
 * @param cb The callback function.
 * @param data An arbitrary pointer to some application specific data.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_application_set_autosave_cb(osso_context_t *osso,
					       osso_application_autosave_cb_f *cb,
					       gpointer data);

/**
 * This function unregisters an autosave user data event handler.
 *
 * @param osso The library context as returned by #osso_initialize.
 * @param cb The callback function.
 * @param data An arbitrary pointer to some application specific data.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_application_unset_autosave_cb(osso_context_t *osso,
						 osso_application_autosave_cb_f *cb,
						 gpointer data);

/**
 * This function is called by the application when the user data
 * has been changed, so that autosaving can take place. The
 * dirty-state will be cleared every time the autosave callback is
 * called.
 *
 * @param osso The library context as returned by #osso_initialize.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_application_userdata_changed(osso_context_t *osso);

/**
 * This function forces a call to the save function, and resets
 * the timeout.
 *
 * @param osso The library context as returned by #osso_initialize.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_application_autosave_force(osso_context_t *osso);

/*@}*/

/**********************************************************************/
/**
 * \defgroup Statusbar Statusbar
 */
/* @{*/

/**
 * This function will send a statusbar event notification over the D-Bus.
 * If the applet is not loaded, the statusbar application will load it.
 * @param osso The library context as returned by #osso_initialize.
 * @param name the name of the applet that should receive the event.
 * @param argument1 This is the first argument of the event. It's
 * interpretation depends on event_type.
 * @param argument2 This is the second argument of the event. It's
 * interpretation depends on event_type.
 * @param argument3 This is the third argument of the event. It's
 * interpretation depends on event_type. NULL can be used here.
 * @param retval This parameter will be used to store possible return
 * data of the call. If NULL, no data will be expected.
 * be expected (and return value is unreliable)
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_statusbar_send_event(osso_context_t *osso,
					const gchar *name,
					gint argument1, gint argument2,
					const gchar *argument3,
					osso_rpc_t *retval);

/* @}*/
/**
 * \defgroup Time Time
 */
/* @{*/

/**
 * This is the type for the time change notification callback function.
 * @param data An application specific data pointer.
 */
typedef void(osso_time_cb_f)(gpointer data);

/**
 * This function registers a callback that is called whenever the time is
 * changed with this library.
 * @param osso The library context as returned by #osso_initialize.
 * @param cb The function that is called when system time is changed.
 * @param data An application specific data pointer.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_time_set_notification_cb(osso_context_t *osso,
					    osso_time_cb_f *cb,
					    gpointer data);
/**
 * This function sets the system and hardware time, and broadcasts the time
 * change notification event over the D-Bus.
 * @param osso The library context as returned by #osso_initialize.
 * @param new_time The new time in epoch format.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if new_time or osso context is invalid.
 */
osso_return_t osso_time_set(osso_context_t *osso, time_t new_time);

/* @}*/
/**
 * \defgroup Sysnotes System notification
 */
/* @{*/

/**
 * This is the type of system note, in other words it specifies what kind
 * of an icon to use on the note.
 */
typedef enum {
    OSSO_GN_WARNING = 0, /**<The message is a warning.*/
	OSSO_GN_ERROR,	 /**<The message is an error.*/
	OSSO_GN_NOTICE	 /**<The message is a notice.*/
}
osso_system_note_type_t;

/**
 * This function will request that a system note (a window that is modal
 * to the whole system.) is shown. This function enables non GUI programs
 * to show some information for the user to notice. Applications
 * that do have a GUI should not use this function but the hildon_note
 * widget directly.
 * @param osso The library context as returned by #osso_initialize.
 * @param message The contents of the message window.
 * @param type The type of system note (warning, notice, error).
 * @param retval This parameter will be used to store possible return
 * data of from the statusbar. If NULL, no data will be expected.
 * @return #OSSO_OK if no errors occur, and #OSSO_INVALID if some parameters
 * are invalid, and #OSSO_ERROR if some other error occurs.
 */
osso_return_t osso_system_note_dialog(osso_context_t *osso,
				      const gchar *message,
				      osso_system_note_type_t type,
				      osso_rpc_t *retval);

/**
 * This function will request that the statusbar shows an infoprint.
 * This allows non-GUI applications to display some information to a user.
 * Applications that do have a GUI should not use this function, but the
 * gtk_infoprint widget directly.
 * @param osso The library context as returned by #osso_initialize.
 * @param text The text to display.
 * data of the call. If NULL, no data will be expected.
 * @param retval This parameter will be used to store possible return
 * data of from the statusbar. If NULL, no data will be expected.
 * @return #OSSO_OK if no errors occur, and #OSSO_INVALID if some parameters
 * are invalid, and #OSSO_ERROR if some other error occurs.
 */
osso_return_t osso_system_note_infoprint(osso_context_t *osso,
					 const gchar *text,
					 osso_rpc_t *retval);

/* @}*/
/**********************************************************************/
/**
 * \defgroup Statesave Statesaving
 * The statesaving functionality is provided in order to allow
 * applications a convenient way of storing/retrieving short-term
 * state information, such as GUI state details (e.g. the view to the
 * last open document). The information saved by using it does not
 * survive over device powerdown; other means must be used when more
 * persistent storage is required. The application should load the
 * state information during startup in order to present user with the
 * previous state from the start. No automatic saving is done; it is
 * left as responsibility of the application to decide when it is
 * approriate to (re)save the state information. (perhaps by using the
 * Autosave functionality?)
 *
 * The new osso_state_read() and osso_state_write() functions take care of
 * state data reading from and writing to to a file. The only limitation is
 * that pointers can not be stored, this includes strings, so these strings
 * should be stored as character arrays. The easiest way to do this is to
 * define a structure e.g.
 * @code
struct state_data{
    gchar string[STRING_LENGTH];
    guint32 i;
    gchar filename[PATH_MAX];
};
 * @endcode
 * 
 * The old deprecated way has a preferred format of saved information
 * is "key=value".
 *
 * Multiple statefiles for same application are not supported
 * (this also includes different versions of the same application).
 *
 */
/* @{*/

/**
 * This structure represensts a state.
 */
typedef struct {
    guint32 state_size; /**< The size of state_data */
    gpointer state_data; /**< A pointer to state data */
}osso_state_t;

/**
 * This function will write the supplied state to a file. Any existing
 * statefile will be over-written.
 * @param osso The library context as returned by #osso_initialize.
 * @param state The state to save.
 * @return 
 *  - OSSO_OK if the opertation succeeds
 *  - OSSO_ERROR if the state could not be saved.
 *  - OSSO_INVALID if any argument is invalid.
 */
osso_return_t osso_state_write(osso_context_t *osso, osso_state_t *state);

/**
 * This function reads a state.
 * @param osso The library context as returned by #osso_initialize.
 * @param state A pointer to an allocated #osso_state_t structure. The data
 * should point to a memoryblock that is atleast as large as state_size.
 * @return OSSO_OK if the state reading was successfull. OSSO_ERROR if the
 * operation failed for some reason. OSSO_STATE_ERROR
 * @code
#define STRING_LEN 77

int func(void) {
    osso_state_t state;
    osso_return_t ret;
    struct state_data{
	int i;
	char s[STRING_LEN];
	int a;
	long double f;
    }statedata;
    
    memset(&statedata, 0, sizeof(struct state_data));
    state.state_data = &statedata;
    state.state_size = sizeof(struct state_data);
    
    ret = osso_state_read(osso, state);
    // ...
    
    // LibOSSO will fill the state_data structure
    return 1;
}
 * @endcode
 */
osso_return_t osso_state_read(osso_context_t *osso, osso_state_t *state);

/**
 * This function opens a statefile for writing. If statefile already
 * exists, it will be truncated silently without any warnings.
 * @deprecated use the more robust osso_state_write() function instead.
 * No new code should use this function.
 * @param osso The library context as returned by #osso_initialize.
 * @return the file descriptor of the opened file. On an error -1 is
 * returned.
 */
int osso_state_open_write(osso_context_t *osso);

/**
 * This function opens a statefile for reading.
 * @deprecated use the more robust osso_state_read() function instead.
 * No new code should use this function.
 * @param osso The library context as returned by #osso_initialize.
 * @return the file descriptor of the opened file. On an error -1 is
 * returned.
 */
int osso_state_open_read(osso_context_t *osso);

/**
 * This function closes the statefile.
 * @deprecated use the more robust osso_state_read() and 
 * osso_state_write() functions instead. No new code should use this
 * function.
 * @param osso The library context as returned by #osso_initialize.
 * @param fd The filedescriptor to close.
 */
void osso_state_close(osso_context_t *osso, gint fd);

/* @}*/
/**********************************************************************/
/**
 * \defgroup Ctrlpanel Control panel
 */
/* @{*/
/**
 * Calls the execute() function of the plugin. the
 * Plugins are loaded using dlopen(3) and after the execute func
 * returns, unloaded with dlclose(3).
 * @param osso The library context as returned by #osso_initialize.
 * @param filename The .so (Shared-object) file of the plugin. It should
 * include the .so-prefix, but not a path.
 * @param data The GTK toplevel widget. It is needed so that the widgets
 * created by the plugin can be made a child of the main application that
 * utilizes the plugin. Type is a gpointer so that the plugin does not need
 * to depend on GTK (in which case it should ignore the parameter).
 * @param user_activated If the plugin was activated by a user (as opposed to
 * activated by restoring software state), set to TRUE, else to FALSE.
 * @return the return value of the plugin on success, or #OSSO_ERROR on
 * error.
 */
osso_return_t osso_cp_plugin_execute(osso_context_t *osso,
				     const gchar *filename,
				     gpointer data, gboolean user_activated);

/**
 * This function is used to notify the plugin that the applicaiton
 *     
 * @param osso The library context as returned by #osso_initialize.
 * @param filename Same as filename parameter of #osso_cp_plugin_execute
 * @param data A pointer that will be passed to the rave_state function of
 * the plugin. 
 * @return OSSO_OK if all goes well. OSSO_INVALID if a parameter is
 * invalid or OSSO_ERROR if some error occures
 */
osso_return_t osso_cp_plugin_save_state(osso_context_t *osso,
					const gchar *filename,
					gpointer data);
/* @}*/
/**********************************************************************/
/**
 * \defgroup Devstate Device state
 * Functions in this group offer a convenient way to receive device status
 * information.
 *
 */


/*@{*/
/**
 * This structure represents the device state.
 */
typedef struct {
    gboolean shutdown_ind; /**<The device is about to be shut down */
    gboolean save_unsaved_data_ind;
    gboolean memory_low_ind;
    gboolean system_inactivity_ind;
    gchar *sig_device_mode_ind; /**<The mode of the device, flight or normal */
}osso_hw_state_t;


/**
 * Request to turn on the display as if the user had pressed the keys
 * or touchscreen. This can be used after completing a long operation
 * such as downloading a webpage or fetching emails.
 * 
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occured.
 */ 

osso_return_t osso_display_state_on(osso_context_t *osso);

/**
 * Request not to blank display, must be repeated in 60 seconds to
 * renew request.. Used for example by the video player during video
 * playback.  Also prevents suspending.
 * 
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occured.
 */

osso_return_t osso_display_blanking_pause(osso_context_t *osso);

/**
 * This is the type for the device event handler.
 * 
 * @param state ste current state of the device.
 * @param data The data that was set with the
 * #osso_hw_set_event_cb function.
 */
typedef void (osso_hw_cb_f)(osso_hw_state_t *state, gpointer data);

/**
 * This function sets an event handler callback that is called whenever
 *  the state of the device changes. The first call to this function will
 * also check the state files for the state of the device, and
 * if the state is set, the corresponding callback function will be called
 * immediately.
 * 
 * @param osso The library context as returned by #osso_initialize.
 * @param state The states we are interested in. NULL can be passed here
 * to indicate that all signals are of interest.
 * @param cb The callback function.
 * @param data Program specific data. 
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_hw_set_event_cb(osso_context_t *osso,
				   osso_hw_state_t *state,
				   osso_hw_cb_f *cb, gpointer data);

/**
 * This function unsets an event handler callback.
 * 
 * @param osso The library context as returned by #osso_initialize.
 * @param state The states we are interested in. NULL can be passed here
 * to indicate that all signals are of interest.
 * @return #OSSO_OK if all goes well, #OSSO_ERROR if an error occurred or
 * #OSSO_INVALID if some parameter is invalid.
 */
osso_return_t osso_hw_unset_event_cb(osso_context_t *osso,
				     osso_hw_state_t *state);

/*@}*/
/**********************************************************************/
/**
 * \defgroup MIME MIME types
 */
/*@{*/

/**
 * This is the typedef for the function that is called whevever the mimeopen
 * D-BUS message is received.
 * @param data The data pointer that was set with #mimeopen_set_cb.
 * @param argc The number of files
 * @param argv An array of file URIs
 */
typedef void (osso_mime_cb_f)(gpointer data, int argc, gchar **argv);

/**
 * Register a callback function that gets called when a D-BUS message that
 * indicates that a certain file should be opened is received.
 * @param osso The library context as returned by the #osso_initialize
 * function.
 * @param cb The callback function to call.
 * @param data Some caller-specific data that will be passed to the
 * callback function.
 * @return OSSO_OK. OSSO_INVALID if some parameter is invalid.
 * OSSO_ERROR if there is an error.
 */
osso_return_t osso_mime_set_cb(osso_context_t *osso, osso_mime_cb_f *cb,
			       gpointer data);

/**
 * Unregister a callback function that gets called when a D-BUS message that
 * indicates that a certain file should be opened is received.
 * @param osso The library context as returned by the LibOSSO function
 * #osso_initialize.
 * @return OSSO_OK. OSSO_INVALID if some parameter is invalid.
 * OSSO_ERROR if there is an error.
 */
osso_return_t osso_mime_unset_cb(osso_context_t *osso);
/*@}*/

/************************************************************************/
/******* **** **** **** ******* **** **** **** ******* **** **** **** ***/
/******* *    *  * *    ******* *    *  * *    ******* *    *  * *    ***/
/******* ***  *  * ***  ******* ***  *  * ***  ******* ***  *  * ***  ***/
/******* *    *  * *    ******* *    *  * *    ******* *    *  * *    ***/
/******* **** **** *    ******* **** **** *    ******* **** **** *    ***/
/************************************************************************/

/**
 * \defgroup outside Not included
 *
 * @note
 * these functions will NOT be part of libosso, since they are used just by
 * email application. The functions here are presented as a proposal for
 * what the email app could use.
 *
 */
/* @{*/

/** This function adds a mail to be displayed in the
 *      mail popup of the tasknavigator.
 *      @param osso The osso library context returned by #osso_initialize
 *      @param id unique identifier for the mail
 *      @param subject the subject string of the email
 *      @param sender the email address of the sender of the mail
 *      @param attachment true iff the mail has an attatchment
 *      @param date the date the mail was received.
 *
 *
 *      @return #OSSO_OK on success, #OSSO_ERROR in case of an error,
 *              #OSSO_INVALID if a parameter is invalid.
 */
osso_return_t osso_tasknav_mail_add(osso_context_t *osso, guint id,
				    const gchar *subject, const gchar *sender,
				    gboolean attachment, const gchar *date);

/** This function removes an email message from the navigator.
 *
 *      @param osso The osso library context returned by #osso_initialize
 *      @param id unique identifier of the mail to be destroyed
 *
 *      @return #OSSO_OK on success, #OSSO_ERROR in case of an error,
 *              #OSSO_INVALID if a paramter is invalid.
 *
 *
 */
osso_return_t osso_tasknav_mail_del(osso_context_t *osso, guint id);

/** This function updates the outbox message count in the task
 *      navigator mail window.
 *
 *      @param osso The osso library context returned by #osso_initialize
 *      @param count New number of messages in outbox
 *
 *      @return #OSSO_OK on success, #OSSO_ERROR in case of an error,
 *              #OSSO_INVALID if a paramter is invalid.
 *
 *
 */
osso_return_t osso_tasknav_mail_set_outbox_count(osso_context_t *osso,
						 guint count);

/** Funcion to log to the syslog.
 *
 * @param level Log level. Set to LOG_CRIT , LOG_ERR, LOG_WARNING or LOG_INFO.
 *                      Usage in similar manner as for syscall(2)
 *
 * @param format Printf styled format string, see printf(2) for details
 *
 * @param ... arguments for the format string.
 *
 */
void osso_log(int level, const char *format, ...);

# define LOG_D  __FILE__,__LINE__,LOG_DEBUG

/** Funcion to log debug info to the syslog.
 *      Will be empty with DEBUG not defined hoping that
 *      compiler will eliminate the call.
 *
 *   The first four arguments can be replaced with 
 *      LOG_D
 *
 * @param file the file this came from
 *
 * @param line the linenumber this came from
 *
 * @param level Log level. Set to CRIT , ERR, WARN or INFO.
 *                      Usage in similar manner as for syscall(2)
 *
 * @param format Printf styled format string, see printf(2) for details
 *
 * @param ... arguments for the format string.
 *
 */
inline void d_log(const char *file,int line, int level,
		  const char *format, ...);

/** Returns dbus session bus connection used by osso
 *  @param osso osso context as returned by #osso_initialize
 *
 *  @return pointer to DBusConnection on a gpointer
 *
 * FIXME Have to returen gpointer now since due to errors
 *      in dbus.h it can not be included to every software.
 *
 */
gpointer osso_get_dbus_connection(osso_context_t *osso);


/** Returns dbus system bus connection used by osso
 *  @param osso osso context as returned by #osso_initialize
 *
 *  @return pointer to DBusConnection on a gpointer
 *
 * FIXME Have to return a gpointer now since due to errors
 *      in dbus.h it can not be included to every software.
 *
 */

gpointer osso_get_sys_dbus_connection(osso_context_t *osso);


/** Osso initialisation routine for applications to call.
 *  This ensures that the environment variables are transferred
 *  correctly before the gtk_init();
 *
 * 
 * @param application The name that this application is known as. This is  
 * not the name that a user sees, but rather the name that other
 * applications use as an address to communicate with this program.
 * @param version The version string of the application. It mus be simply
 * comparable with strcmp().
 * @param cb The callback function to used when application is topped.
 * @param callback_data An arbitrary pointer to some application specific data
 *              to pass to the top callback
*/
osso_context_t * osso_application_initialize(const gchar *application,
					     const gchar *version,
					     osso_application_top_cb_f *cb,
					     gpointer callback_data);


G_END_DECLS

#endif /* LIBOSSO_H_*/
