/*
 * This file is part of libosso-help-sdk
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Karoliina Salminen <karoliina.t.salminen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
  @file osso-helplib.h
  
  Integration of applications with the OSSO Help system is done via 
  'libossohelp' shared library, and the interface in this header file.
  <p>
  
*/
#ifndef OSSO_HELPLIB_H
#define OSSO_HELPLIB_H

#include <libosso.h>

#ifdef __cplusplus
  extern "C" {
#endif

/*--- Public interface (for any applications) ---*/

/**
  Show Help Topic or Folder.  This same function can be used either for
  Help Dialogs (displaying a single Help Topic _without links_ in the
  calling application's dialog box), or for launching Help Application
  itself, providing Folder browsing, Search and History functions, too.
  <p>  
  Normally, applications would only use this to launch help topics, but
  it can be used for opening a folder view as well.
  
  @param osso		OSSO handle of the calling application, required for RPC
					calls to HelpApp or Browser.

  @param help_id	Identification string (topic id of the form "xxx_fname_zzz"
					or "//fname/a/[.../]NNN").

  @param flags		0 for default (open Help as a separate application)
					+1 (#OSSOHELP_SHOW_DIALOG) for opening as system dialog,
						without navigation possibilities (just the topic)
					+2 (#OSSOHELP_SHOW_JUSTASK) to query, whether a certain 
						topic, help file or folder is available on the system.
					
  @return	#OSSO_OK		All well, help found (and being displayed)
			#OSSO_ERROR		No help for such topic ID
			#OSSO_RPC_ERROR	Unable to contact HelpApp or Browser (RPC failed)
			#OSSO_INVALID	Param not formally right (NULL, or bad ID format)
*/
enum {
	OSSO_HELP_SHOW_DIALOG= 0x01,
	OSSO_HELP_SHOW_JUSTASK= 0x02,
	};
	
osso_return_t ossohelp_show( osso_context_t *osso,
							 const gchar *help_id,
							 guint flags );

// Helper macros (deprecated)
#define ossohelp_dialog( osso, help_id ) \
		ossohelp_show( (osso), (help_id), OSSO_HELP_SHOW_DIALOG )
#define ossohelp_exists( help_id ) \
		ossohelp_show( NULL, (help_id), OSSO_HELP_SHOW_JUSTASK )

#ifdef __cplusplus
  }
#endif

#endif	// OSSO_HELPLIB_H

