/*
 * This file is part of lessertunjo
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Kuisma Salonen <kuisma.salonen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
 * @file sapp_init.c
 * shadow application library, game side library,
 * initialization part
 * <p>
 *
 * @author Kuisma Salonen <Kuisma.Salonen@nokia.com>
 */

/**********************************************************/

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/types.h>
#include <unistd.h>
#include <memory.h>

#include "shadowapp.h"

int sapp_error = SAPP_ERROR_NONE;

sapp_data *sapp_initialize(int argc, char *argv[])
{
  int rc, msg;
  sapp_data *retval;

  if(argc < 3) {
    sapp_error = SAPP_ERROR_INVALIDARGS;
    return NULL;
  }

  retval = (sapp_data *)malloc(sizeof(sapp_data));
  if(retval == NULL) {
    sapp_error = SAPP_ERROR_NOMEMORY;
    return NULL;
  }
  memset(retval, 0, sizeof(sapp_data));

  retval->command = atoi(argv[2]);

  retval->sockfd = socket(PF_UNIX, SOCK_STREAM, 0);
  if(retval->sockfd == -1) {
    sapp_error = SAPP_ERROR_SOCKET;
    free(retval);
    return NULL;
  }

  retval->configuration_socket = argv[3];

  retval->saddr = (struct sockaddr_un *)malloc(sizeof(struct sockaddr_un));
  if(retval->saddr == NULL) {
    sapp_error = SAPP_ERROR_NOMEMORY;
    close(retval->sockfd);
    free(retval);
    return NULL;
  }

  retval->saddr->sun_family = AF_UNIX;
  sprintf(retval->saddr->sun_path, "%s", argv[1]);

  if(connect(retval->sockfd, (struct sockaddr *)retval->saddr,
             (socklen_t)sizeof(struct sockaddr_un)) == -1) {
    sapp_error = SAPP_ERROR_CONNECTION;
    close(retval->sockfd);
    free(retval->saddr);
    free(retval);
    return NULL;
  }

  /* sends our pid to server to determine if our pid is correct */
  msg = getpid();
  rc = send(retval->sockfd, &msg, 4, 0);
  if(rc == -1) {
    sapp_error = SAPP_ERROR_CONNECTION;
    close(retval->sockfd);
    free(retval->saddr);
    free(retval);
    return NULL;
  }

  /* response, SAPP_ERROR_NONE if succeed */
  rc = recv(retval->sockfd, &msg, 4, 0);
  if(rc == 0) { /* connection lost */
    sapp_error = SAPP_ERROR_CONNECTION;
    close(retval->sockfd);
    free(retval->saddr);
    free(retval);
    return NULL;
  } else if(rc != 4) { /* not handled accurately yet */
    sapp_error = SAPP_ERROR_CONNANOTHER;
    close(retval->sockfd);
    free(retval->saddr);
    free(retval);
    return NULL;
  }

  if(msg != SAPP_ERROR_NONE) {
    sapp_error = msg;
    close(retval->sockfd);
    free(retval->saddr);
    free(retval);
    return NULL;
  }

  /* obsolete */
/*fcntl(retval->sockfd, F_SETFL, O_NONBLOCK);*/

  retval->deinit = sapp_deinitialize;
  retval->smain = sapp_mainloop;

  return retval;
}

int sapp_deinitialize(sapp_data *context)
{
  int sent = 0, rc;

  if(context == NULL) {
    return SAPP_ERROR_INVALIDPARAMS;
  }

  do {
    rc = send(context->sockfd, "bye\n" + sent, 4 - sent, 0);
    if(rc == -1) {
      close(context->sockfd);
      free(context->saddr);
      free(context);
      return SAPP_ERROR_BYESEND;
      break;
    } else if(rc == 0) {
      close(context->sockfd);
      free(context->saddr);
      free(context);
      return SAPP_ERROR_BYESEND;
      break;
    } else {
      sent += rc;
    }
  } while(sent < 4);

  close(context->sockfd);
  free(context->saddr);
  free(context);

  return SAPP_ERROR_NONE;
}

int sapp_quit(sapp_data *context)
{
  int rc;
  int error = SAPP_ERROR_NONE;

  if(context == NULL) {
    return SAPP_ERROR_INVALIDPARAMS;
  }

  /* when we block, we block until everything is sent */
  rc = send(context->sockfd, "quit\n", 5, 0);
  if(rc != 5) {
    error = SAPP_ERROR_BYESEND;
  }

  close(context->sockfd);
  free(context->saddr);
  free(context);

  return error;
}

sapp_boolean sapp_check(sapp_data *context)
{
  switch(context->command)
  {
  case 1:
    return SAPP_TRUE;
    break;
  case 2:
    context->msg_cb_flush(SAPP_ERROR_QUIT);
    context->msg_cb_quit(SAPP_ERROR_QUIT);
    sapp_quit(context);
    sapp_error = SAPP_ERROR_QUIT;
    return SAPP_FALSE;
    break;
  case 3:
    context->msg_cb_flush(SAPP_ERROR_NONE);
    return SAPP_TRUE;
    break;
  default:
    sapp_deinitialize(context);
    break;
  }

  return SAPP_FALSE;
}
