/* -*- mode: C; c-file-style: "stroustrup"; indent-tabs-mode: nil; -*- */


/*
 * This file is part of hildon-status-bar
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Karoliina Salminen <karoliina.t.salminen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/* Hildon includes */
#include "hildon-status-bar-lib.h"
#include "hildon-status-bar-item.h"
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-note.h>

#include <libosso.h>
/*#include <osso-log.h>*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/* System includes */
#include <string.h>

/* GTK includes */
#include <gtk/gtkwindow.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkfixed.h>

/* GDK includes */
#include <gdk/gdkwindow.h>
#include <gdk/gdkx.h>

/* GLib include */
#include <glib.h>

/* X include */
#include <X11/X.h>
#include <X11/Xatom.h>
#include <X11/Xlib.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define HILDON_STATUS_BAR_NAME             "statusbar"
#define HILDON_STATUS_BAR_VERSION          "1.0"

#define HILDON_STATUS_BAR_MAX_NO_OF_ITEMS  30  /* maximum number of items */
#define HILDON_STATUS_BAR_VISIBLE_ITEMS    7   /* maximum number of items to 
                                                * show */

/* hardcoded pixel positions for the icons */
/* CHECKME: Why isn't y=0 at top? -2 seems to be much better.. */
#define HILDON_STATUS_BAR_ITEM0_X          240
#define HILDON_STATUS_BAR_ITEM0_Y          0
#define HILDON_STATUS_BAR_ITEM1_X          200
#define HILDON_STATUS_BAR_ITEM1_Y          0
#define HILDON_STATUS_BAR_ITEM2_X          160
#define HILDON_STATUS_BAR_ITEM2_Y          0
#define HILDON_STATUS_BAR_ITEM3_X          120 
#define HILDON_STATUS_BAR_ITEM3_Y          0
#define HILDON_STATUS_BAR_ITEM4_X          80 
#define HILDON_STATUS_BAR_ITEM4_Y          0
#define HILDON_STATUS_BAR_ITEM5_X          40 
#define HILDON_STATUS_BAR_ITEM5_Y          0
#define HILDON_STATUS_BAR_ITEM6_X          0  
#define HILDON_STATUS_BAR_ITEM6_Y          0


/* hardcoded slot position for the four prespecified items, 0,1,2.. */
#define HILDON_STATUS_BAR_FIRST_DYN_SLOT  5   /* first slot for dynamic 
                                               * plugins */
#define HILDON_STATUS_BAR_DISPLAY_SLOT    4
#define HILDON_STATUS_BAR_SOUND_SLOT      3
#define HILDON_STATUS_BAR_INTERNET_SLOT   2
#define HILDON_STATUS_BAR_GATEWAY_SLOT    1
#define HILDON_STATUS_BAR_BATTERY_SLOT    0

typedef struct status_bar_st StatusBar;

struct status_bar_st
{
    GtkWidget *window;                                   /* GTK Window */
    GtkWidget *items[HILDON_STATUS_BAR_MAX_NO_OF_ITEMS]; /* Plugin items */
    GtkWidget *fixed;                                    /* Fixed container 
                                                          * for the items */

    /* fixed X pixel coordinates for the plugins */
    gint plugin_pos_x[HILDON_STATUS_BAR_MAX_NO_OF_ITEMS];

    /* fixed Y pixel coordinates for the plugins */
    gint plugin_pos_y[HILDON_STATUS_BAR_MAX_NO_OF_ITEMS];

};

static int init_dock( StatusBar *panel );
static void add_prespecified_items( StatusBar *panel );
static void show_infoprint( const gchar *msg );
static gint rpc_cb( const gchar *interface, 
                    const gchar *method,
                    GArray *arguments, 
                    gpointer data,
                    osso_rpc_t *retval );
static HildonStatusBarItem *add_item( StatusBar *panel, const gchar *plugin );
static HildonStatusBarItem *get_item( StatusBar *panel, const gchar *method );

static void reorder_items( StatusBar *panel );
static void free_slot( StatusBar *panel );
static void destroy_item( GtkObject *object,
                          gpointer user_data );

static int init_dock( StatusBar *panel )
{
    Atom atoms[3];
    gint i;
    Display   *dpy;
    Window    win;

    for( i = 0; i < HILDON_STATUS_BAR_MAX_NO_OF_ITEMS; ++i )
        panel->items[i] = NULL;

    panel->window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
    gtk_window_set_accept_focus(GTK_WINDOW(panel->window), FALSE);
    gtk_container_set_border_width( GTK_CONTAINER( panel->window ), 0 );
    gtk_widget_set_name( panel->window, "HildonStatusBar" );

    /* Set window type as DOCK */
    gtk_window_set_type_hint( GTK_WINDOW( panel->window ), 
                              GDK_WINDOW_TYPE_HINT_DOCK );

    /* GtkWindow must be realized before using XChangeProperty */
    gtk_widget_realize( GTK_WIDGET( panel->window ) );

    /* We need to set this here, because otherwise the GDK will replace 
     * our XProperty settings. The GDK needs to a patch for this, this 
     * has nothing to do with the vanilla GDK. */
    g_object_set_data( G_OBJECT( panel->window->window ),
                       "_NET_WM_STATE", (gpointer)PropModeAppend );

    dpy = GDK_DISPLAY();    
    win = GDK_WINDOW_XID( panel->window->window );

    /* Crete the atoms we need */
    atoms[0] = XInternAtom( dpy, "_NET_WM_STATE", False);
    atoms[1] = XInternAtom( dpy, "_MB_WM_STATE_DOCK_TITLEBAR", False);
    atoms[2] = XInternAtom( dpy, "_MB_DOCK_TITLEBAR_SHOW_ON_DESKTOP", False);

    /* Set proper properties for the matchbox to show dock as the dock we want*/
    XChangeProperty( dpy, win, 
                     atoms[0], XA_ATOM, 32, 
                     PropModeReplace, 
                     (unsigned char *) &atoms[1], 2 );

    panel->fixed = gtk_fixed_new();
    gtk_container_set_border_width( GTK_CONTAINER( panel->fixed ), 0 );

    gtk_container_add( GTK_CONTAINER( panel->window ), panel->fixed );

    panel->plugin_pos_x[0] = HILDON_STATUS_BAR_ITEM0_X;
    panel->plugin_pos_x[1] = HILDON_STATUS_BAR_ITEM1_X;
    panel->plugin_pos_x[2] = HILDON_STATUS_BAR_ITEM2_X;
    panel->plugin_pos_x[3] = HILDON_STATUS_BAR_ITEM3_X;
    panel->plugin_pos_x[4] = HILDON_STATUS_BAR_ITEM4_X;
    panel->plugin_pos_x[5] = HILDON_STATUS_BAR_ITEM5_X;
    panel->plugin_pos_x[6] = HILDON_STATUS_BAR_ITEM6_X;

    panel->plugin_pos_y[0] = HILDON_STATUS_BAR_ITEM0_Y;
    panel->plugin_pos_y[1] = HILDON_STATUS_BAR_ITEM1_Y;
    panel->plugin_pos_y[2] = HILDON_STATUS_BAR_ITEM2_Y;
    panel->plugin_pos_y[3] = HILDON_STATUS_BAR_ITEM3_Y;
    panel->plugin_pos_y[4] = HILDON_STATUS_BAR_ITEM4_Y;
    panel->plugin_pos_y[5] = HILDON_STATUS_BAR_ITEM5_Y;
    panel->plugin_pos_y[6] = HILDON_STATUS_BAR_ITEM6_Y;

    return 0;
}

static void add_prespecified_items( StatusBar *panel )
{
    add_item( panel, "sound" );
    add_item( panel, "internet" );
    add_item( panel, "gateway" );
    add_item( panel, "battery" );
    add_item( panel, "display" );
}

static void show_infoprint( const gchar *msg )
{
    g_return_if_fail( msg );

    gtk_infoprint( NULL, msg );
}

static gint rpc_cb( const gchar *interface, 
                    const gchar *method,
                    GArray *arguments, 
                    gpointer data, 
                    osso_rpc_t *retval )
{
    HildonStatusBarItem *item;
    StatusBar *panel;
    osso_rpc_t *val[3];
    gint i;
    
    /* These must be specified */
    if( !interface || !method || !arguments || !data )
    {
        retval->type = DBUS_TYPE_STRING;
        retval->value.s = g_strdup_printf( "NULL parameter" );
        g_warning( "%s\n", retval->value.s );
        return OSSO_ERROR;
    }

    panel = (StatusBar*)data;
 
    /* get the parameters */
    for( i = 0; i < arguments->len; ++i )
        val[i] = &g_array_index( arguments, osso_rpc_t, i );

    /* check the method */
    /* infoprint */
    if( !strcmp( "system_note_infoprint", method ) )
    {
        /* Check that the parameter is of right type */
        if( arguments->len < 1 ||
            val[0]->type != DBUS_TYPE_STRING )
        {
            retval->type = DBUS_TYPE_STRING;
            if( arguments->len < 1 )
                retval->value.s = 
                    g_strdup_printf( "Not enough arguments to infoprint" );
            else
                retval->value.s = 
                    g_strdup_printf("Wrong type of parameter to infoprint (%d)",
                                     val[0]->type );
            osso_log( LOG_ERR, retval->value.s );

            return OSSO_ERROR;       
        }
        show_infoprint( val[0]->value.s );
    }
    /* dialog */
    else if( !strcmp( "system_note_dialog", method ) )
    {
        /* check arguments */
        if( arguments->len < 2 || 
            val[0]->type != DBUS_TYPE_STRING || 
            val[1]->type != DBUS_TYPE_INT32 ) 
        {
            retval->type = DBUS_TYPE_STRING;
            if( arguments->len < 2 )
                retval->value.s = 
                    g_strdup_printf( "Not enough arguments to dialog" );
            else
                retval->value.s = 
                    g_strdup_printf("Wrong type of arguments to dialog (%d,%d)",
                                    val[0]->type, val[1]->type );
            osso_log( LOG_ERR, retval->value.s );
            return OSSO_ERROR;       
        }

        hildon_status_bar_lib_prepare_dialog( val[1]->value.i, NULL, 
                                              val[0]->value.s, 0, NULL, NULL );
    }
    /* plugin */
    else if( !strcmp( "event", method ) )
    {
        /* check arguments */
        if( arguments->len < 4 || 
            val[0]->type != DBUS_TYPE_STRING ||
            val[1]->type != DBUS_TYPE_INT32 ||
            val[2]->type != DBUS_TYPE_INT32 ||
            ( val[3]->type != DBUS_TYPE_STRING && 
              val[3]->type != DBUS_TYPE_NIL ) )
        {
            retval->type = DBUS_TYPE_STRING;
            if( arguments->len < 4 )
                retval->value.s = 
                    g_strdup_printf( "Not enough arguments to plugin" );
            else
                retval->value.s = 
                    g_strdup_printf( "Wrong type of arguments to a plugin "
                                     "(%d,%d,%d,%d)",
                                     val[0]->type, val[1]->type, val[2]->type, 
                                     val[3]->type);
            osso_log( LOG_ERR, retval->value.s );
            return OSSO_ERROR;
        }

        /* Check if the method item is not already loaded */
        if( !(item = get_item( panel, val[0]->value.s ) ) )
            item = add_item( panel, val[0]->value.s );
        
        /* plugin loading failed */
        if( !item )
        {
            retval->type = DBUS_TYPE_STRING;
            retval->value.s = g_strdup_printf( "Failed to load plugin %s", 
                                               val[0]->value.s );
            osso_log( LOG_ERR, retval->value.s );
            return OSSO_ERROR;
        }

        hildon_status_bar_item_update( item, 
                                       val[1]->value.i, 
                                       val[2]->value.i, 
                                       val[3]->value.s );
    }
    else
    {
        retval->type = DBUS_TYPE_STRING;
        retval->value.s = g_strdup_printf( "Unknown method '%s'", method );
        osso_log( LOG_ERR, retval->value.s );
        return OSSO_ERROR;
    }

    return OSSO_OK;
}

static HildonStatusBarItem *get_item( StatusBar *panel, const gchar *method )
{
    gint i;

    g_return_val_if_fail( panel, NULL );
    g_return_val_if_fail( method, NULL );

    /* Go through the array and check if the method match */
    for( i = 0; i < HILDON_STATUS_BAR_MAX_NO_OF_ITEMS; ++i )
    {
        if( panel->items[i] != NULL &&
            !strcmp( hildon_status_bar_item_get_name(
                         HILDON_STATUS_BAR_ITEM( panel->items[i] ) ), 
                         method ) )
            return HILDON_STATUS_BAR_ITEM( panel->items[i] );
    }
    
    /* No match found */
    return NULL;
}

static HildonStatusBarItem *add_item( StatusBar *panel, const gchar *plugin )
{
    HildonStatusBarItem *item;
    gint slot = -1;

    g_return_val_if_fail( panel, NULL );
    g_return_val_if_fail( plugin, NULL );

    /* check for prespecified items */
    if( !strcmp( "sound", plugin ) )
        slot = HILDON_STATUS_BAR_SOUND_SLOT;
    else if( !strcmp( "internet", plugin ) )
        slot = HILDON_STATUS_BAR_INTERNET_SLOT;
    else if( !strcmp( "gateway", plugin ) )
        slot = HILDON_STATUS_BAR_GATEWAY_SLOT;
    else if( !strcmp( "battery", plugin ) )
        slot = HILDON_STATUS_BAR_BATTERY_SLOT;
    else if( !strcmp( "display", plugin ) )
        slot = HILDON_STATUS_BAR_DISPLAY_SLOT;

    item = hildon_status_bar_item_new( plugin );

    if( !item )
    {
        osso_log( LOG_ERR, "Error creating HildonStatusBarItem: %s", plugin );
        return NULL;
    }

    /* ref the item, so it is not destroyed when removed temporarily 
     * from container, or destroyed too early when the plugin is removed 
     * permanently */
    g_object_ref( item );

    /* The item is prespecified */
    if( slot != -1 )
    {
        panel->items[slot] = GTK_WIDGET( item );

        /* Add it to fixed container */
        gtk_fixed_put( GTK_FIXED( panel->fixed ), 
                       panel->items[slot], 
                       panel->plugin_pos_x[slot],
                       panel->plugin_pos_y[slot] );
        
    }
    /* unknown plugin */
    else
    {

        /* free the first slot and put the new item there */
        free_slot( panel );
       
        panel->items[HILDON_STATUS_BAR_FIRST_DYN_SLOT] = GTK_WIDGET( item );
        gtk_fixed_put( GTK_FIXED( panel->fixed ), 
                       panel->items[HILDON_STATUS_BAR_FIRST_DYN_SLOT],
                       panel->plugin_pos_x[HILDON_STATUS_BAR_FIRST_DYN_SLOT],
                       panel->plugin_pos_y[HILDON_STATUS_BAR_FIRST_DYN_SLOT] );
        gtk_widget_show( GTK_WIDGET( item ) );

    }

    /* catch the event of plugin destroying itself*/
    g_signal_connect( G_OBJECT( item ), "destroy",
                      G_CALLBACK( destroy_item ), 
                      panel );

    return item;
}

static void free_slot( StatusBar *panel )
{
    gint i, j;

    g_return_if_fail( panel );

    /* remove last item if too many items */
    if( panel->items[HILDON_STATUS_BAR_MAX_NO_OF_ITEMS - 1] != NULL )
    {
        osso_log( LOG_ERR, "Too many plugins. Defined maximum is %d!", 
                  HILDON_STATUS_BAR_MAX_NO_OF_ITEMS );
        
        gtk_widget_destroy(panel->items[HILDON_STATUS_BAR_MAX_NO_OF_ITEMS - 1]);
        g_object_unref( panel->items[HILDON_STATUS_BAR_MAX_NO_OF_ITEMS - 1]);
        panel->items[HILDON_STATUS_BAR_MAX_NO_OF_ITEMS - 1] = NULL;
    }    
        

    /* find the last item */
    for( i = HILDON_STATUS_BAR_MAX_NO_OF_ITEMS - 2; 
         i >= HILDON_STATUS_BAR_FIRST_DYN_SLOT; --i )
    {
        if( panel->items[i] == NULL )
            continue;
        
        /* move every item by one */
        for( j = i; j >= HILDON_STATUS_BAR_FIRST_DYN_SLOT; --j )
        {
            panel->items[j + 1] = panel->items[j];
            panel->items[j] = NULL;

            /* remove item from container if moved out of the 7 slots */
            if( j + 1 == HILDON_STATUS_BAR_VISIBLE_ITEMS )
                gtk_container_remove( GTK_CONTAINER( panel->fixed ), 
                                      panel->items[j + 1] );

            /* update the fixed if item moved inside the 7 slots */
            if( j + 1 < HILDON_STATUS_BAR_VISIBLE_ITEMS  )
                gtk_fixed_move( GTK_FIXED( panel->fixed ), panel->items[j + 1],
                                panel->plugin_pos_x[j + 1],
                                panel->plugin_pos_y[j + 1] );
        }
    }
}

/* FIXME: this should do some real ordering according to time and priority
 * but the spec does not yet specifiy how.
 * For now it only removes empty slots.
 */
static void reorder_items( StatusBar *panel )
{
    gint i, j;
    gboolean last;

    g_return_if_fail( panel );

    /* go through all dynamic items */
    for( i = HILDON_STATUS_BAR_FIRST_DYN_SLOT; 
         i < HILDON_STATUS_BAR_MAX_NO_OF_ITEMS; ++i )
    {
        /* skip to next, if i is an item */
        if( panel->items[i] != NULL )
            continue;

        last = TRUE;
        /* check if i-1 is the last item */
        for( j = i + 1; j < HILDON_STATUS_BAR_MAX_NO_OF_ITEMS; ++j )
        {
            if( panel->items[j] != NULL )
            {
                last = FALSE;
                break;
            }
        }

        if( last )
            break;

        /* i is a hole in the array, fill it */

        /* Switch this and the next item (which is NULL )*/
        for( j = i; j < HILDON_STATUS_BAR_MAX_NO_OF_ITEMS - 1; ++j )
        {
            panel->items[j] = panel->items[j + 1];
            panel->items[j + 1] = NULL;
            
            if( panel->items[j] == NULL )
                continue;
            
            /* update the fixed because location changes */
            if( j < HILDON_STATUS_BAR_VISIBLE_ITEMS - 1 )
                gtk_fixed_move( GTK_FIXED( panel->fixed ), panel->items[j],
                                panel->plugin_pos_x[j],
                                panel->plugin_pos_y[j] );
            
            /* add the plugin to the fixed */
            else if( j == HILDON_STATUS_BAR_VISIBLE_ITEMS - 1)
            {
                gtk_fixed_put( GTK_FIXED( panel->fixed ), panel->items[j],
                               panel->plugin_pos_x[j],
                               panel->plugin_pos_y[j] );
                gtk_widget_show( panel->items[j] );
            }
        }
    }
}

static void destroy_item( GtkObject *object,
                          gpointer user_data )
{
    gint i;
    StatusBar *panel;
    GtkWidget *item;

    g_return_if_fail( object );
    g_return_if_fail( user_data );

    panel = (StatusBar*)user_data;
    item = (GtkWidget *)object;

    /* find the item and destroy it */
    for( i = 0; i < HILDON_STATUS_BAR_MAX_NO_OF_ITEMS; ++i )
    {
        if( panel->items[i] == item )
        {
            g_object_unref( item );
            /*gtk_widget_destroy( panel->items[i] );*/
            panel->items[i] = NULL;
            break;
        }
    }

    reorder_items( panel );
}

int main( int argc, char *argv[] )
{
  StatusBar *panel;
  osso_context_t *osso;
  gint i;

  gtk_init( &argc, &argv );

  /* initialize osso */
  osso = osso_initialize( HILDON_STATUS_BAR_NAME, 
                          HILDON_STATUS_BAR_VERSION, FALSE, NULL );
  if( !osso )
  {  
      osso_log( LOG_ERR, "osso_initialize failed! Exiting.." );
      return 1;
  }
  
  panel = g_new0( StatusBar, 1 );
  if( !panel )
  {  
      osso_log( LOG_ERR, "g_new0 failed! Exiting.." );
      osso_deinitialize( osso );
      return 1;
  }

  /* initialize panel */
  if( init_dock( panel ) == -1 )
  {
      osso_log( LOG_ERR, "init_dock failed! Exiting.." );
      osso_deinitialize( osso );
      g_free( panel );
      return 1 ;
  }

  add_prespecified_items( panel );

  gtk_widget_show_all( panel->window );
  
  /* set RPC cb */
  if( osso_rpc_set_default_cb_f( osso, rpc_cb, panel ) != OSSO_OK )
  {
      osso_log( LOG_ERR, "osso_rpc_set_default_cb_f() failed" );
  }
  gtk_main();
  
  for( i = 0; i < HILDON_STATUS_BAR_MAX_NO_OF_ITEMS; ++i )
  {
      if( panel->items[i] != NULL )
      {
          gtk_widget_destroy( panel->items[i] );
      }
  }
   
  gtk_widget_destroy( panel->window );
  g_free( panel );
  osso_deinitialize( osso );

  return 0;
}

