/* -*- mode: C; c-file-style: "stroustrup"; indent-tabs-mode: nil; -*- */ 

/*
 * This file is part of hildon-control-panel
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Karoliina Salminen <karoliina.t.salminen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


#include "hildon-control-panel-main.h"

typedef struct _AppData AppData;

struct _AppData
{
    HildonApp *app;
    HildonAppView *appview;
    GtkWidget * grid;
    GtkWindow *window;
    osso_context_t *osso;

    /* for state save data */
    gint icon_size;
    gchar * focused_filename;
    gchar * saved_focused_filename;
    gint scroll_value;
    gint execute;
};


AppData state_data;

/* delay to wait from callback to actually reading the entries in msecs*/
#define DIRREADDELAY 500

/* Message bus definitions */
#define HILDON_CONTROL_PANEL_MCE_REQUEST_PATH "mce/request"
#define HILDON_CONTROL_PANEL_DBUS_MCE_SERVICE "mce"
#define HILDON_CONTROL_PANEL_MCE_PASSWORD_VALIDATE "validate_device_lockcode"
#define HILDON_CONTROL_PANEL_DEFAULT_SALT "IO"
#define HILDON_CONTROL_PANEL_DBUS_APPKILLER_SERVICE "osso_app_killer"
#define HILDON_CONTROL_PANEL_APPKILLER_RESTORE_DEFAULTS \
 "restore_factory_settings"
#define HILDON_CONTROL_PANEL_DBUS_STARTUP_WIZARD_SERVICE "osso_startup_wizard"
#define HILDON_CONTROL_PANEL_STARTUP_WIZARD_LAUNCH "launch_startup_wizard"

/* Sets DNOTIFY callback for the .desktop dir */
static int _init_dnotify(char * path );
static void _dnotify_callback_f(char * path,gpointer data);


/* Create the GUI entries */
static void _init_gui (void);
/* Set the grid to use large/small icons */
static void _large_icons(void);
static void _small_icons(void);


/* Create & free state data */
static void _create_data(void);
static void _destroy_data( void );
/* State saving functions */
static void _save_state(void);
static void _retrieve_state(void);
static void _enforce_state(void);


/* Callback for topping */
static void _topmost_status_acquire(void);
/* Callback for saving state */
static void _topmost_status_lose(void);
/* Callback for resetting factory settings */
static void _reset_factory_settings( GtkWidget *widget, gpointer data );
/* Callback for startup wizard launch */
static void _run_startup_wizard( GtkWidget *widget, gpointer data );
/* Callback for help function */
static void _launch_help( GtkWidget *widget, gpointer data );
/* Callback for menu item small/large icons */
static void _iconsize( GtkWidget *widget, gpointer data );
static void _my_open( GtkWidget *widget, gpointer data );
static void _my_quit( GtkWidget *widget, gpointer data );


/* Callback for launching an item */
static void _launch (MBDotDesktop *, gpointer, gboolean);
static void _focus_change (MBDotDesktop *, gpointer);



/**********************************************************************
 * Main program
 **********************************************************************/
int 
main(int argc, char ** argv)
{
 
    setlocale( LC_ALL, "" );
    
    bindtextdomain( PACKAGE, LOCALEDIR );

    bind_textdomain_codeset( PACKAGE, "UTF-8" );

    textdomain( PACKAGE );
    
    _init_dnotify( CONTROLPANEL_ENTRY_DIR );

    _create_data();       /* Create state data */

    _retrieve_state();    /* State data from disk */
    
    gtk_init( &argc, &argv );
    
    _init_gui();  /* Create GUI components */

    _enforce_state();     /* realize the saved state */
    
    gtk_widget_show_all( GTK_WIDGET( state_data.app ) );

    if(state_data.execute==1) {
        _launch(hildon_cp_applist_get_entry(state_data.focused_filename),
                NULL, FALSE );
    }
    
    gtk_main();

    _destroy_data( );
    
    return 0;
}

/**********************************************************************
 * DNotify stuff
 **********************************************************************/

static int callback_pending = 0;

/* called by dnotify_callback_f after a timeout (to prevent exessive
   callbacks. */

static gboolean _dnotify_reread_desktop_entries()
{
    callback_pending = 0;
    hildon_cp_applist_reread_dot_desktops();
    return FALSE; /* do not have the timeout called again */
}

static void _dnotify_callback_f(char * path,gpointer data)
{
    if(!callback_pending)
    {
        callback_pending = 1;
        g_timeout_add( DIRREADDELAY, 
                       (GSourceFunc)_dnotify_reread_desktop_entries,
                       path );
    }
}

static int _init_dnotify(char * path )
{
  hildon_return_t ret;

  ret = hildon_dnotify_handler_init();
  if( ret!=HILDON_OK )
  {
      return ret;
  }
  ret = hildon_dnotify_set_cb( _dnotify_callback_f, path, NULL );
  if( ret!=HILDON_OK )
  {
      return ret;
  }
  return HILDON_OK;
}



/**********************************************************************
 * GUI helper functions
 **********************************************************************/

static void _large_icons(void)
{
    if(state_data.grid)
        hildon_grid_set_style(HILDON_GRID(state_data.grid),
                              "largeicons-cp");
}


static void _small_icons(void)
{
    if(state_data.grid)
        hildon_grid_set_style(HILDON_GRID(state_data.grid),
                              "smallicons-cp");

}

static void _init_gui(void)
{
    GtkMenu *menu=NULL;
    GtkWidget *sub_view=NULL;
    GtkWidget *sub_tools=NULL;
    GtkWidget *mi = NULL;
    GSList *menugroup = NULL;
    GtkWidget *grid = NULL;

    state_data.app = HILDON_APP( hildon_app_new() );

    hildon_app_set_title( state_data.app, _("ctrp_ap_control_panel") );

    hildon_app_set_two_part_title( state_data.app, FALSE );

    g_signal_connect( G_OBJECT( state_data.app ), "destroy", 
                      G_CALLBACK( _my_quit ),NULL);

    state_data.appview = HILDON_APPVIEW( hildon_appview_new(
                                             "MainView" ) );

    hildon_app_set_appview( state_data.app, state_data.appview );
    
    g_signal_connect(G_OBJECT(state_data.app), "topmost_status_lose",
                     G_CALLBACK(_topmost_status_lose), NULL);
    g_signal_connect(G_OBJECT(state_data.app), "topmost_status_acquire",
                     G_CALLBACK(_topmost_status_acquire), NULL);
    
    /* Menu */
    menu = GTK_MENU( hildon_appview_get_menu( state_data.appview ) );
   
    mi = gtk_menu_item_new_with_label( _("copa_me_open") );
    
    g_signal_connect( G_OBJECT( mi ), "activate", 
                      G_CALLBACK( _my_open ), NULL ); 

    gtk_menu_shell_append( GTK_MENU_SHELL( menu ), mi ); 
    
    
    /* View submenu */
    sub_view = gtk_menu_new();

    mi = gtk_menu_item_new_with_label( _("copa_me_view") );
    
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), sub_view);

    gtk_menu_shell_append( GTK_MENU_SHELL( menu ), mi );
        
    /* Small icon size */
    mi = gtk_radio_menu_item_new_with_label( menugroup,
                                             _("copa_me_view_small") );
    menugroup = gtk_radio_menu_item_get_group( GTK_RADIO_MENU_ITEM (mi) );

    if(state_data.icon_size == 0) {
        gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(mi), TRUE);
    }


    gtk_menu_shell_append( GTK_MENU_SHELL( sub_view ), mi );

    
    g_signal_connect( G_OBJECT( mi ), "activate", G_CALLBACK(_iconsize),
                      "small");

    /* Large icon size */
    mi = gtk_radio_menu_item_new_with_label( menugroup,
                                             _("copa_me_view_large") );

    if (state_data.icon_size == 1) {
        gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mi), TRUE);
    }
    menugroup = gtk_radio_menu_item_get_group( GTK_RADIO_MENU_ITEM (mi) );
   
    gtk_menu_shell_append( GTK_MENU_SHELL( sub_view ), mi );
    
    g_signal_connect( G_OBJECT( mi ), "activate", G_CALLBACK(_iconsize),
                      "large");



    /* Tools submenu*/
    sub_tools = gtk_menu_new();

    mi = gtk_menu_item_new_with_label( _("copa_me_tools") );
    
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), sub_tools);

    gtk_menu_shell_append( GTK_MENU_SHELL( menu ), mi );


    /* Run startup wizard */
    mi = gtk_menu_item_new_with_label
        ( _("copa_me_tools_setup_wizard") );

    gtk_menu_shell_append( GTK_MENU_SHELL( sub_tools ), mi );
    
    g_signal_connect( G_OBJECT( mi ), "activate", 
                      G_CALLBACK(_run_startup_wizard), NULL);

    /* Reset Factory Settings */
    mi = gtk_menu_item_new_with_label
        ( _("copa_me_tools_rfs") );

    gtk_menu_shell_append( GTK_MENU_SHELL( sub_tools ), mi );
    
    g_signal_connect( G_OBJECT( mi ), "activate", 
                      G_CALLBACK(_reset_factory_settings), NULL);

    
    /* Help! */

    mi = gtk_menu_item_new_with_label
        ( _("copa_me_tools_help") );

    gtk_menu_shell_append( GTK_MENU_SHELL( sub_tools ), mi );
    
    g_signal_connect( G_OBJECT( mi ), "activate", 
                      G_CALLBACK(_launch_help), NULL);

    /* close */
    mi = gtk_menu_item_new_with_label( _("copa_me_close") );
    
    g_signal_connect(GTK_OBJECT(mi), "activate",
                     G_CALLBACK(_my_quit),
                     NULL);

    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    
    gtk_widget_show_all( GTK_WIDGET( menu ));
    
    hildon_cp_applist_initialize( _launch, NULL,
                                    _focus_change, NULL,
                                    GTK_WIDGET( state_data.appview ), 
                                    CONTROLPANEL_ENTRY_DIR);

    grid = hildon_cp_applist_get_grid();
    {
        GValue val = {0,};
        g_value_init(&val, G_TYPE_STRING);
        g_value_set_string(&val, _("ctrp_ap_no_plugins_to_show"));
        g_object_set_property(G_OBJECT (grid), "empty_label", &val);
    }

    gtk_widget_show_all(GTK_WIDGET(state_data.app));

    state_data.window = GTK_WINDOW( gtk_widget_get_toplevel(
                                        GTK_WIDGET( state_data.app ) ) );

    state_data.grid = hildon_cp_applist_get_grid();
}


/************************************************************************
 * Application state data handling
 ************************************************************************/

/* Create a struct for the application state data */

static void _create_data(void)
{
    state_data.osso = osso_initialize(APP_NAME, APP_VERSION, TRUE, NULL);
    if(!state_data.osso)
    {
        osso_log(LOG_ERR,
                "Error initializing osso -- check that D-BUS is running");
        exit(-1);
    }
    
    g_assert(state_data.osso);
    state_data.icon_size=0;
    state_data.focused_filename=NULL;
    state_data.saved_focused_filename=NULL;
    state_data.execute = 0;
    state_data.scroll_value = 0;
}

/* Clean the application state data struct */

static void _destroy_data( void )
{
    if( state_data.osso )      
        osso_deinitialize(state_data.osso);
    if( state_data.focused_filename )   
        g_free( state_data.focused_filename );
    if( state_data.saved_focused_filename )  
        g_free( state_data.saved_focused_filename );
}

/* save state data on disk */

static void _save_state( void ) {
    gint scrollvalue = hildon_grid_get_scrollbar_pos(HILDON_GRID(
                                                        state_data.grid));
    gchar contents[HILDON_CONTROL_PANEL_STATEFILE_MAX];
    gint fd, ret;
    
    
    ret = g_snprintf(contents, HILDON_CONTROL_PANEL_STATEFILE_MAX,
                     "iconsize=%d\n"
                     "focused=%s\n"
                     "scroll=%d\n"
                     "execute=%d\n",
                     state_data.icon_size,
                     state_data.focused_filename != NULL ?
                     state_data.focused_filename : "",
                     scrollvalue,
                     state_data.execute);
    if(ret > HILDON_CONTROL_PANEL_STATEFILE_MAX)
    {
        osso_log(LOG_ERR, "Error saving state -- too long state");
        return;
    }   

    fd = osso_state_open_write(state_data.osso);
    if( fd == -1)
    {
        osso_log(LOG_ERR, "Opening state file for writing failed");
        return;
    }

    ret = write(fd, contents, strlen(contents));
    if(ret != strlen(contents))
    {
        osso_log(LOG_ERR, 
      "Writing state save file failed. Tried to write %d bytes, wrote %d",
                 strlen(contents), ret);
    }
    osso_state_close(state_data.osso, fd);
}

/* Read the saved state from disk */

static void _retrieve_state( void ) {
    gchar buff[HILDON_CONTROL_PANEL_STATEFILE_MAX];
    size_t rc;
    gint fd;
    gchar *eq;
    gchar *nl;
    gchar *start;
    gint length;

    fd = osso_state_open_read(state_data.osso);
    if( fd == -1)
    {
        return;
    }

    rc = read(fd, buff, HILDON_CONTROL_PANEL_STATEFILE_MAX);
    osso_state_close(state_data.osso, fd);

    if(rc >= HILDON_CONTROL_PANEL_STATEFILE_MAX)
    {
        osso_log(LOG_ERR,
                 "Error retrieving state -- too long state file %d", rc);
        return;
    }   
   
    start = buff;
    length = rc;
    while( ((nl = g_strstr_len(start, length, "\n")) != NULL) &&
           ((eq = g_strstr_len(start, length, "=")) != NULL)) {
        *nl = '\0';
        *eq = '\0';
        
        if(strcmp(start, "iconsize")==0) {
            sscanf(eq+1, "%d", &state_data.icon_size);
        } else if(strcmp(start, "focused")==0) {
            state_data.saved_focused_filename = g_strdup(eq+1);
        } else if(strcmp(start, "scroll")==0) {
            sscanf(eq+1, "%d", &state_data.scroll_value);
        } else if(strcmp(start, "execute")==0) {
            sscanf(eq+1, "%d", &state_data.execute);
        } else {
            osso_log(LOG_ERR, 
                     "Error retrieving state -- unknown field %s", start);
        }
        
        length = length - (nl - start + 1);
        start=nl+1;
    }
}

static void _enforce_state(void)
{
    /* Actually enforce the saved state */
    state_data.icon_size == 1 ? _large_icons() : _small_icons();
    
    if(state_data.saved_focused_filename)
    {
        hildon_cp_applist_focus_item( state_data.saved_focused_filename );
 
        g_free(state_data.saved_focused_filename);
        state_data.saved_focused_filename = NULL;
    }

    hildon_grid_set_scrollbar_pos(HILDON_GRID(state_data.grid),
                                  state_data.scroll_value);
    /* main() will start the possible plugin in state_data.execute*/
}


/***********************************************************************
 * Callbacks
 ***********************************************************************/

/***** Top/Untop callbacks *****/

/* save state when we get untopped*/
static void _topmost_status_lose(void)
{
    _save_state();
    hildon_app_set_killable(state_data.app, TRUE);
}

/* Remove the "save to OOM kill" hint when topped */

static void _topmost_status_acquire(void)
{
    mb_util_window_activate(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(
                                GTK_WIDGET(state_data.app)->window));
    hildon_app_set_killable(state_data.app, FALSE);
}


/****** Menu callbacks ******/

static void _my_open( GtkWidget *widget, gpointer data )
{
    _launch( hildon_cp_applist_get_entry( state_data.focused_filename ),
             NULL, TRUE );
}

static void _my_quit(GtkWidget *widget, gpointer data)
{
    _save_state();
    hildon_app_set_killable(state_data.app, FALSE);
    gtk_main_quit();
}

static void _iconsize( GtkWidget *widget, gpointer data )
{
    if(!gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget)))
        return;
    
    /* Avoid instant segfault, but can stuff be left uninitialized?
       Hope not. */
    if (data == NULL)
        return;
    
    if( strcmp((gchar *)data, "large") == 0)
    {
        _large_icons();
        state_data.icon_size = 1;
    }
    else if( strcmp((gchar *)data, "small") == 0)
    {
        _small_icons();
        state_data.icon_size = 0;
    }
}

static void _run_startup_wizard( GtkWidget *widget, gpointer data)
{

    osso_rpc_t returnvalues;
    osso_return_t returnstatus;
    returnstatus = osso_rpc_run_with_defaults
        (state_data.osso, HILDON_CONTROL_PANEL_DBUS_STARTUP_WIZARD_SERVICE,
         HILDON_CONTROL_PANEL_STARTUP_WIZARD_LAUNCH,
         &returnvalues, 
         DBUS_TYPE_INVALID);    

    if (returnstatus != OSSO_OK) {
        switch (returnstatus)
        {
        case OSSO_OK:
            osso_log(LOG_ERR, "1=2!");
            break;
        case OSSO_INVALID:
            fprintf(stderr, "Invalid parameter\n");
            osso_log(LOG_ERR, 
                     "Invalid parameter in startup_wizard launch");
            break;
        case OSSO_RPC_ERROR:
        case OSSO_ERROR:
        case OSSO_ERROR_NAME:
        case OSSO_ERROR_NO_STATE:
        case OSSO_ERROR_STATE_SIZE:
            if (returnvalues.type == DBUS_TYPE_STRING) {
                osso_log(LOG_ERR, "Startup wizard launch failed: %s\n", 
                         returnvalues.value.s);
            }
            else {
                osso_log(LOG_ERR, 
                         "Startup wizard launch failed, unspecified");
            }
            break;            

        }
    }
}

static void _reset_factory_settings( GtkWidget *widget, gpointer data )
{



    gint i;
    gchar *password;
    GtkWidget *confirm_dialog;
    
    if ( data == NULL ) {
        confirm_dialog = hildon_note_new_confirmation
            (GTK_WINDOW(widget), 
             _("refs_la_all_application_must_be_closed"));
    }
    else {
        confirm_dialog = hildon_note_new_confirmation
            (GTK_WINDOW(widget), 
             _("foo_bar_password_incorrect???"));        
    }

    gtk_widget_show_all(confirm_dialog);
    
    i = gtk_dialog_run(GTK_DIALOG(confirm_dialog));

    if (i == GTK_RESPONSE_CANCEL ) {
        gtk_widget_destroy(GTK_WIDGET(confirm_dialog));
        return;    
    }

    gtk_widget_destroy(GTK_WIDGET(confirm_dialog));
    
    GtkWidget *pw_dialog = (hildon_get_password_dialog_new(
                                GTK_WINDOW(widget), FALSE));
    gtk_widget_show (pw_dialog);
    
    switch(gtk_dialog_run (GTK_DIALOG(pw_dialog)))
    {
    case GTK_RESPONSE_CANCEL:
        gtk_widget_destroy(GTK_WIDGET(pw_dialog));        
        return;
    }  
    password = strdup(hildon_get_password_dialog_get_password
                      (HILDON_GET_PASSWORD_DIALOG(pw_dialog)));


    
    if (password == NULL) {
        return;
    }
    else {
        fprintf(stderr, "Password is: %s\n", password);
    }


    if (i == GTK_RESPONSE_OK) {  
        osso_rpc_t returnvalues;
        osso_return_t returnstatus;


        returnstatus = osso_rpc_run_with_defaults
            (state_data.osso, HILDON_CONTROL_PANEL_DBUS_MCE_SERVICE, 
             HILDON_CONTROL_PANEL_MCE_PASSWORD_VALIDATE, &returnvalues, 
             DBUS_TYPE_STRING, 
             crypt(password, HILDON_CONTROL_PANEL_DEFAULT_SALT), 
             DBUS_TYPE_INVALID);
        
        switch (returnstatus)
        {
        case OSSO_INVALID:
            fprintf(stderr, "Invalid parameter\n");
            break;
        case OSSO_RPC_ERROR:
        case OSSO_ERROR:
        case OSSO_ERROR_NAME:
        case OSSO_ERROR_NO_STATE:
        case OSSO_ERROR_STATE_SIZE:
            if (returnvalues.type == DBUS_TYPE_STRING) {
                osso_log(LOG_ERR, "Lockcode query failed: %s\n", 
                        returnvalues.value.s);
            }
            else {
                osso_log(LOG_ERR, "Lockcode query failed, unspecified");
            }
            break;
            
        case OSSO_OK:
            if (returnvalues.type==DBUS_TYPE_BOOLEAN &&
                returnvalues.value.b == TRUE) {
                
                returnstatus = osso_rpc_run_with_defaults
                    (state_data.osso, 
                     HILDON_CONTROL_PANEL_DBUS_APPKILLER_SERVICE, 
                     HILDON_CONTROL_PANEL_APPKILLER_RESTORE_DEFAULTS, 
                     &returnvalues, 
                     DBUS_TYPE_INVALID);
                
                switch (returnstatus)
                {
                case OSSO_INVALID:
                    fprintf(stderr, "Invalid parameter\n");
                    break;
                case OSSO_RPC_ERROR:
                case OSSO_ERROR:
                case OSSO_ERROR_NAME:
                case OSSO_ERROR_NO_STATE:
                case OSSO_ERROR_STATE_SIZE:
                    if (returnvalues.type == DBUS_TYPE_STRING) {
                        osso_log(LOG_ERR, "Lockcode query failed: %s\n", 
                                 returnvalues.value.s);
                    }
                    else {
                        osso_log(LOG_ERR, 
                                 "Lockcode query failed, unspecified");
                    }
                    break;
                    
                case OSSO_OK:
                    break;
                    /* There is no need for case OSSO_OK: we'll be dead
                       ... if everyhting goes as planned, anyway! */

                }
            }
            else {
                fprintf(stderr, "What is the password?\n");
                _reset_factory_settings(widget, &returnvalues);

            }
        }
        
        
    }
    g_free ( password );

    gtk_widget_destroy(GTK_WIDGET(pw_dialog));

}

static void _launch_help( GtkWidget *widget, gpointer data)
{
    

    osso_return_t help_ret;
    
    help_ret = ossohelp_show(state_data.osso, OSSO_HELP_ID_CONTROL_PANEL, 0);



    switch (help_ret)
     {
         case OSSO_OK:
             break;
         case OSSO_ERROR:
             g_warning("HELP: ERROR (No help for such topic ID)\n");
             break;
         case OSSO_RPC_ERROR:
             g_warning("HELP: RPC ERROR (RPC failed for HelpApp/Browser)\n");
             break;
         case OSSO_INVALID:
             g_warning("HELP: INVALID (invalid argument)\n");
             break;
         default:
             g_warning("HELP: Unknown error!\n");
             break;
     }

}

/***** Applist callbacks *****/

static void _launch(MBDotDesktop* dd, gpointer data, gboolean user_activated )
{
    int ret;

    if(dd) {
        state_data.execute=1;
        /* FIXME: What if mb_dotdesktop_get returns NULL? --> Segfault */
        ret = osso_cp_plugin_execute( state_data.osso, 
                                      (char *)mb_dotdesktop_get(
                                          dd, "X-control-panel-plugin"),
                                      (gpointer)state_data.window,
				      user_activated );
        state_data.execute = 0;
    }
}

static void _focus_change(MBDotDesktop* dd, gpointer data )
{
    if(dd)
    {
        if(state_data.focused_filename)
            g_free(state_data.focused_filename);
        state_data.focused_filename = g_strdup(
            mb_dotdesktop_get_filename(dd));
    }
}
