/*
 * This file is part of DSP Gateway version 3.3.1
 *
 * Copyright (C) 2003-2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Toshihiro Kobayashi <toshihiro.kobayashi@nokia.com>
 *          Kiyotaka Takahashi <kiyotaka.takahashi@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

%{

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "dsp_dld.h"
#include "dld_malloc.h"
#include "dld_daemon.h"
#include "dld_cmd.h"
#include "dld_memmgr.h"

static inline void list_splice_tail_nostub(struct list_head *list,
					   struct list_head *head)
{
	struct list_head *oldlast = head->prev;
	struct list_head *last = list->prev;

	list->prev = oldlast;
	oldlast->next = list;

	last->next = head;
	head->prev = last;
}

#define VALIDATE_POINTER(p) \
do { \
	if ((p) == NULL) { \
		y_err = -1; \
		YYABORT; \
	} \
} while(0)

#define DIR_SCN_APPEND(dst,src) \
do { \
	if (dir_scn_append((dst), (src)) < 0) { \
		y_err = -1; \
		YYABORT; \
	} \
} while(0)

/* nyrt: not yet registered tree */
#define NYRT_CNT_MAX	256
static struct expr_tree *nyrt_pool[NYRT_CNT_MAX];
static int nyrt_cnt = 0;

static int nyrt_add(struct expr_tree *p)
{
	int ret = nyrt_cnt;

	nyrt_pool[nyrt_cnt] = p;
	if (++nyrt_cnt == NYRT_CNT_MAX) {
		prmsg("nyrt_cnt overflow!\n");
		exit(1);
	}

	return ret;
}

static void nyrt_del(int id)
{
	nyrt_pool[id] = NULL;
}

static void nyrt_clear(void)
{
	nyrt_cnt = 0;
}

static void nyrt_free(void)
{
	int i;

	for (i = 0; i < nyrt_cnt; i++) {
		if (nyrt_pool[i])
			expr_tree_free(nyrt_pool[i]);
	}
	nyrt_cnt = 0;
}

static struct expr_tree *EXPR_TREE_NEW(enum expr_type t, u32 val, char *s)
{
	struct expr_tree *p = expr_tree_new(t, val, s);

	if (p != NULL)
		nyrt_add(p);
	return p;
}

#define EXPR_TREE_FREE(exprwr) \
do { \
	expr_tree_free((exprwr).p); \
	nyrt_del((exprwr).nyrt_id); \
} while(0)

typedef struct Codeval {
int val;
char* name;
} codeval;

struct dirgrp {
	struct list_head list_head;
	struct directive *dir;
};

#define dirgrp_for_each(pos, head) \
	list_for_each_entry(pos, head, list_head)
#define dirgrp_for_each_safe(pos, n, head) \
	list_for_each_entry_safe(pos, n, head, list_head)

static LIST_HEAD(g_dirgrp);

static struct dirgrp *dirgrp_new(struct directive *dir)
{
	struct dirgrp *dg = dld_malloc(sizeof(struct dirgrp), "dirgrp");

	if (dg == NULL)
		return NULL;
	dg->dir = dir;
	INIT_LIST_HEAD(&dg->list_head);

	return dg;
}

static void dirgrp_add(struct dirgrp *dg)
{
	list_add_tail(&dg->list_head, &g_dirgrp);
}

static void dirgrp_freelist(void)
{
	struct dirgrp *dg, *tmp;

	dirgrp_for_each_safe(dg, tmp, &g_dirgrp) {
		list_del(&dg->list_head);
		/* do not free dir */
		dld_free(dg, "dirgrp");
	}
}

static int dirgrp_setinfo(struct directive *src)
{
	struct dirgrp *dg;
	int ret = 0;

	/*
	 * mem is MEMORY name string at this moment
	 */
	list_for_each_entry(dg, &g_dirgrp, list_head) {
		if (dir_scn_append(&dg->dir->load, &src->load) < 0) {
			ret = -1;
			goto out;
		}
		if (dir_scn_append(&dg->dir->run, &src->run) < 0) {
			ret = -1;
			goto out;
		}
	}
out:
	dir_scn_clear(&src->load);
	dir_scn_clear(&src->run);

	return ret;
}

static struct dir_scn tmp_dir_scn;
static struct directive *cur_directive = NULL;
static struct lkcmd *user_lkcmd;
static int y_err;

int yylex(void);
int yyerror(char *);
void dir_convert_memptr(struct list_head *dirlist);

%}

%union {
	struct expr_tree_wrapper {
		struct expr_tree *p;
		int nyrt_id;
	} expr;
}

%token MEMORY
%token ORIGIN
%token <expr> ID
%token LENGTH
%token COMMA

%token IDENTIFIER
%token <expr> CONSTANT
%token RIGHT_OP
%token SECTIONS
%token ALIGN
%token ATTR
%token BLOCK
%token COPY
%token DSECT
%token FILL
%token GROUP
%token LOAD
%token NOLOAD
%token PAGE
%token RANGE
%token RUN
%token SPARE
%token TYPE
%token UNION

%token '.'
%token ':'
%token '['
%token ']'
%token '>'
%token '|'

%token '('
%token ')'
%token ','
%token '{'
%token '}'
%token ';'
%token '='
%left '+' '-'
%left '*' '/'

%token ADD_ASSIGN
%token SUB_ASSIGN
%token MUL_ASSIGN
%token DIV_ASSIGN
%token MOD_ASSIGN
%token AND_ASSIGN
%token XOR_ASSIGN
%token OR_ASSIGN
%token LEFT_OP
%token LE_OP
%token GE_OP
%token EQ_OP
%token NE_OP

%type <expr.p> assignment
%type <expr.p> E
%type <expr.p> T
%type <expr.p> F
%%

commands	: directives
{
}
		;

directives	: directives directive
{
}
		|
{
}
		;
directive 	: MEMORY '{' mementries '}'
{
	/* MEMORY directive */
}
		| SECTIONS
{
	/* SECTIONS directive */
	cur_directive = directive_new();
	VALIDATE_POINTER(cur_directive);
	dir_scn_init(&tmp_dir_scn);
}
'{' sctentries '}'
{
	directive_free(cur_directive);
	cur_directive = NULL;
}
		| assignment
{
	/* Global assignment */
	struct globexpr *e;

	e = globexpr_new($1);
	VALIDATE_POINTER(e);
	globexpr_add(&user_lkcmd->exprlist, e);
	nyrt_clear();
}
		| '-' ID
{
	/* Linkage option */
	struct lopt *o;

	VALIDATE_POINTER($2.p);
	o = lopt_new(1 + strlen($2.p->name) + 1);
	VALIDATE_POINTER(o);
	sprintf(o->opt, "-%s", $2.p->name);
	lopt_add(&user_lkcmd->loptlist, o);
	EXPR_TREE_FREE($2);
}
		| '-' ID ID
{
	/* Linkage option */
	struct lopt *o;

	VALIDATE_POINTER($2.p);
	VALIDATE_POINTER($3.p);
	o = lopt_new(1 + strlen($2.p->name) + 1 + strlen($3.p->name) + 1);
	VALIDATE_POINTER(o);
	sprintf(o->opt, "-%s %s", $2.p->name, $3.p->name);
	lopt_add(&user_lkcmd->loptlist, o);
	EXPR_TREE_FREE($2);
	EXPR_TREE_FREE($3);
}
		| '-' ID CONSTANT
{
	/* Linkage option */
	struct lopt *o;

	VALIDATE_POINTER($2.p);
	VALIDATE_POINTER($3.p);
	o = lopt_new(1 + strlen($2.p->name) + 1 + 2 + 8 + 1);
	VALIDATE_POINTER(o);
	sprintf(o->opt, "-%s 0x%08lx", $2.p->name, $3.p->val);
	lopt_add(&user_lkcmd->loptlist, o);
	EXPR_TREE_FREE($2);
	EXPR_TREE_FREE($3);
}
		;

mementries	: mementries pageopt mementry
{
}
		|
{
}
		;

sctentries	: sctentries sctentry
{
}
		|
{
}
		;

pageopt		: PAGE CONSTANT ':'
{
	VALIDATE_POINTER($2.p);
	if ($2.p->val != 0) {
		prmsg("PAGE %d\n", $2.p->val);
		prmsg("MEMORY directive: PAGE option other than 0 is not supported.\n");
		y_err = -1;
		YYABORT;
	}
	EXPR_TREE_FREE($2);
}
		|
{
}
		;

mementry 	: ID ':' ORIGIN '=' CONSTANT ',' LENGTH '=' CONSTANT
{
	struct memmgr	*memmgr;

	VALIDATE_POINTER($1.p);
	VALIDATE_POINTER($5.p);
	VALIDATE_POINTER($9.p);
	memmgr = memmgr_new($1.p->name, $5.p->val, $9.p->val);
	VALIDATE_POINTER(memmgr);
	EXPR_TREE_FREE($1);
	EXPR_TREE_FREE($5);
	EXPR_TREE_FREE($9);
	memmgr->lkcmd = user_lkcmd;
	memmgr_add(&user_lkcmd->memlist, memmgr);
}
		| ID '(' ID ')' ':' ORIGIN '=' CONSTANT ',' LENGTH '=' CONSTANT
{
	struct memmgr	*memmgr;

	VALIDATE_POINTER($1.p);
	VALIDATE_POINTER($3.p);
	VALIDATE_POINTER($8.p);
	VALIDATE_POINTER($12.p);
	memmgr = memmgr_new($1.p->name, $8.p->val, $12.p->val);
	VALIDATE_POINTER(memmgr);
	EXPR_TREE_FREE($1);
	EXPR_TREE_FREE($3);
	EXPR_TREE_FREE($8);
	EXPR_TREE_FREE($12);
	memmgr->lkcmd = user_lkcmd;
	memmgr_add(&user_lkcmd->memlist, memmgr);
}
		|
{
}
		;

sctentry	: secname secinfo
{
	DIR_SCN_APPEND(&cur_directive->load, &tmp_dir_scn);
	dir_scn_flush(&tmp_dir_scn);
	directive_add(&user_lkcmd->dirlist, cur_directive);
	cur_directive = directive_new();
	VALIDATE_POINTER(cur_directive);
}
		| secname secoptions '{' inputsections '}' secinfo
{
	DIR_SCN_APPEND(&cur_directive->load, &tmp_dir_scn);
	dir_scn_flush(&tmp_dir_scn);
	directive_add(&user_lkcmd->dirlist, cur_directive);
	cur_directive = directive_new();
	VALIDATE_POINTER(cur_directive);
}
		| secname secinfo '{' inputsections '}' secinfo
{
	DIR_SCN_APPEND(&cur_directive->load, &tmp_dir_scn);
	dir_scn_flush(&tmp_dir_scn);
	directive_add(&user_lkcmd->dirlist, cur_directive);
	cur_directive = directive_new();
	VALIDATE_POINTER(cur_directive);
}
		| GROUP
{
	prmsg("Warning: GROUP is not supported.\n");
}
'{' groupentries '}' secinfo
{
	/*
	 * cur_directive has only secinfo.
	 * copy this to all directives in dirgrp, then free.
	 */
	if (dirgrp_setinfo(cur_directive) < 0) {
		y_err = -1;
		YYABORT;
	}
	directive_free(cur_directive);
	cur_directive = directive_new();
	VALIDATE_POINTER(cur_directive);
	dirgrp_freelist();
}
		;

groupentries	: groupentries groupentry
{
}
		|
{
}
		;

groupentry	: secname
{
	struct dirgrp *dg;

	dg = dirgrp_new(cur_directive);
	VALIDATE_POINTER(dg);
	dirgrp_add(dg);
	directive_add(&user_lkcmd->dirlist, cur_directive);
	cur_directive = directive_new();
	VALIDATE_POINTER(cur_directive);
}
		| secname secoptions '{' inputsections '}'
{
	struct dirgrp *dg;

	dg = dirgrp_new(cur_directive);
	VALIDATE_POINTER(dg);
	dirgrp_add(dg);
	directive_add(&user_lkcmd->dirlist, cur_directive);
	cur_directive = directive_new();
	VALIDATE_POINTER(cur_directive);
}
		| secname '{' inputsections '}'
{
	struct dirgrp *dg;

	dg = dirgrp_new(cur_directive);
	VALIDATE_POINTER(dg);
	dirgrp_add(dg);
	directive_add(&user_lkcmd->dirlist, cur_directive);
	cur_directive = directive_new();
	VALIDATE_POINTER(cur_directive);
}
		|
{
}
		;

secinfo		: loadsec runsec
{
}
		| runsec loadsec
{
}
		;

loadsec		: '>' ID secoptions
{
	/* LOAD allocation rule */
	VALIDATE_POINTER($2.p);
	tmp_dir_scn.mem = dld_strdup($2.p->name, "dir_scn->mem");
	VALIDATE_POINTER(tmp_dir_scn.mem);

	/* Add properties related to this load allocation */
	DIR_SCN_APPEND(&cur_directive->load, &tmp_dir_scn);
	dir_scn_flush(&tmp_dir_scn);
	EXPR_TREE_FREE($2);
}
		| LOAD '=' ID secoptions
{
	/* LOAD allocation rule */
	VALIDATE_POINTER($3.p);
	tmp_dir_scn.mem = dld_strdup($3.p->name, "dir_scn->mem");
	VALIDATE_POINTER(tmp_dir_scn.mem);

	/* Add properties related to this load allocation */
	DIR_SCN_APPEND(&cur_directive->load, &tmp_dir_scn);
	dir_scn_flush(&tmp_dir_scn);
	EXPR_TREE_FREE($3);
}
		| LOAD '=' CONSTANT secoptions
{
	/* LOAD allocation rule */
	VALIDATE_POINTER($3.p);
	tmp_dir_scn.addr = $3.p->val;

	/* Add properties related to this load allocation */
	DIR_SCN_APPEND(&cur_directive->load, &tmp_dir_scn);
	dir_scn_flush(&tmp_dir_scn);
	EXPR_TREE_FREE($3);
}
		|
{
	DIR_SCN_APPEND(&cur_directive->load, &tmp_dir_scn);
	dir_scn_flush(&tmp_dir_scn);
}
		;

runsec		: RUN '=' ID secoptions
{
	/* RUN allocation rule */
	VALIDATE_POINTER($3.p);
	tmp_dir_scn.mem = dld_strdup($3.p->name, "dir_scn->mem");
	VALIDATE_POINTER(tmp_dir_scn.mem);

	/* Add a property for MEMORY directive */
	DIR_SCN_APPEND(&cur_directive->run, &tmp_dir_scn);
	dir_scn_flush(&tmp_dir_scn);
	EXPR_TREE_FREE($3);
}
		| RUN '>' ID secoptions
{
	/* RUN allocation rule */
	VALIDATE_POINTER($3.p);
	tmp_dir_scn.mem = dld_strdup($3.p->name, "dir_scn->mem");
	VALIDATE_POINTER(tmp_dir_scn.mem);

	/* Add properties related to this run allocation */
	DIR_SCN_APPEND(&cur_directive->run, &tmp_dir_scn);
	dir_scn_flush(&tmp_dir_scn);
	EXPR_TREE_FREE($3);
}
		|
{
}
		;

secname		: ID
{
	/* Allocate section name */
	VALIDATE_POINTER($1.p);
	if (cur_directive->scnnm == NULL) {
		cur_directive->scnnm = dld_strdup($1.p->name, "directive->scnnm");
		VALIDATE_POINTER(cur_directive->scnnm);
	} else {
		prmsg("*** ID ***\n");
	}
	EXPR_TREE_FREE($1);
}
		| ID secoptions ':'
{
	/* Allocate section name */
	VALIDATE_POINTER($1.p);
	if (cur_directive->scnnm == NULL) {
		cur_directive->scnnm = dld_strdup($1.p->name, "directive->scnnm");
		VALIDATE_POINTER(cur_directive->scnnm);
	} else {
		prmsg("*** ID secoptions ***\n");
	}
	EXPR_TREE_FREE($1);
}
		;

secoptions	: secoption secoptions
{
}
		| secoption ',' secoptions
{
}
		| 
{
}
		;

/*
 * Following allocation parameters are added to a temporary list.
 * Because it doesn't know whether this parameter is for load or run.
 * Parameters added to the temporary list splice an appropriate list,
 * when this section is parsed.
 */
secoption	: ALIGN '(' CONSTANT ')'
{
	VALIDATE_POINTER($3.p);
	tmp_dir_scn.align = $3.p->val;
	EXPR_TREE_FREE($3);
}
		| ALIGN '=' CONSTANT
{
	VALIDATE_POINTER($3.p);
	tmp_dir_scn.align = $3.p->val;
	EXPR_TREE_FREE($3);
}
		| BLOCK '(' CONSTANT ')'
{
	VALIDATE_POINTER($3.p);
	tmp_dir_scn.block = $3.p->val;
	EXPR_TREE_FREE($3);
}
		| BLOCK '=' CONSTANT
{
	VALIDATE_POINTER($3.p);
	tmp_dir_scn.block = $3.p->val;
	EXPR_TREE_FREE($3);
}
		| FILL '(' CONSTANT ')'
{
	VALIDATE_POINTER($3.p);
	tmp_dir_scn.fill = $3.p->val;
	EXPR_TREE_FREE($3);
}
		| FILL '=' CONSTANT
{
	VALIDATE_POINTER($3.p);
	tmp_dir_scn.fill = $3.p->val;
	EXPR_TREE_FREE($3);
}
		| '(' COPY ')'
{
	tmp_dir_scn.stype |= DIRSCN_STYPE_COPY;
}
		| '(' DSECT ')'
{
	tmp_dir_scn.stype |= DIRSCN_STYPE_DSECT;
}
		| '(' NOLOAD ')'
{
	tmp_dir_scn.stype |= DIRSCN_STYPE_NOLOAD;
}
		| PAGE CONSTANT
{
	VALIDATE_POINTER($2.p);
	if ($2.p->val != 0) {
		prmsg("PAGE %d\n", $2.p->val);
		prmsg("SECTIONS directive: PAGE option other than 0 is not supported.\n");
		y_err = -1;
		YYABORT;
	}
	EXPR_TREE_FREE($2);
}
		| 
{
}
		;

inputsections	: inputsections inputsection
{
}
		|
{
}
		;

inputsection	: ID '(' ID ')'
{
	/* Specify an input section from an input file */
	struct dir_iscn *p;

	VALIDATE_POINTER($1.p);
	VALIDATE_POINTER($3.p);
	p = dir_iscn_new_iscn($1.p->name, $3.p->name);
	VALIDATE_POINTER(p);
	list_add_tail((struct list_head *)p, &cur_directive->iscnlist);
	EXPR_TREE_FREE($1);
	EXPR_TREE_FREE($3);
}
		| '*' '(' ID ')'
{
	/* Specify an input section from all input files */
	struct dir_iscn *p;

	VALIDATE_POINTER($3.p);
	p = dir_iscn_new_iscn("*", $3.p->name);
	VALIDATE_POINTER(p);
	list_add_tail((struct list_head *)p, &cur_directive->iscnlist);
	EXPR_TREE_FREE($3);
}
		| inputassign
{
}
		;

inputassign	: '.' '=' ALIGN '(' CONSTANT ')' ';'
{
	/* Create a hole to align '.' */
	struct dir_iscn *p;

	VALIDATE_POINTER($5.p);
	p = dir_iscn_new_hole(HL_EQ_ALIGN, $5.p->val);
	VALIDATE_POINTER(p);
	list_add_tail((struct list_head *)p, &cur_directive->iscnlist);
	EXPR_TREE_FREE($5);
}
		| '.' ADD_ASSIGN CONSTANT ';'
{
	/* Create a hole with specified size */
	struct dir_iscn *p;

	VALIDATE_POINTER($3.p);
	p = dir_iscn_new_hole(HL_EQPLUS_ALIGN, $3.p->val);
	VALIDATE_POINTER(p);
	list_add_tail((struct list_head *)p, &cur_directive->iscnlist);
	EXPR_TREE_FREE($3);
}
		| '.' '=' CONSTANT ';'
{
	/* Create a hole with specified size */
	struct dir_iscn *p;

	VALIDATE_POINTER($3.p);
	p = dir_iscn_new_hole(HL_EQ, $3.p->val);
	VALIDATE_POINTER(p);
	list_add_tail((struct list_head *)p, &cur_directive->iscnlist);
	EXPR_TREE_FREE($3);
}
		| assignment
{
	/*
	 * Define global symbols and assign values to them
	 * A statement has binary tree structure.
	 */
	struct dir_iscn *p;

	p = dir_iscn_new_expr($1);
	VALIDATE_POINTER(p);
	list_add_tail((struct list_head *)p, &cur_directive->iscnlist);
	nyrt_clear();
	/*
	 * so don't free.
	 */
}

		;

assignment	: ID '=' E ';'
{
	struct expr_tree *p;

	VALIDATE_POINTER($1.p);
	p = EXPR_TREE_NEW(ET_EQ, 0, NULL);
	VALIDATE_POINTER(p);
	list_splice_tail_nostub((struct list_head*)p,  (struct list_head*)$1.p);
	list_splice_tail_nostub((struct list_head*)$3, (struct list_head*)$1.p);
	$$ = $1.p; // return first element
}
		;

E		: E '+' T
{
	struct expr_tree *p;

	p = EXPR_TREE_NEW(ET_PLUS, 0, NULL);
	VALIDATE_POINTER(p);
	list_splice_tail_nostub((struct list_head*)p,  (struct list_head*)$1);
	list_splice_tail_nostub((struct list_head*)$3, (struct list_head*)$1);
	$$ = $1;
}
		| E '-' T
{
	struct expr_tree *p;

	p = EXPR_TREE_NEW(ET_MINUS, 0, NULL);
	VALIDATE_POINTER(p);
	list_splice_tail_nostub((struct list_head*)p,  (struct list_head*)$1);
	list_splice_tail_nostub((struct list_head*)$3, (struct list_head*)$1);
	$$ = $1;
}
		| T
{
}
		;

T		: T '*' F
{
	struct expr_tree *p;

	p = EXPR_TREE_NEW(ET_MULT, 0, NULL);
	VALIDATE_POINTER(p);
	list_splice_tail_nostub((struct list_head*)p,  (struct list_head*)$1);
	list_splice_tail_nostub((struct list_head*)$3, (struct list_head*)$1);
	$$ = $1;
}
		| T '/' F
{
	struct expr_tree *p;

	p = EXPR_TREE_NEW(ET_DIV, 0, NULL);
	VALIDATE_POINTER(p);
	list_splice_tail_nostub((struct list_head*)p,  (struct list_head*)$1);
	list_splice_tail_nostub((struct list_head*)$3, (struct list_head*)$1);
	$$ = $1;
}
		| F
{
	$$ = $1;
}
		;

F		: ID
{
	VALIDATE_POINTER($1.p);
	$$ = $1.p;
}
		| CONSTANT
{
	VALIDATE_POINTER($1.p);
	$$ = $1.p;
}
		| '(' E ')'
{
	$$ = $2;
}
		| '.'
{
	$$ = EXPR_TREE_NEW(ET_VAR, 0, ".");
	VALIDATE_POINTER($$);
}
		;



%%

#include "lex.yy.c"

int yyerror(char *s)
{
	prmsg("%s\n", s);
	return -1;
}

/**
 * lkcmd_read - parse a command file
 * @lkcmd: pointer to lkcmd struct to be stored
 *
 * Parse a command file and build each lists.
 */
int lkcmd_read(struct lkcmd *lkcmd)
{
	yyin = fopen(lkcmd->fn, "r");
	if (yyin == NULL) {
		prmsg("%s open failed\n", lkcmd->fn);
		return -1;
	}

	user_lkcmd = lkcmd;
	y_err = 0;
	dir_scn_init(&tmp_dir_scn);

	yyparse();

	fclose(yyin);

	/* clear temporarily allocated resources */
	dir_scn_clear(&tmp_dir_scn);
	if (cur_directive)
		directive_free(cur_directive);
	dirgrp_freelist();
	nyrt_free();

	if ((yynerrs > 0) || (y_err < 0))
		return -1;
	return 0;
}
