/*
 * Copyright (c) 2003, 2004 Nokia
 * Author: tsavola@movial.fi
 *
 * This program is licensed under GPL (see COPYING for details)
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <syslog.h>
#include <pwd.h>
#include <netdb.h>
#include <signal.h>
#include <errno.h>
#include <pty.h>
#include <utmp.h>
#include <fcntl.h>
#include <getopt.h>
#include <time.h>
#include <limits.h>
#include <libgen.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/select.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <assert.h>
#include "types.h"
#include "daemon.h"
#include "common.h"
#include "protocol.h"
#include "buffer.h"
#include "mount.h"
#include "fakeroot.h"

#define ISSET(fd, set)           ((fd) >= 0 && FD_ISSET((fd), (set)))

#define debug_vector             if (debug_file) print_debug_vector

#define send_empty(data, ptype)  write_packet((data)->sd, (ptype), NULL, 0)

/* The environment of the process. */
extern char **environ;

/** Debug is enabled if this is set. */
FILE *debug_file = NULL;
static char *debug_filename = NULL;

static int daemon_pid = -1;

/** Server port number. */
static int port = DEFAULT_PORT;

static char *mount_cmd = DEFAULT_MOUNT_CMD;
static char *umount_cmd = DEFAULT_UMOUNT_CMD;
static char *bind_opt = DEFAULT_BIND_OPT;
static int mount_expiration = DEFAULT_MOUNT_EXPIRATION;

static char **valid_shells = NULL;

/** Doubly linked list of mount entries. */
static struct { mount_t *head, *tail; } all_mounts = { NULL, NULL };

/** The mounts of the handler processes. */
static pid_mounts_t *pid_mounts = NULL;

static struct { pid_t pid; const char *name; } debug_info = { -1, NULL };

void set_debug_name(const char *name)
{
	debug_info.pid = getpid();
	debug_info.name = name;
}

static void create_auth(handler_t *data,struct passwd *pass)
{
	char* path = malloc(strlen(pass->pw_dir) + 
			    strlen("/" CONFIG_NAME) + 1);
	
	if (path!=NULL) {
		FILE* f;
		strcpy(path, pass->pw_dir);
		strcat(path, "/" CONFIG_NAME);
		debug("Created auth (%s) %i",path);
		debug("Auth for (%s) is %s\n",data->host,
		      data->auth.pwd);
		f=fopen(path,"a+");
		if (f!=0) {
			fprintf(f,"%s %s\n",data->host,
				data->auth.pwd);
			fclose(f);
		}
		free (path);
	}
}

static int check_createuser(char* user)
{
	int status;
	pid_t pid=fork();

	if (pid < 0) {
		return 0;
	}
	
	if (pid == 0) {
		char* args[] = {
			"/usr/bin/check_createuser.sh",
			"Shall we create user",
			user,NULL};
		execv(args[0],args);
		sleep(1);
		abort();
	}
	
	TEMP_FAILURE_RETRY(waitpid(pid,&status,0));
	{
		int rstat=WEXITSTATUS(status);
		
		debug("Returned with status %i",rstat);
		
		return !rstat;
	}
	return 1;
}

/**
 * Prints only timestamp and pid to debug log.
 */
static void print_debug_prefix(void)
{
	struct timeval tv;
	struct tm *t;
	const char *domain = "";

	gettimeofday(&tv, NULL);
	t = gmtime(&tv.tv_sec);

	if (debug_info.pid == getpid()) {
		domain = debug_info.name;
	}

	fprintf(debug_file,
		"%02d-%02d-%04d %02d:%02d:%02d.%03ld %5d %7s ",
		t->tm_mday, t->tm_mon + 1, 1900 + t->tm_year,
		t->tm_hour, t->tm_min, t->tm_sec, tv.tv_usec / 1000,
		getpid(), domain);
}

static void open_debug_log(void)
{
	if (debug_file) {
		debug("Debugging is already enabled");
		return;
	}

	if (!debug_filename) {
		debug_filename = malloc(strlen(DEFAULT_DEBUG_FILENAME_F) + 5);
		if (!debug_filename) {
			error(oom);
			return;
		}

		sprintf(debug_filename, DEFAULT_DEBUG_FILENAME_F, port);
	}

	debug_file = fopen(debug_filename, "a");
	if (!debug_file) {
		error("Can't append to %s", debug_filename);
		return;
	}

	debug("Debugging enabled");
}

static void close_debug_log(void)
{
	if (debug_file) {
		debug("Debugging disabled");

		fclose(debug_file);
		debug_file = NULL;
	}
}

/**
 * Prints message to debug log.
 */
void print_debug(const char *msg, ...)
{
	va_list arg;

	print_debug_prefix();

	va_start(arg, msg);
	vfprintf(debug_file, msg, arg);
	va_end(arg);

	fprintf(debug_file, "\n");
	fflush(debug_file);
}

/**
 * Prints string vector to debug log.
 */
static void print_debug_vector(const char *msg,
			       char **vec)
{
	print_debug_prefix();
	fprintf(debug_file, "%s", msg);

	while (*vec) {
		fprintf(debug_file, " %s", *vec++);
	}

	fprintf(debug_file, "\n");
	fflush(debug_file);
}

/**
 * Prints message and errno description (if non-zero) to syslog and sends it to
 * the client (if data is non-null).
 */
static void print_error(handler_t *data,
			const char *progname,
			const char *msg,
			va_list arg,
			int priority)
{
	char str[1024] = { '\0' }, *err;
	size_t len;

	vsnprintf(str, sizeof (str) - 1, msg, arg);
	len = strlen(str);

	err = strerror(errno);
	if (errno && err && (len + strlen(err) + 3) < sizeof (str)) {
		strcat(str, " (");
		strcat(str, err);
		strcat(str, ")");

		len = strlen(str);
	}

	if (progname) {
		fprintf(stderr, "%s: %s\n", progname, str);
	} else {
		syslog(priority, "%s", str);
		debug(priority == LOG_WARNING ? "Warning: %s" : "Error: %s",
		      str);
	}

	if (data) {
		write_packet(data->sd, PTYPE_ERROR, str, len);
	}
}

void send_error(handler_t *data,
		const char *msg,
		...)
{
	va_list arg;

	va_start(arg, msg);
	print_error(data, NULL, msg, arg, LOG_ERR);
	va_end(arg);
}

/**
 * Prints message and errno description to syslog and stderr.
 */
static void error_err(const char *progname,
		      const char *msg,
		      ...)
{
	va_list arg;

	va_start(arg, msg);
	print_error(NULL, progname, msg, arg, LOG_ERR);
	va_end(arg);
}

/*
 * Prints message and errno description to syslog.
 */
void error(const char *msg,
	   ...)
{
	va_list arg;

	va_start(arg, msg);
	print_error(NULL, NULL, msg, arg, LOG_ERR);
	va_end(arg);
}

/**
 * Prints message to syslog.
 */
static void warn(const char *msg,
		 ...)
{
	va_list arg;

	errno = 0;

	va_start(arg, msg);
	print_error(NULL, NULL, msg, arg, LOG_WARNING);
	va_end(arg);
}

/**
 * Sends message to the client.
 */
static void send_message(handler_t *data,
			 const char *msg)
{
	debug("Sending message: %s", msg);
	write_packet(data->sd, PTYPE_MESSAGE, (char *) msg, strlen(msg));
}

/**
 * Passes the contents of BUF to send_error().
 */
static void flush(handler_t *data,
		  char *buf,
		  size_t *lenp)
{
	if (*lenp > 0) {
		buf[*lenp] = '\0';
		*lenp = 0;

		errno = 0;
		send_error(data, buf);
	}
}

/**
 * Executes argv[0] in a child process.
 * @return -1 on error, 0 otherwise
 */
static int execute(handler_t *data,
		   char **argv)
{
	pid_t pid;
	int status, err[2];
	size_t pos = 0;
	char line[1024];

	if (pipe(err) < 0) {
		send_error(data, "Can't create pipe");
		return -1;
	}

	pid = fork();
	if (pid < 0) {
		send_error(data, "Can't fork");
		return -1;
	}

	if (pid == 0) {
		/* Child */

		set_debug_name("EXECUTE");

		if (dup2(err[1], STDOUT_FILENO) != STDOUT_FILENO) {
			send_error(data, "Can't duplicate pipe as stdout");
			exit(1);
		}
		if (dup2(err[1], STDERR_FILENO) != STDERR_FILENO) {
			send_error(data, "Can't duplicate pipe as stderr");
			exit(1);
		}
		close(err[0]);

		execv(argv[0], argv);

		send_error(data, "Can't execute command: %s", argv[0]);
		exit(1);
	}

	/* Parent */

	close(err[1]);

	while (1) {
		ssize_t len;
		char c;

		len = read(err[0], &c, 1);
		if (len <= 0) {
			flush(data, line, &pos);

			if (len < 0 && errno == EINTR) {
				continue;
			}

			break;
		}

		if (c == '\n' || c == '\0') {
			flush(data, line, &pos);
			continue;
		}

		if (pos == sizeof (line)) {
			flush(data, line, &pos);
		}

		line[pos++] = c;
	}

	close(err[0]);

	if (waitpid(pid, &status, 0) != pid) {
		return -1;
	}

	if (status < 0) {
		return -1;
	}

	errno = 0;
	return (WEXITSTATUS(status) != 0) ? -1 : 0;
}

/**
 * Check that shell is listed in valid_shells.
 */
static bool_t validate_shell(handler_t *data,
			     char *shell)
{
	char **s;

	if (!shell || *skip_spaces(shell) == '\0') {
		debug("No shell defined");
		return FALSE;
	}

	for (s = valid_shells; *s; s++) {
		if (strcmp(shell, *s) == 0) {
			debug("Shell %s is valid", shell);
			return TRUE;
		}
	}

	debug("Shell %s is invalid", shell);
	return FALSE;
}

static void parse_shells(const char *progname,
			 const char *filename)
{
	bool_t unspecified = FALSE;
	char buf[MAXPATHLEN];
	FILE *file;
	size_t len;
	char **ptr;

	if (!filename) {
		filename = DEFAULT_SHELLS;
		unspecified = TRUE;
	}

	file = fopen(filename, "r");
	if (!file) {
		error_err(progname, "Can't open %s", filename);
		if (unspecified) {
			errno = 0;
			error_err(progname, "You must use -s or -S option");
		}
		exit(1);
	}

	for (len = 0; read_line(file, buf, sizeof (buf)) >= 0; len++);

	valid_shells = calloc(len + 1, sizeof (char *));
	if (!valid_shells) {
		error_err(progname, oom);
		exit(1);
	}

	rewind(file);

	for (ptr = valid_shells; TRUE; ) {
		char *str;

		if (read_line(file, buf, sizeof (buf)) < 0) {
			break;
		}

		str = trim_string(buf);
		if (*str == '\0') {
			continue;
		}

		*ptr++ = strdup(str);
	}

	fclose(file);

	if (*valid_shells == NULL) {
		errno = 0;
		error_err(progname, "%s doesn't contain any shells", filename);
		if (unspecified) {
			error_err(progname, "You must use -s or -S option");
		}
		exit(1);
	}
}

static void parse_shell_list(const char *progname,
			     char *list)
{
	size_t len;
	char *s, **vec, *name;

	for (len = 1, s = list; *s != '\0'; s++) {
		if (*s == ':') {
			len++;
		}
	}

	valid_shells = calloc(len + 1, sizeof (char *));
	if (!valid_shells) {
		error_err(progname, oom);
		exit(1);
	}

	for (vec = valid_shells, name = list, s = list; 1; s++) {
		if (*s == '\0') {
			name = trim_string(name);
			if (*name != '\0') {
				*vec++ = name;
			}

			break;
		}

		if (*s == ':') {
			*s = '\0';

			name = trim_string(name);
			if (*name != '\0') {
				*vec++ = name;
			}

			name = s + 1;
		}
	}
}

static void check_for_busybox(const char *progname)
{
	char mount_buf[PATH_MAX], *real_mount;

	real_mount = realpath(mount_cmd, mount_buf);
	if (!real_mount) {
		error_err(progname, "Can't get real path of %s", mount_cmd);
		exit(1);
	}

	if (strcmp(basename(real_mount), "busybox") == 0) {
		debug("%s is Busybox", mount_cmd);
		bind_opt = DEFAULT_BIND_OPT_BUSYBOX;
	}
}

/**
 * Finds a host from a config file and returns the associated password.
 *
 * Example config file entry:
 * 192.168.0.2  foobar
 *
 * @param data handler state
 * @param filename the config file path
 * @return the password if found, NULL if not
 */
static char *find_pwd(handler_t *data,
		      const char *filename)
{
	FILE *file;
	char buf[1024], *bufp, *pwd = NULL;
	ssize_t len;

	file = fopen(filename, "r");
	if (!file) {
		error("Can't open %s", filename);
		return NULL;
	}

	while (1) {
		len = read_line(file, buf, sizeof (buf));
		if (len < 0) {
			break;
		}
		if (len == 0 || isspace(buf[0])) {
			continue;
		}

		bufp = find_space(buf);
		if (*bufp == '\0') {
			continue;
		}
		*bufp++ = '\0';

		if (strcmp(buf, data->host) != 0) {
			continue;
		}

		bufp = skip_spaces(bufp);
		if (*bufp == '\0') {
			continue;
		}
		*find_space(bufp) = '\0';

		pwd = strdup(bufp);
		if (!pwd) {
			send_error(data, oom);
		}
		break;
	}

	fclose(file);

	return pwd;
}

/**
 * Recursively creates all needed directories (with MKDIR_PERMS).
 */
static int mkdirs(const char *path)
{
	char *parent, *p;
	int rc = -1;

	if (mkdir(path, MKDIR_PERMS) == 0 || errno == EEXIST) {
		return 0;
	}

	if (errno != ENOENT) {
		return -1;
	}

	parent = strdup(path);
	if (!parent) {
		return -1;
	}

	p = strrchr(parent, '/');
	if (p && parent != p) {
		*p = '\0';
		rc = 0;
	}

	if (rc == 0) {
		rc = mkdirs(parent);
	}

	free(parent);

	if (rc == 0) {
		rc = mkdir(path, MKDIR_PERMS);
	}

	return rc;
}

/**
 * Frees all resources associated with a mount entry.
 */
static void mnt_free(mount_t *m)
{
	if (m->info.opts) {
		free(m->info.opts);
	}

	if (m->info.device) {
		free(m->info.device);
	}

	if (m->info.point) {
		free(m->info.point);
	}

	free(m);
}

/**
 * Finds an entry from the MOUNTS list.
 * @param point the mount point of the entry
 */
static mount_t *mnt_list_find(const char *point)
{
	mount_t *mnt;

	for (mnt = all_mounts.head; mnt; mnt = mnt->next) {
		if (strcmp(mnt->info.point, point) == 0) {
			break;
		}
	}

	return mnt;
}

/**
 * Adds an entry to the MOUNTS list. The entries are kept in ascending order
 * based on their mount point strings.
 */
static void mnt_list_add(mount_t *mnt)
{
	mount_t *prev, *next;

	prev = NULL;
	next = all_mounts.head;
	while (next) {
		if (strcmp(next->info.point, mnt->info.point) > 0) {
			break;
		}

		prev = next;
		next = next->next;
	}

	mnt->prev = prev;
	mnt->next = next;

	if (prev) {
		prev->next = mnt;
	} else {
		all_mounts.head = mnt;
	}

	if (next) {
		next->prev = mnt;
	} else {
		all_mounts.tail = mnt;
	}
}

/**
 * Removes and frees an entry from the MOUNTS list.
 */
static void mnt_list_del(mount_t *mnt)
{
	if (mnt->prev) {
		mnt->prev->next = mnt->next;
	} else {
		all_mounts.head = mnt->next;
	}

	if (mnt->next) {
		mnt->next->prev = mnt->prev;
	} else {
		all_mounts.tail = mnt->prev;
	}

	mnt_free(mnt);
}

/**
 * Executes mount_cmd.
 */
static int do_mount(handler_t *data,
		    const mount_info_t *mi)
{
	char *argv[8];

	memset(argv, 0, sizeof (argv));

	argv[0] = mount_cmd;
	argv[1] = mi->device;
	argv[2] = mi->point;

	switch (mi->type) {
	case MTYPE_NFS:
		argv[3] = "-t";
		argv[4] = "nfs";
		argv[5] = mi->opts ? "-o" : NULL;
		argv[6] = mi->opts;

		if (mi->opts) {
			debug("Executing: %s %s %s -t nfs -o %s",
			      argv[0], argv[1], argv[2], argv[6]);
		} else {
			debug("Executing: %s %s %s -t nfs",
			      argv[0], argv[1], argv[2]);
		}

		break;

	case MTYPE_BIND:
		split_string(bind_opt,
			     &argv[3], &argv[4], &argv[5], &argv[6], NULL);

		debug("Executing: %s %s %s %s %s %s %s",
		      argv[0], argv[1], argv[2],
		      argv[3] ? argv[3] : "",
		      argv[4] ? argv[4] : "",
		      argv[5] ? argv[5] : "",
		      argv[6] ? argv[6] : "");

		break;

	default:
		errno = EINVAL;
		return -1;
	}

	return execute(data, argv);
}

static int do_unmount(handler_t *data,
		      char *point)
{
	char *argv[] = { umount_cmd, point, NULL };

	debug("Executing: %s %s", argv[0], argv[1]);

	return execute(data, argv);
}

static mount_t *mnt_create(handler_t *data,
			   mount_info_t *mi)
{
	mount_t *mnt;

	mnt = mnt_list_find(mi->point);
	if (mnt) {
		return mnt;
	}

	mnt = calloc(1, sizeof (mount_t));
	if (!mnt) {
		send_error(data, oom);
		return NULL;
	}

	mntinfo_copy(&mnt->info, mi);

	mnt_list_add(mnt);

	return mnt;
}

/**
 * Checks /etc/mtab if mount_info_t is already mounted.
 * @param data handler state
 * @return -1 on error, 0 otherwise
 */
static int is_mounted(handler_t *data,
		      const char *point)
{
	int mounted = FALSE;
	FILE *file;
	char buf[1024], buf1[PATH_MAX], buf2[PATH_MAX], *point1;

	point1 = realpath(point, buf1);
	if (!point1) {
		return FALSE;
	}

	file = fopen(PROC_MOUNTS, "r");
	if (!file) {
		send_error(data, "Can't open " PROC_MOUNTS);
		return -1;
	}

	while (1) {
		char *device, *point2;

		if (read_line(file, buf, sizeof (buf)) < 0) {
			break;
		}

		split_string(buf, &device, &point2, NULL);
		if (!device || !point2) {
			continue;
		}

		if (strncmp(point2, data->root, strlen(data->root)) != 0) {
			continue;
		}

		point2 = realpath(point2, buf2);
		if (!point2) {
			error("Can't get realpath of %s", point2);
			continue;
		}

		if (strcmp(point1, point2) == 0) {
			mounted = TRUE;
			break;
		}
	}

	fclose(file);

	debug(mounted ? "%s is mounted" : "%s is not mounted", point);

	return mounted;
}

/**
 * Mounts a filesystem if not already mounted.
 */
static mount_t *add_mount(handler_t *data,
			  mount_info_t *mi,
			  struct passwd *pass)
{
	int mounted;
	mount_t *mnt;

	mounted = is_mounted(data, mi->point);
	if (mounted < 0) {
		return NULL;
	}

	mnt = mnt_create(data, mi);
	if (!mnt) {
		return NULL;
	}

	if (!mounted) {
		debug("Creating directory %s as %s",
		      mnt->info.point, pass->pw_name);

		if (seteuid(pass->pw_uid) < 0) {
			send_error(data, "Can't change effective uid to %d",
				   pass->pw_uid);
			goto _restore;
		}

		if (mkdirs(mnt->info.point) < 0) {
			send_error(data, "Can't create mount point: %s",
				   mnt->info.point);
			goto _restore;
		}

		if (seteuid(0) < 0) {
			send_error(data, "Can't restore effective root uid");
			return NULL;
		}

		if (do_mount(data, &mnt->info) < 0) {
			return NULL;
		}
	}

	mnt->usage++;

	return mnt;

_restore:
	if (seteuid(0) < 0) {
		send_error(data, "Can't restore effective root uid");
	}
	return NULL;
}

/**
 * Unmounts a filesystem.
 * @return -1 on error, 0 otherwise
 */
static int remove_mount(handler_t *data,
			const mount_info_t *mi)
{
	int mounted;
	mount_t *mnt;

	mounted = is_mounted(data, mi->point);
	if (mounted < 0) {
		return -1;
	}

	mnt = mnt_list_find(mi->point);
	if (mnt) {
		if (mnt->usage == 0) {
			mnt_list_del(mnt);
		} else {
			send_error(data, "Warning: %s usage is %d",
				   mnt->info.point, mnt->usage);
		}
	}

	if (mounted && do_unmount(data, mi->point) < 0) {
		return -1;
	}

	return 0;
}

/**
 * Decrements the usage count of a mount and possibly sets its expiration time.
 */
static void release_mount(mount_t *mnt)
{
	mnt->usage--;

	debug("Releasing mount %s (%d users remain)",
	      mnt->info.point, mnt->usage);

	if (mnt->usage <= 0) {
		if (mount_expiration >= 0) {
			mnt->expiration = time(NULL) + mount_expiration;
		}
	}
}

/**
 * Steps through the MOUNTS list and unmounts all unused and expired entries.
 */
static void expire_mounts(void)
{
	mount_t *prev, *curr;
	time_t now;

	debug("Checking for expired mounts");

	now = time(NULL);

	prev = NULL;
	curr = all_mounts.tail;
	while (curr) {
		prev = curr->prev;

		if (curr->usage <= 0 &&
		    curr->expiration <= now) {
			debug("Unmounting %s", curr->info.point);

			if (curr->next) {
				curr->next->prev = curr->prev;
			} else {
				all_mounts.tail = curr->prev;
			}

			if (curr->prev) {
				curr->prev->next = curr->next;
			} else {
				all_mounts.head = curr->next;
			}

			do_unmount(NULL, curr->info.point);
			mnt_free(curr);
		}

		curr = prev;
	}
}

/**
 * Unmounts all entries in the MOUNTS list.
 */
static void unmount_all(void)
{
	mount_t *prev, *mnt;

	mnt = all_mounts.tail;
	while (mnt) {
		prev = mnt->prev;

		debug("Unmounting %s", mnt->info.point);

		do_unmount(NULL, mnt->info.point);
		mnt_free(mnt);

		mnt = prev;
	}

	all_mounts.head = NULL;
	all_mounts.tail = NULL;
}

/**
 * Unmounts all filesystems listed in the command info's mount info vector.
 * @return 0 or -1 on error
 */
static int unmount_infos(handler_t *data)
{
	mount_info_t **mip;
	int cnt, rc = 0;

	debug("Unmounting filesystems");

	/* Count the options and go to end of the vector. */
	for (cnt = 0, mip = data->cmd.mounts; *mip; ++cnt, ++mip);

	/* Try to unmount filesystems in reverse order. */
	for (--mip; cnt-- > 0; --mip) {
		if (remove_mount(data, *mip) < 0) {
			rc = -1;
		}
	}

	return rc;
}

/**
 * Gets the amount of data we can read from a file (which may be a tty).
 */
static ssize_t get_data_length(handler_t *data,
			       int fd,
			       size_t max)
{
	ssize_t len;

	len = max;

	if (isatty(fd)) {
		if (ioctl(fd, FIONREAD, &len) < 0) {
			send_error(data, "Can't check pty for available data");
			return -1;
		}

		if (len < 0) {
			send_error(data, "ioctl() gave invalid read length");
			return -1;
		}

		if (len > max) {
			len = max;
		}
	}

	return len;
}

/**
 * Reads some data from a fd and writes it into the socket (sd) in a packet.
 * The fd will be set to -1 at EOF.
 * @param data handler state
 * @param out describes the output
 */
static void send_data(handler_t *data,
		      output_desc_t *out)
{
	ssize_t len;

	len = get_data_length(data, out->fd, out->req);
	if (len < 0) {
		goto _error;
	}

	if (len == 0) {
		/* Nothing to do. */
		return;
	}

	len = read_ni(out->fd, data->tmp_buf, len);
	if (len < 0) {
		send_error(data, "Can't read output of the process");
		goto _error;
	}

	if (len) {
		debug("Sending %s DATA packet (%d bytes)",
		      output_desc_is_stdout(out) ? "OUT" : "ERR", len);

		if (write_packet(data->sd, out->data_type,
				 data->tmp_buf, len) < 0) {
			error("Can't write packet to socket");
			goto _error;
		}

		out->req -= len;
	} else {
		debug(output_desc_is_stdout(out) ?
		      "Stdout hit EOF" : "Stderr hit EOF");

		out->fd = -1;
		out->req = 0;
	}

	return;

_error:
	data->error = TRUE;
}

/**
 * Writes an RC or AUTH OK packet.
 */
static int send_int16(handler_t *data,
		      ptype_t ptype,
		      int16_t val)
{
	val = htons(val);

	if (write_packet(data->sd, ptype, &val, sizeof (val)) < 0) {
		error("Can't write int16 packet to socket");
		return -1;
	}

	return 0;
}

/**
 * Sends a REQuest for more DATA.
 */
static void send_request(handler_t *data,
			 input_desc_t *in)
{
	/* Already waiting? */
	if (in->wait) {
		return;
	}

	debug("Sending IN REQ packet");

	if (write_packet(data->sd, in->req_type, NULL, 0) < 0) {
		error("Can't write packet to socket");
		goto _error;
	}

	in->wait = TRUE;
	return;

_error:
	data->error = TRUE;
}

/**
 * Writes (some of) in->buf to in->fd.
 */
static void write_buffer(handler_t *data,
			 input_desc_t *in)
{
	debug("Writing buffer to stdin");

	if (buf_write_out(&in->buf, &in->fd) < 0) {
		debug("Stdin is closed (%s)", strerror(errno));
	}

	if (buf_is_empty(&in->buf)) {
		in->wait = FALSE;
	}
}

/**
 * Reads bytes from socket to buf_in.
 * @param data handler state
 * @param len the maximum number of bytes to be read
 */
static void receive_stream(handler_t *data,
			   size_t len)
{
	if (len > 0) {
		debug("Receiving IN DATA packet (%d bytes)", len);

		if (buf_read_in(&data->in.buf, data->sd, len) < 0) {
			send_error(data,"Can't read IN DATA packet to buffer");
			goto _error;
		}
	} else {
		debug("Receiving IN DATA packet: EOF");

		buf_set_eof(&data->in.buf);
	}

	return;

_error:
	data->error = TRUE;
}

/**
 * Read an IN DATA or OUT/ERR REQ packet from socket.
 * @return 1 if EOF from client, 0 otherwise
 */
static int receive_packet(handler_t *data)
{
	phead_t head;

	if (read_phead(data->sd, &head) < 0) {
		if (errno == 0) {
			debug("EOF from client");
			return 1;
		}

		error("Can't read packet header from socket");
		goto _error;
	}

	switch (head.type) {
	case PTYPE_IN_DATA:
		receive_stream(data, head.size);
		return 0;

	case PTYPE_OUT_REQ:
		debug("Receiving OUT REQ packet");

		if (head.size) {
			send_error(data, "OUT REQ packet size was non-zero");
			goto _error;
		}

		data->out.req = BUFFER_SIZE;
		return 0;

	case PTYPE_ERR_REQ:
		debug("Receiving ERR REQ packet");

		if (head.size) {
			send_error(data, "ERR REQ packet size was non-zero");
			goto _error;
		}

		data->err.req = BUFFER_SIZE;
		return 0;
	}

	send_error(data, "Packet has invalid type: %d", head.type);

_error:
	data->error = TRUE;
	return 0;
}

/**
 * Prints information on an exited process (pid) to debug log.
 */
static void print_status(int pid,
			 int status)
{
	int sig;

	if (WIFEXITED(status)) {
		debug("Process %d returned: %d", pid, WEXITSTATUS(status));
	} else if (WIFSIGNALED(status)) {
		sig = WTERMSIG(status);
		debug("Process %d terminated by signal: %s (%d)",
		      pid, strsignal(sig), sig);
	} else if (WIFSTOPPED(status)) {
		sig = WSTOPSIG(status);
		debug("Process %d stopped by signal: %s (%d)",
		      pid, strsignal(sig), sig);
	} else {
		error("Invalid status %d for pid %d", status, pid);
	}
}

#define print_debug_bool(m, b)  print_debug(m, (b) ? "yes" : "no")
#define debug_bool(m, b)        if (debug_file) print_debug_bool(m, b)

#define debug_status() \
	if (debug_file) { \
		print_debug_bool("Process alive  = %s", alive); \
		print_debug_bool("Stdin open     = %s", data->in.fd >= 0); \
		print_debug_bool("Stdin waiting  = %s", data->in.wait); \
		print_debug     ("Stdin buffered = %d bytes%s", \
				 buf_size(&data->in.buf), \
				 data->in.buf.eof ? " (EOF set)" : ""); \
		print_debug_bool("Stdout open    = %s", data->out.fd >= 0); \
		print_debug     ("Stdout request = %d bytes", data->out.req); \
		print_debug_bool("Stderr open    = %s", data->err.fd >= 0); \
		print_debug     ("Stderr request = %d bytes", data->err.req); \
		print_debug_bool("Error          = %s", data->error); \
	}

#define debug_select(msg) \
	if (debug_file) { \
		print_debug(msg " [%3s %2s %3s %3s][%2s]", \
			    ISSET(data->sd,      &readfds) ? "NET" : "", \
			    ISSET(data->in.fd,   &readfds) ? "IN"  : "", \
			    ISSET(data->out.fd,  &readfds) ? "OUT" : "", \
			    ISSET(data->err.fd,  &readfds) ? "ERR" : "", \
			    ISSET(data->in.fd,  &writefds) ? "IN"  : ""); \
	}

/**
 * Manages the streams of the command executor (= the child-child process).
 * @param data handler state
 * @param pid of the child process
 * @return the return code (-1 on error)
 */
static int16_t handler_manage(handler_t *data,
			      pid_t pid)
{
	fd_set readfds, writefds;
	int maxfd1, status;
	bool_t alive = TRUE;

	debug("Managing process %d", pid);

	maxfd1 = MAX(data->sd, data->in.fd);
	maxfd1 = MAX(maxfd1, data->out.fd);
	maxfd1 = MAX(maxfd1, data->err.fd);
	maxfd1++;

	if (!buf_init(&data->in.buf)) {
		send_error(data, oom);
		goto _kill;
	}

	while (1) {
		int count, val;

		debug_status();

		if (data->error) {
			debug("Ending loop due to error");
			goto _kill;
		}

		/* Non-TTY mode: Check if we should exit. */
		if (!data->cmd.is_tty &&
		    data->in.fd < 0 &&
		    data->out.fd < 0 &&
		    data->err.fd < 0) {
			debug("Ending loop due to closed stdin, stdout"
			      " and stderr descriptors");
			break;
		}

		FD_ZERO(&readfds);
		FD_ZERO(&writefds);

		FD_SET(data->sd, &readfds);

		if (data->out.fd >= 0 && data->out.req > 0) {
			FD_SET(data->out.fd, &readfds);
		}

		if (!data->cmd.is_tty &&
		    data->err.fd >= 0 && data->err.req > 0) {
			FD_SET(data->err.fd, &readfds);
		}

		if (data->in.fd >= 0) {
			if (!data->cmd.is_tty) {
				/* We want select to wake up if stdin hits EOF
				   (child exited but we haven't noticed yet) */
				FD_SET(data->in.fd, &readfds);
			}

			if (!buf_is_empty(&data->in.buf) || !data->in.wait) {
				FD_SET(data->in.fd, &writefds);
			}
		}

		debug_select("Selecting");

		count = select(maxfd1, &readfds, &writefds, NULL, NULL);
		if (count < 0) {
			if (errno == EINTR) {
				debug("Select interrupted");
			} else {
				error("Select failed");
				debug("Ending loop due to failed select");
				goto _kill;
			}
		} else if (count > 0) {
			debug_select("Selected ");

			/* TTY mode: Check if we should exit. */
			if (data->cmd.is_tty &&
			    !alive &&
			    ISSET(data->out.fd, &readfds) &&
			    get_data_length(data, data->out.fd, 1) <= 0) {
				debug("Ending loop due to dead process and"
				      " empty stdout buffer");
				break;
			}

			if (ISSET(data->sd, &readfds)) {
				if (receive_packet(data) == 1) {
					/* EOF from client */

					close(data->sd);
					data->sd = -1;

					debug("Ending loop due to closed"
					      " socket");
					goto _kill;
				}
			}

			if (ISSET(data->out.fd, &readfds)) {
				send_data(data, &data->out);
			}

			if (ISSET(data->err.fd, &readfds)) {
				send_data(data, &data->err);
			}

			if (ISSET(data->in.fd, &writefds)) {
				if (buf_is_empty(&data->in.buf)) {
					/* Request data to buffer. */
					send_request(data, &data->in);
				} else {
					/* Flush buffer. */
					write_buffer(data, &data->in);
				}
			}

			/* Stdin at EOF? */
			if (!data->cmd.is_tty &&
			    ISSET(data->in.fd, &readfds)) {
				data->in.fd = -1;
			}
		}

		/* Collect late children. */

		val = waitpid(-1, &status, WNOHANG);
		if (val < 0 && errno != ECHILD) {
			send_error(data, "Can't wait for children");
			return -1;
		}

		if (val > 0) {
			if (val == pid) {
				alive = FALSE;
			}

			print_status(val, status);
		}
	}

	if (alive) {
		if (waitpid(pid, &status, 0) < 0) {
			send_error(data, "Can't wait for child %d", pid);
			return -1;
		}

		print_status(pid, status);
	}

	if (WIFEXITED(status)) {
		return WEXITSTATUS(status);
	}
	return -1;

_kill:
	debug("Sending SIGTERM to command process %d", pid);
	kill(pid, SIGTERM);
	return -1;
}

/**
 * @return TRUE/FALSE or -1 on error
 */
static int authenticate(handler_t *data,
			struct passwd *pass)
{
	char *path, *pwd;
	bool_t ok = FALSE;

	if (!validate_shell(data, pass->pw_shell)) {
		return FALSE;
	}

	if (!pass->pw_dir || *pass->pw_dir == '\0') {
		send_error(data, "Invalid home directory");
		return -1;
	}
	if (*pass->pw_dir != '/') {
		send_error(data, "Invalid home directory: %s", pass->pw_dir);
		return -1;
	}

	path = malloc(strlen(pass->pw_dir) + strlen("/" CONFIG_NAME) + 1);
	if (!path) {
		send_error(data, oom);
		return FALSE;
	}
	strcpy(path, pass->pw_dir);
	strcat(path, "/" CONFIG_NAME);

	debug("Searching for password in %s", path);

	pwd = find_pwd(data, path);

	if (!pwd) {
		debug("Address/password not found in %s", path);
		goto _ret;
	}

	ok = (strcmp(pwd, data->auth.pwd) == 0);

	free(pwd);

	if (!ok) {
		warn("Passwords don't match");
	}

_ret:
	free(path);

	return ok;
}

/**
 * More or less like forkpty.
 * @return pid of the child process (0 for child) or -1 on error
 */
static pid_t fork_pty(handler_t *data,
		      int *ptyfd)
{
	int master, slave;
	pid_t pid;
	struct winsize ws;

	ws.ws_row = data->cmd.tty_row;
	ws.ws_col = data->cmd.tty_col;
	ws.ws_xpixel = data->cmd.tty_xpixel;
	ws.ws_ypixel = data->cmd.tty_ypixel;

	if (openpty(&master, &slave, NULL, NULL, &ws) < 0) {
		send_error(data, "Can't open a pseudo-tty");
		return -1;
	}

	pid = fork();
	if (pid < 0) {
		send_error(data, "Can't fork");
		return -1;
	}

	if (pid == 0) {
		/* Child */

		set_debug_name("COMMAND");

		close(master);
		if (login_tty(slave)) {
			send_error(data, "Can't login to a pseudo-tty");
			exit(1);
		}
	} else {
		/* Parent */

		close(slave);
		*ptyfd = master;
	}

	return pid;
}

/**
 * More or less like fork, but the stdin/stdout/stderr of the child process
 * are set to point to sockets.
 * @param data handler state
 * @param infd a place to write the parent-end of the child's stdin socketpair
 * @param outfd a place to write the parent-end of the child's stdin socketpair
 * @param errfd a place to write the parent-end of the child's stdin socketpair
 * @return pid of the child process (0 for child) or -1 on error
 */
static pid_t fork_sockets(handler_t *data,
			  int *infd,
			  int *outfd,
			  int *errfd)
{
	int insd[2], outsd[2], errsd[2];
	pid_t pid;

	if (socketpair(AF_UNIX, SOCK_STREAM, 0, insd) < 0 ||
	    socketpair(AF_UNIX, SOCK_STREAM, 0, outsd) < 0 ||
	    socketpair(AF_UNIX, SOCK_STREAM, 0, errsd) < 0) {
		send_error(data, "Can't create socket pairs");
		return -1;
	}

	pid = fork();
	if (pid < 0) {
		send_error(data, "Can't fork");
		return -1;
	}

	if (pid == 0) {
		/* Child */

		set_debug_name("COMMAND");

		close(insd[0]);
		close(outsd[0]);
		close(errsd[0]);

		if (dup2(insd[1], STDIN_FILENO) != STDIN_FILENO) {
			send_error(data, "Can't duplicate socket as stdin");
			exit(1);
		}

		if (dup2(outsd[1], STDOUT_FILENO) != STDOUT_FILENO) {
			send_error(data, "Can't duplicate socket as stdout");
			exit(1);
		}

		if (dup2(errsd[1], STDERR_FILENO) != STDERR_FILENO) {
			send_error(data, "Can't duplicate socket as stderr");
			exit(1);
		}
	} else {
		/* Parent */

		if (set_nonblocking(insd[0], TRUE)) {
			send_error(data, "Can't make socket non-blocking");
			return -1;
		}

		*infd = insd[0];
		*outfd = outsd[0];
		*errfd = errsd[0];
	}

	close(insd[1]);
	close(outsd[1]);
	close(errsd[1]);

	return pid;
}

/**
 * Read "SBRSH_RLIMIT"-fields from ENVIRONment and set them in place.
 * @return -1 on error, 0 otherwise
 */
static int set_rlimits(handler_t *data)
{
	static struct { const char *key; int resource; } info[] = {
		{ ENV_RLIMIT_PREFIX "CPU",	RLIMIT_CPU	},
		{ ENV_RLIMIT_PREFIX "FSIZE",	RLIMIT_FSIZE	},
		{ ENV_RLIMIT_PREFIX "DATA",	RLIMIT_DATA	},
		{ ENV_RLIMIT_PREFIX "STACK",	RLIMIT_STACK	},
		{ ENV_RLIMIT_PREFIX "CORE",	RLIMIT_CORE	},
		{ ENV_RLIMIT_PREFIX "RSS",	RLIMIT_RSS	},
		{ ENV_RLIMIT_PREFIX "NPROC",	RLIMIT_NPROC	},
		{ ENV_RLIMIT_PREFIX "NOFILE",	RLIMIT_NOFILE	},
		{ ENV_RLIMIT_PREFIX "MEMLOCK",	RLIMIT_MEMLOCK	},
		{ ENV_RLIMIT_PREFIX "AS",	RLIMIT_AS	},
		{ NULL,				0		}
	};

	const int prefixlen = strlen(ENV_RLIMIT_PREFIX);
	char *resname;
	int i;
	char *str;
	struct rlimit lim;

	for (i = 0; info[i].key; i++) {
		resname = (char *) info[i].key + prefixlen;

		str = getenv(info[i].key);
		if (!str) {
			continue;
		}

		str = skip_spaces(str);
		if (strlen(str) == 0) {
			continue;
		}

		debug("Setting %s resource to %s", resname, str);

		if (getrlimit(info[i].resource, &lim) < 0) {
			send_error(data, "Can't get %s resource limit",
				   resname);
			return -1;
		}

		lim.rlim_cur = RLIM_INFINITY;

		if (strncmp(str, ENV_RLIMIT_UNLIMITED,
			    strlen(ENV_RLIMIT_UNLIMITED)) != 0) {
			lim.rlim_cur = atol(str);

			if (lim.rlim_cur == 0 && strcmp(str, "0") != 0) {
				send_error(data, "Invalid %s resource limit "
					   "value: %s", resname, str);
				return -1;
			}
		}

		if (setrlimit(info[i].resource, &lim) < 0) {
			send_error(data, "Can't set %s resource limit to %d"
				   " while maximum is %d", resname,
				   lim.rlim_cur, lim.rlim_max);
			return -1;
		}
	}

	return 0;
}

static void modify_environment_vector(char **src,
				      char *entry,
				      char *newentry)
{
	char **dest, *key, *value;
	size_t len = 0;

	key = newentry;

	/* Length of the key (including '='). */
	value = strchr(key, '=');
	if (value) {
		value++;
		len = value - key;
	}

	/* Copy the vector onto itself skipping ENTRY and the key that
	 * equals KEY. */
	for (dest = src; *src; src++) {
		if (*src != entry && strncmp(*src, key, len) != 0) {
			*dest++ = *src;
		}
	}

	/* Put NEWENTRY at the end of the vector if
	 * its value is not "(UNSET)". */
	if (value && strcmp(value, ENV_RENAME_UNSET) != 0) {
		*dest++ = newentry;
	}

	/* Nul-terminate the vector (which has a nul-terminator at its
	 * old end). */
	while (*dest) {
		*dest++ = NULL;
	}
}

static void set_environment(char **env)
{
	char **p, *entry;
	const size_t prelen = strlen(ENV_RENAME_PREFIX);

	/* Strip "SBRSH_ENV_" prefices. (This is a bit dirty, but doesn't
	 * matter since we'll exec after this.) */

	while (1) {
		entry = NULL;

		/* Find first SBRSH_ENV-entry. */
		for (p = env; *p; p++) {
			entry = *p;
			if (strncmp(entry, ENV_RENAME_PREFIX, prelen) == 0) {
				break;
			} else {
				entry = NULL;
			}
		}

		/* Quit if there are no entries left. */
		if (!entry) {
			break;
		}

		debug("Converting \"%s\"", entry);
		modify_environment_vector(env, entry, entry + prelen);
	}

	/* Hit it! */
	environ = env;
}

/**
 * Copies a string and adds a "/" prefix if it doesn't have one.
 * abort()s if ENOMEM. This function is evil!
 */
static char *add_root_prefix(handler_t *data,
			     char *orig)
{
	char *dir;

	if (orig[0] == '/') {
		return orig;
	}

	dir = malloc(1 + strlen(orig) + 1);
	if (!dir) {
		send_error(data, oom);
		exit(1);
	}

	strcpy(dir, "/");
	strcat(dir, orig);

	return dir;
}

/**
 * Executes /The/ command. Never returns. Dies with abort() so that cleanup()
 * is not called.
 */
static void execute_command(handler_t *data)
{
	char *dir;
	struct passwd *pass;
	char **argv;

	/* Change root directory. */

	dir = add_root_prefix(data, data->root);
	debug("Changing root directory to %s", dir);
	if (chroot(dir) < 0) {
		send_error(data, "Can't change root directory to: %s", dir);
		goto _error;
	}

	chdir("/");

	/* Get uid/gid/shell under the new root. */

	debug("Getting passwd struct of user %s", data->auth.user);
	pass = getpwnam(data->auth.user);
	if (!pass) {
		send_error(data, "getpwnam: Can't get information "
			   "about user: %s [after chroot]",data->auth.user);
		goto _error;
	}
	
	/* Change gid and uid (in that order because we need to be root). */

	debug("Changing gid to %d", pass->pw_gid);
	if (setgid(pass->pw_gid) < 0) {
		send_error(data, "Can't change group id to %d [after chroot]",
			   pass->pw_gid);
		goto _error;
	}

	debug("Changing uid to %d", pass->pw_uid);
	if (setuid(pass->pw_uid) < 0) {
		send_error(data, "Can't change user id to %d [after chroot]",
			   pass->pw_uid);
		goto _error;
	}

	/* Change environment. */

	set_environment(data->cmd.env);

	if (data->fakerootkey &&
	    putenv(data->fakerootkey) < 0) {
		send_error(data, "Can't put %s to environment",
			   data->fakerootkey);
		goto _error;
	}

	/* Read resource limits from environment. */

	if (set_rlimits(data) < 0) {
		goto _error;
	}

	/* Pass daemon version to command. */

	if (setenv(ENV_VERSION, VERSION, TRUE) < 0) {
		send_error(data, oom);
		goto _error;
	}

	/* Command and arguments. */

	argv = data->cmd.argv;

	if (argv[0] == NULL) {
		argv = calloc(2, sizeof (char *));
		if (!argv) {
			send_error(data, oom);
			goto _error;
		}

		argv[0] = pass->pw_shell;
		if (argv[0] == NULL) {
			send_error(data, "User %s has no shell",
				   data->auth.user);
			goto _error;
		}
	}

	/* Change current directory. */

	dir = add_root_prefix(data, data->cmd.curdir);
	debug("Changing current directory to %s", dir);
	if (chdir(dir) < 0) {
		send_error(data, "Can't change current directory to: %s "
			   "[after chroot]", dir);
		goto _error;
	}

	/* Do it! */

	debug_vector("Executing command:", argv);

	set_closeonexec(data->sd);
	execvp(argv[0], argv);
	send_error(data, "Can't execute command: %s", argv[0]);

_error:
	exit(1);
}

/**
 * Does the handshake & stuff for a client.
 * @return the mounts of this handler; NULL on error
 */
static mount_t **handler_startup(handler_t *data)
{
	struct passwd *pass = NULL;
	int ok;
	size_t mntcnt, i;
	mount_t **mounts = NULL;

	debug("Client IP address is %s", data->host);

	debug("Sending VERSION packet");

	if (send_version(data->sd) < 0) {
		error("Can't send protocol version packet");
		goto _error;
	}

	debug("Reading client's VERSION packet");

	data->client_version = get_version(data->sd);
	if (data->client_version < 0) {
		error("Can't read protocol version packet");
		goto _error;
	}

	if (data->client_version < 3) {
		errno = 0;
		send_error(data, "Client protocol version %d is too old",
			   data->client_version);
		goto _error;
	}
	if (data->client_version != PROTOCOL_VERSION) {
		warn("Client uses %s protocol version %d",
		     data->client_version < PROTOCOL_VERSION ?
		     "older" : "newer", data->client_version);
	} else {
		debug("Protocol versions match");
	}

	debug("Reading auth info struct from socket");

	if (read_auth(data->sd, &data->auth) < 0) {
		error("Can't read authentication info");
		goto _error;
	}

	debug("Getting passwd struct of user %s", data->auth.user);

	pass = getpwnam(data->auth.user);
	if (!pass) {
	        /* Check if user wants to create the user... */
	        if (!check_createuser(data->auth.user)) {
			send_error(data, "getpwnam: Can't get information "
				   "about user: %s [before chroot 1]",
				   data->auth.user);
			goto _error;
		}
		pass = getpwnam(data->auth.user);
		if (!pass) {	
			send_error(data, "getpwnam: Can't get information "
				   "about user: %s [before chroot]",
				   data->auth.user);
			goto _error;
		}
		/* Ok. Lets try to create auth file */
		create_auth(data,pass);
	}
	
	ok = authenticate(data, pass);
	if (ok < 0) {
		goto _error;
	}

	if (ok) {
		debug("Sending AUTH OK packet");

		if (send_empty(data, PTYPE_AUTH_OK) < 0) {
			error("Can't write AUTH OK packet to socket");
			goto _error;
		}
	} else {
		send_message(data, "Permission denied");
		goto _error;
	}

	debug("Reading command info struct from socket");

	if (read_cmd(data->sd, &data->cmd) < 0) {
		error("Can't read command info");
		goto _error;
	}

	/* Build path to new root. */

	data->root = malloc(strlen(pass->pw_dir) + 1 +
			    strlen(data->host) + 1 +
			    strlen(data->cmd.target) + 1);
	if (!data->root) {
		send_error(data, oom);
		goto _error;
	}

	strcpy(data->root, pass->pw_dir);
	strcat(data->root, "/");
	strcat(data->root, data->host);
	strcat(data->root, "-");
	strcat(data->root, data->cmd.target);

	/* Prepend root path to mount points. */

	mntcnt = calc_vec_len((void **) data->cmd.mounts);

	for (i = 0; i < mntcnt; ++i) {
		mount_info_t *mi;
		char *str;

		mi = data->cmd.mounts[i];

		str = malloc(strlen(data->root) + strlen(mi->point) + 1);
		if (!str) {
			send_error(data, oom);
			goto _error;
		}

		strcpy(str, data->root);
		strcat(str, mi->point);

		free(mi->point);

		mi->point = str;
	}

	/* Should we just unmount all filesystems and return? */
	if (data->cmd.umount) {
		static mount_t *nul = NULL;

		debug("--umount-all requested");

		if (unmount_infos(data) < 0) {
			goto _error;
		}

		/* Return a dummy pointer (non-error). */
		return &nul;
	}

	/* Mount filesystems. */

	mntinfo_sort_vec(data->cmd.mounts);

	mounts = calloc(mntcnt + 1, sizeof (mount_t *));
	if (!mounts) {
		send_error(data, oom);
		goto _error;
	}

	for (i = 0; i < mntcnt; ++i) {
		mounts[i] = add_mount(data, data->cmd.mounts[i], pass);
		if (!mounts[i]) {
			errno = 0;
			send_error(data, "Can't mount to point: %s",
				   data->cmd.mounts[i]->point);
			goto _error;
		}
	}

	data->mounts = mounts;

	return mounts;

_error:
	if (mounts) {
		/* The contents aren't owned by mounts but must be released. */
		free_vec((void **) mounts, (free_func_t *) release_mount);
	}
	return NULL;
}

static void send_rc(handler_t *data,
		    int16_t rc)
{
	if (data->sd < 0) {
		debug("Not sending RC packet");
		return;
	}

	debug("Sending RC packet: %d", rc);

	/* Send the return code to the client. */
	if (send_int16(data, PTYPE_RC, rc) >= 0) {
		fd_set fds;

		/* Do something until the client goes away. */
		while (1) {
			FD_ZERO(&fds);
			FD_SET(data->sd, &fds);

			if (select(data->sd + 1, &fds, NULL, NULL, NULL) < 0) {
				if (errno == EINTR) {
					continue;
				} else {
					break;
				}
			}

			if (read(data->sd, data->tmp_buf, BUFFER_SIZE) <= 0) {
				break;
			}
		}
	}

	close(data->sd);
	data->sd = -1;
}

static void handler_handle(handler_t *data)
{
	int16_t rc = -1;
	pid_t relay_pid, pid;

	relay_pid = fakeroot_relay(data);
	if (relay_pid < 0) {
		goto _rc;
	}

	if (data->cmd.is_tty) {
		int fd;

		debug("Creating command process in a pty");

		pid = fork_pty(data, &fd);

		data->in.fd = fd;
		data->out.fd = fd;
		data->err.fd = -1;
	} else {
		debug("Creating command process without a pty");

		pid = fork_sockets(data, &data->in.fd,
				   &data->out.fd, &data->err.fd);
	}

	if (pid == 0) {
		/* Child: */

		execute_command(data);

		/* Not reached. */
	}

	/* Parent: */

	if (pid > 0) {
		rc = handler_manage(data, pid);
	}

	if (relay_pid > 0) {
		debug("Sending SIGTERM to relay process %d", relay_pid);
		kill(relay_pid, SIGTERM);
	}

_rc:
	if (rc < 0) {
		rc = INTERNAL_ERROR_CODE;
	}

	send_rc(data, rc);
}

static void release_mounts(mount_t **vec)
{
	while (*vec) {
		release_mount(*vec++);
	}
}

static handler_t *alloc_handler(void)
{
	handler_t *h;

	h = calloc(1, sizeof (handler_t));

	if (h) {
		h->sd = -1;

		h->in.req_type = PTYPE_IN_REQ;
		h->out.data_type = PTYPE_OUT_DATA;
		h->err.data_type = PTYPE_ERR_DATA;

		h->error = 0;
	}

	return h;
}

/**
 * Frees all resources used by a handler.
 */
static void free_handler(handler_t *data)
{
	if (data->fakerootkey) {
		free(data->fakerootkey);
	}

	buf_free(&data->in.buf);

	if (data->root) {
		free(data->root);
	}

	free_cmd(&data->cmd);

	if (data->sd >= 0) {
		close(data->sd);
	}

	free(data);
}

static void pid_mounts_add(pid_t pid,
			   mount_t **vec)
{
	pid_mounts_t *pm, *node;

	/* Allocate */

	pm = calloc(1, sizeof (pid_mounts_t));
	if (!pm) {
		error(oom);
		return;
	}

	pm->pid = pid;
	pm->mounts = vec;

	/* Append */

	if (pid_mounts == NULL) {
		pid_mounts = pm;
	} else {
		for (node = pid_mounts; node->next; node = node->next);
		node->next = pm;
	}
}

static void pid_mounts_del(pid_t pid)
{
	pid_mounts_t *pm = NULL, *node;

	if (!pid_mounts) {
		return;
	}

	/* Find and remove */

	if (pid_mounts->pid == pid) {
		pm = pid_mounts;
		pid_mounts = pm->next;
	} else {
		for (node = pid_mounts; node->next; node = node->next) {
			if (node->next->pid == pid) {
				pm = node->next;
				node->next = pm->next;
				break;
			}
		}
	}

	if (!pm) {
		return;
	}

	debug("Found mounts for pid %d", pid);

	/* Deallocate */

	if (pm->mounts) {
		release_mounts(pm->mounts);
		free(pm->mounts);
	}

	free(pm);
}

static void accept_conn(int srvsd)
{
	struct sockaddr_in addr;
	socklen_t len = sizeof (addr);
	int clisd;
	handler_t *data;
	mount_t **mounts = NULL;
	pid_t pid;

	clisd = accept(srvsd, (struct sockaddr *) &addr, &len);
	if (clisd < 0) {
		error("Can't accept connection");
		return;
	}

	debug("New connection");

	data = alloc_handler();
	if (!data) {
		error(oom);
		close(clisd);
		return;
	}

	data->sd = clisd;

	/* Get the IP address. */
	if (getnameinfo((struct sockaddr *) &addr, len,
			data->host, sizeof (data->host),
			NULL, 0, NI_NUMERICHOST)) {
		error("Can't get client's IP address");
		goto _error;
	}

	mounts = handler_startup(data);
	if (!mounts) {
		goto _error;
	}

	if (data->cmd.umount) {
		/* We've done all we should for the client. */

		send_rc(data, 0);

		free_handler(data);
		return;
	}

	/* Start handler process. */
	pid = fork();
	if (pid < 0) {
		error("Can't fork");
		goto _error;
	}

	if (pid == 0) {
		/* Child: */

		set_debug_name("HANDLER");

		close(srvsd);

		handler_handle(data);

		free_handler(data);
		free(mounts);

		debug("Handler process exiting");
		exit(0);
	}

	/* Parent: */

	/* Map pid to mounts vector. */
	pid_mounts_add(pid, mounts);

	free_handler(data);
	return;

_error:
	send_rc(data, INTERNAL_ERROR_CODE);

	free_handler(data);
	if (mounts) {
		free(mounts);
	}
}

/**
 * Unmounts all filesystems that we've mounted and exits.
 */
static void clean_exit(int rc)
{
	unmount_all();

	debug("sbrshd exiting");
	exit(rc);
}

static void sig_dummy(int sig)
{
}

/*
 * Exits. If invoked on the DAEMON process, then it also unmounts everything.
 */
static void sig_exit(int sig)
{
	debug(strsignal(sig));

	if (getpid() == daemon_pid) {
		clean_exit(0);
	}
	exit(0);
}

static void sig_debug(int sig)
{
	int stored_errno = errno;

	debug(strsignal(sig));

	if (getpid() != daemon_pid) {
		goto _ret;
	}

	if (sig == SIGUSR1) {
		open_debug_log();
	} else if (sig == SIGUSR2) {
		close_debug_log();
	}

_ret:
	errno = stored_errno;
}

static char *get_absolute_path(const char *progname,
			       char *relpath)
{
	char *tmp1, *tmp2, *dir, *file, *abspath;

	tmp1 = relpath;
	tmp2 = strdup(relpath);
	if (!tmp2) {
		error_err(progname, oom);
		exit(1);
	}

	dir = dirname(tmp1);
	file = basename(tmp2);

	abspath = malloc(PATH_MAX + 1);
	if (!abspath) {
		error_err(progname, oom);
		exit(1);
	}

	abspath = realpath(dir, abspath);
	if (!abspath) {
		error_err(progname, "Can't get real path of %s", dir);
		exit(1);
	}

	if (strlen(abspath) + 1 + strlen(file) > PATH_MAX) {
		error_err(progname, "Path is too long: %s/%s", abspath, file);
		exit(1);
	}

	strcat(abspath, "/");
	strcat(abspath, file);

	free(tmp2);

	return abspath;
}

static void usage(char *progname)
{
	fprintf(stderr, "Usage: %s [-p|--port <port>]"
				 " [-d|--debug <path>]"
				 " [-m|--mount-bin <path>]"
				 " [-u|--umount-bin <path>]"
				 " [-t|--mount-tab <path>]"
				 " [-b|--bind-opt <options>]"
				 " [-e|--mount-expiration <minutes>|none]"
				 " [-s|--shells <path>]"
				 " [-S|--shell-list <paths>]\n"
			"       %s -v|--version\n"
			"       %s -h|--help\n",
		progname, progname, progname);

	exit(0);
}

/**
 * Reads options. Prints usage and exits when necessary.
 */
static void read_args(char *progname,
		      int argc,
		      char **argv)
{
	const char *const optstring = "hvp:d:m:u:t:b:e:s:S:";
	struct option longopts[] = {
		{ "help",             no_argument,       0, 'h' },
		{ "version",          no_argument,       0, 'v' },
		{ "port",             required_argument, 0, 'p' },
		{ "debug",            required_argument, 0, 'd' },
		{ "mount-bin",        required_argument, 0, 'm' },
		{ "umount-bin",       required_argument, 0, 'u' },
		{ "bind-opt",         required_argument, 0, 'b' },
		{ "mount-expiration", required_argument, 0, 'e' },
		{ "shells",           required_argument, 0, 's' },
		{ "shell-list",       required_argument, 0, 'S' },
		{ 0 }
	};

	char *debugname = NULL;
	char *exp_str = NULL;
	char *shells = NULL, *shell_list = NULL;
	bool_t default_bind_opt = TRUE;

	while (1) {
		int c;

		c = getopt_long(argc, argv, optstring, longopts, NULL);
		if (c < 0) {
			break;
		}

		switch (c) {
		case 'p':
			port = atoi(optarg);
			break;

		case 'd':
			debugname = optarg;
			break;

		case 'm':
			mount_cmd = optarg;
			break;

		case 'u':
			umount_cmd = optarg;
			break;

		case 'b':
			bind_opt = optarg;
			default_bind_opt = FALSE;
			break;

		case 'e':
			exp_str = optarg;
			break;

		case 's':
			shells = optarg;
			break;

		case 'S':
			shell_list = optarg;
			break;

		case 'v':
			fprintf(stderr,
				"Scratchbox Remote Shell daemon " VERSION "\n"
				"Protocol version %d\n"
				"Compiled at %s %s\n",
				PROTOCOL_VERSION, __DATE__, __TIME__);
			exit(0);

		case 'h':
		case '?':
		default:
			usage(progname);
		}
	}

	if (shells && shell_list) {
		error_err(progname, "Do not define --shells and --shell-list"
				    " at the same time");
		exit(1);
	}

	if (port <= 0 || port >= 65536) {
		error_err(progname, "Invalid port number: %d", port);
		exit(1);
	}

	if (exp_str) {
		if (strcmp(exp_str, MOUNT_EXPIRATION_NONE) == 0) {
			mount_expiration = -1;
		} else if (strcmp(exp_str, "0") == 0) {
			mount_expiration = 0;
		} else {
			int i = atoi(exp_str);
			if (i <= 0) {
				error_err(progname,
					"Invalid expiration time: %s minutes",
					exp_str);
				exit(1);
			}
			mount_expiration = i * 60;
		}
	}

	if (debugname) {
		/* We need the absolute path since we chdir to / */
		debug_filename = get_absolute_path(progname, debugname);
	}

	if (shell_list) {
		parse_shell_list(progname, shell_list);
	} else {
		parse_shells(progname, shells);
	}

	if (default_bind_opt) {
		check_for_busybox(progname);
	}
}

/**
 * Do stuff, close fds and direct in/out/err to /dev/null or debug log.
 * To make this function sufficiently odd, setsid() will not be called.
 * @param listenfd this descriptor won't be closed
 */
int daemonize(int listenfd)
{
	int debugfd, fd;

	chdir("/");
	umask(0);

	/* Don't close debug file */
	debugfd = debug_file ? fileno(debug_file) : -1;

	for (fd = getdtablesize(); fd-- > 0; ) {
		if (fd != listenfd && fd != debugfd) {
			close(fd);
		}
	}

	fd = open(NULL_FILE, O_RDWR);
	if (fd < 0) {
		error("Can't open " NULL_FILE);
		return -1;
	}

	assert(fd == STDIN_FILENO);

	if (debugfd >= 0) {
		fd = debugfd;
	}

	if (dup2(fd, STDOUT_FILENO) != STDOUT_FILENO) {
		error("Can't duplicate descriptor %d as stdout", fd);
		return -1;
	}

	if (dup2(fd, STDERR_FILENO) != STDERR_FILENO) {
		error("Can't duplicate descriptor %d as stderr", fd);
		return -1;
	}

	return 0;
}

/*
 * Startup and main loop.
 */
int main(int argc,
	 char **argv)
{
	char *progname;
	int srvsd;
	struct sockaddr_in srvaddr = { 0 };
	struct sigaction act_dummy, act_exit, act_debug;
	pid_t pid;
	unsigned int timeout = 0;
	struct timeval tv;

	/* Read config */

	progname = get_progname(argv[0]);
	openlog(progname, LOG_PID, LOG_DAEMON);
	read_args(progname, argc, argv);

	/* Create a socket, bind it and listen to it. */

	srvsd = socket(PF_INET, SOCK_STREAM, 0);
	if (srvsd < 0) {
		error_err(progname, "Can't create socket");
		return 1;
	}

	if (setsockopt_bool(srvsd, SOL_SOCKET, SO_REUSEADDR, TRUE) < 0) {
		error_err(progname, "Can't set socket option (SO_REUSEADDR)");
		return 1;
	}

	srvaddr.sin_family = AF_INET;
	srvaddr.sin_addr.s_addr = htonl(INADDR_ANY);
	srvaddr.sin_port = htons(port);

	if (bind(srvsd, (struct sockaddr *) &srvaddr, sizeof (srvaddr)) < 0) {
		error_err(progname, "Can't bind socket to port %d", port);
		return 1;
	}

	if (listen(srvsd, SOMAXCONN) < 0) {
		error_err(progname, "Can't listen with socket");
		return 1;
	}

	/* Daemonize */

	pid = fork();
	if (pid < 0) {
		error_err(progname, "Can't fork");
		return 1;
	}
	if (pid > 0) {
		printf("%d\n", pid);
		return 0;
	}

	set_debug_name("DAEMON");
	if (debug_filename) {
		open_debug_log();
	}

	setsid();
	if (daemonize(srvsd) < 0) {
		return 1;
	}

	/* Signal handlers */

	daemon_pid = getpid();

	act_dummy.sa_handler = sig_dummy;
	sigemptyset(&act_dummy.sa_mask);
	act_dummy.sa_flags = 0;

	sigaction(SIGINT, &act_dummy, NULL);
	sigaction(SIGHUP, &act_dummy, NULL);
	sigaction(SIGCHLD, &act_dummy, NULL);
	sigaction(SIGPIPE, &act_dummy, NULL);

	act_exit.sa_handler = sig_exit;
	sigemptyset(&act_exit.sa_mask);
	act_exit.sa_flags = SA_ONESHOT;

	sigaction(SIGTERM, &act_exit, NULL);

	act_debug.sa_handler = sig_debug;
	sigemptyset(&act_debug.sa_mask);
	act_debug.sa_flags = 0;

	sigaction(SIGUSR1, &act_debug, NULL);
	sigaction(SIGUSR2, &act_debug, NULL);

	/* The main loop */

	debug("sbrshd version " VERSION " (protocol version %d)",
	      PROTOCOL_VERSION);
	debug("Listening at port %d", port);
	debug_vector("Valid login shells:", valid_shells);

	/* Mount stuff */

	if (mount_expiration >= 0) {
		timeout = mount_expiration / MOUNT_EXPIRATION_FREQUENCY;

		tv.tv_sec = timeout;
		tv.tv_usec = 0;

		debug("Mounts expire after %d seconds", mount_expiration);
	} else {
		debug("Mounts never expire");
	}

	while (1) {
		fd_set fds;
		int count;

		debug("Waiting for connection");

		FD_ZERO(&fds);
		FD_SET(srvsd, &fds);

		count = select(srvsd + 1, &fds, NULL, NULL,
			       mount_expiration > 0 ? &tv : NULL);

		if (count < 0) {
			/* Select failed? */
			if (errno != EINTR) {
				error("Select failed");
				clean_exit(1);
			}

			/* We received a signal. */
			while (1) {
				int status;

				pid = waitpid(-1, &status, WNOHANG);
				if (pid <= 0) {
					break;
				}

				/* Release mounts if found. */
				print_status(pid, status);
				pid_mounts_del(pid);
			}
		} else if (count == 1) {
			/* We received a connection. */
			accept_conn(srvsd);
		}

		/* Expire mounts. */
		if (mount_expiration >= 0) {
			expire_mounts();

			tv.tv_sec = timeout;
			tv.tv_usec = 0;
		}
	}

	/* Never reached. */
	return 0;
}
