/**
  @file dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
    
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <stdio.h>

#include <glib.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif


#ifdef USE_BTCOND
# include <dbus/dbus.h>
# include <bt-dbus.h>
static char *used_dev = NULL;
#endif

#include "dbus.h"


#ifdef USE_BTCOND
static char *rfcomm_connect(DBusConnection *c,
                            const char *bda,
                            const char *svc,
                            gboolean auth,
                            gboolean encrypt,
                            const char *role) {
    char *rfcomm_dev = NULL;
    const char *r = role ? role : "any";
    gboolean ret;
    dbus_bool_t auto_dc = TRUE;
    DBusError err;
    DBusMessage *msg, *reply;

    msg = dbus_message_new_method_call(BTCOND_SERVICE,
                                       BTCOND_REQ_PATH,
                                       BTCOND_REQ_INTERFACE,
                                       BTCOND_RFCOMM_CONNECT_REQ);
    if (msg == NULL) {
        fprintf(stderr, "Out of memory during dbus_message_new_method_call()\n");
        exit(EXIT_FAILURE);
    }

    if (!dbus_message_append_args(msg,
                                  DBUS_TYPE_STRING, &bda,
                                  DBUS_TYPE_STRING, &svc,
                                  DBUS_TYPE_BOOLEAN, &auto_dc,
                                  DBUS_TYPE_BOOLEAN, &auth,
                                  DBUS_TYPE_BOOLEAN, &encrypt,
                                  DBUS_TYPE_STRING, &r,
                                  DBUS_TYPE_INVALID)) {
        fprintf(stderr, "Out of memory!\n");
        exit(EXIT_FAILURE);
    }

    dbus_error_init(&err);
    reply = dbus_connection_send_with_reply_and_block(c, msg, 1000 * 20, &err);
    dbus_message_unref(msg);
    if (dbus_error_is_set(&err)) {
        printf("btcond returned error: %s\n", err.name);
        dbus_error_free(&err);
        return NULL;
    }

    ret = dbus_message_get_args(reply, NULL,
                                DBUS_TYPE_STRING, &rfcomm_dev,
                                DBUS_TYPE_INVALID);
    dbus_message_unref(reply);
    if (ret == FALSE) {
        printf("get_args failed\n");
        return NULL;
    }

    return g_strdup(rfcomm_dev);
}
#endif

char *get_obex_dev(char *bda, gboolean ftp, gboolean auth, gboolean encrypt, char *role) {
#ifdef USE_BTCOND
    char *rfcomm_dev = NULL;
    DBusConnection *c;

    c = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
    if (c == NULL) {
        fprintf(stderr, "Unable to connect to system D-BUS\n");
        return NULL;
    }

    if (ftp) {
        rfcomm_dev = rfcomm_connect(c, bda, "NFTP", auth, encrypt, role); 
        if (rfcomm_dev == NULL)
            rfcomm_dev = rfcomm_connect(c, bda, "FTP", auth, encrypt, role); 
    }
    else
        rfcomm_dev = rfcomm_connect(c, bda, "OPP", auth, encrypt, role); 

    if (rfcomm_dev != NULL) {
        g_free(used_dev);
        used_dev = g_strdup(rfcomm_dev);
    }

    return rfcomm_dev;
#else
    return NULL;
#endif
}

void disconnect_obex_dev(void) {
#ifdef USE_BTCOND
    DBusError err;
    DBusConnection *c;
    DBusMessage *msg, *reply;

    if (used_dev == NULL)
        return;

    c = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
    if (c == NULL) {
        fprintf(stderr, "Unable to connect to system D-BUS\n");
        return;
    }

    msg = dbus_message_new_method_call(BTCOND_SERVICE,
                                       BTCOND_REQ_PATH,
                                       BTCOND_REQ_INTERFACE,
                                       BTCOND_RFCOMM_DISCONNECT_REQ);
    if (msg == NULL) {
        fprintf(stderr, "Out of memory during dbus_message_new_method_call()\n");
        exit(EXIT_FAILURE);
    }

    if (!dbus_message_append_args(msg,
                                  DBUS_TYPE_STRING, &used_dev,
                                  DBUS_TYPE_INVALID)) {
        fprintf(stderr, "Out of memory!\n");
        exit(EXIT_FAILURE);
    }

    dbus_error_init(&err);
    reply = dbus_connection_send_with_reply_and_block(c, msg, 1000 * 20, &err);
    dbus_message_unref(msg);
    if (dbus_error_is_set(&err)) {
        printf("btcond returned error: %s\n", err.name);
        dbus_error_free(&err);
    }

    g_free(used_dev);
    used_dev = NULL;
#endif 
}

gboolean btcond_is_active(void) {
#ifdef USE_BTCOND
    gboolean ret;
    DBusConnection *c;

    c = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
    if (c == NULL) {
        fprintf(stderr, "Unable to connect to the system D-BUS\n");
        return FALSE;
    }

    ret = dbus_bus_name_has_owner(c, BTCOND_SERVICE, NULL);
    if (!ret)
        printf("btcond is not connected to the system D-BUS\n");

    return ret;
#else
    return FALSE;
#endif
}


