/**
  @file ask-pin.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>
#include <glib.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <bluetooth/bluetooth.h>

static gchar *ask_pin(DBusConnection *connection, const gchar *bdastr) {
    DBusMessage *request, *reply;
    dbus_bool_t out = TRUE;
    bdaddr_t bda;
    bdaddr_t *bda_ptr = &bda;
    gchar *pin;
    char *pin_arg;

    if (str2ba(bdastr, &bda) < 0) {
        fprintf(stderr, "Invalid BDA: %s", bdastr);
        return NULL;
    }

    request = dbus_message_new_method_call("org.bluez.PinAgent",
                                           "/org/bluez/PinAgent",
                                           "org.bluez.PinAgent",
                                           "PinRequest");
    dbus_message_append_args(request,
                             DBUS_TYPE_BOOLEAN, &out,
                             DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE, &bda_ptr, sizeof(bda),
                             DBUS_TYPE_INVALID);
    reply = dbus_connection_send_with_reply_and_block(connection, request,
                                                      -1, NULL);
    dbus_message_unref(request);
    if (reply == NULL) {
        fprintf(stderr, "Send message failed, got NULL reply\n");
        return NULL;
    }

    if (dbus_message_get_type(reply) == DBUS_MESSAGE_TYPE_ERROR) {
        fprintf(stderr, "Got error return\n");
        return NULL;
    }

    if (!dbus_message_get_args(reply, NULL,
                               DBUS_TYPE_STRING, &pin_arg,
                               DBUS_TYPE_INVALID)) {
        fprintf(stderr, "Getting arguments from PIN reply failed\n");
        return NULL;
    }

    pin = g_strdup(pin_arg);

    return pin;
}

int main(int argc, char *argv[]) {
    DBusConnection *connection;
    gchar *pin;

    if (argc < 2) {
        fprintf(stderr, "Usage: %s <BT Address>\n", argv[0]);
        exit(EXIT_FAILURE);
    }

    connection = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
    if (connection == NULL) {
        fprintf(stderr, "D-BUS connection failed\n");
        exit(EXIT_FAILURE);
    }
    dbus_connection_setup_with_g_main(connection, NULL);

    pin = ask_pin(connection, argv[1]);
    if (pin == NULL) {
        fprintf(stderr, "Failed to get PIN\n");
        exit(EXIT_FAILURE);
    }

    printf("Got PIN reply: %s\n", pin);
    g_free(pin);

    exit(EXIT_SUCCESS);
}
