/**
  @file btpin-dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <glib.h>

#include <dbus/dbus.h>

#include <bluetooth/bluetooth.h>

#include "log.h"
#include "dbus.h"
#include "dbus-helper.h"
#include "btpin-stack.h"
#include "bttools-dbus.h"
#include "btpin-dbus.h"

#define BLUEZ_PIN_AGENT_PATH "/org/bluez/PinAgent"

#define TIMEOUT (1000 * 60)

extern GMainLoop *event_loop;

static const char bda_sgn[]	= { DBUS_TYPE_STRING, DBUS_TYPE_STRING, DBUS_TYPE_INVALID };
static const char nobda_sgn[]	= { DBUS_TYPE_STRING, DBUS_TYPE_INVALID };

static gchar *read_pin(void) {
    char pin[128], *ptr;
    FILE *pin_file;

    pin_file = fopen(PIN_FILE, "r");
    if (pin_file == NULL) {
        error("Unable to open %s: %s", PIN_FILE, strerror(errno));
        return g_strdup(DEFAULT_PIN);
    }
    
    ptr = fgets(pin, sizeof(pin), pin_file);
    fclose(pin_file);
    if (ptr == NULL) {
        error("Unable to read %s", PIN_FILE);
        return g_strdup(DEFAULT_PIN);
    }

    /* Remove possible newline */
    ptr = strchr(pin, '\n');
    if (ptr != NULL)
        *ptr = '\0';

    return g_strdup(pin);
}

static gboolean send_pin_reply(DBusConnection *connection,
                               DBusMessage    *message,
                               const gchar    *pin) {
    DBusMessage *reply;
    reply = new_dbus_method_return(message);
    if (pin != NULL) {
        debug("Replying with PIN: %s", pin);
        append_dbus_args(reply,
                         DBUS_TYPE_STRING, &pin,
                         DBUS_TYPE_INVALID);
    }
    else
        debug("Replying with error");
    return send_and_unref(connection, reply);
}

static void notify_cb(DBusPendingCall *pending,
                      DBusMessage     *message) {
    DBusConnection *connection;
    DBusMessage *reply;
    gchar *pin = NULL;

    reply = dbus_pending_call_steal_reply(pending);
    if (reply == NULL)
        error("dbus_pending_call_get_reply returned NULL!");
    if (dbus_message_get_type(reply) == DBUS_MESSAGE_TYPE_ERROR)
        error("Got error reply: %s", dbus_message_get_error_name(reply));
    else {
        if (!dbus_message_get_args(reply, NULL,
                                   DBUS_TYPE_STRING, &pin,
                                   DBUS_TYPE_INVALID))
            error("Invalid arguments in PIN reply");
        else
            debug("Got reply from handler: %s", pin);
    }

    if (reply)
        dbus_message_unref(reply);

    connection = get_dbus_connection();

    if (!send_pin_reply(connection, message, pin))
        error("Sending PIN reply failed");

    dbus_message_unref(message);
}

/* Handler for D-Bus requests */
static DBusHandlerResult bluez_req_handler(DBusConnection     *connection,
                                           DBusMessage        *message,
                                           void               *user_data) {
    gchar *service = NULL, *path = NULL;
    const char *dev_path, *address;
    DBusPendingCall *pending;
    DBusMessage *newmsg;
    dbus_bool_t out = TRUE;
    bdaddr_t ba;
    char *ptr = (char *)&ba;

    if (!dbus_message_is_method_call(message, "org.bluez.PasskeyAgent", "Request"))
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    if (!dbus_message_get_args(message, NULL,
                               DBUS_TYPE_STRING, &dev_path,
                               DBUS_TYPE_STRING, &address,
                               DBUS_TYPE_INVALID)) {
        error("Unexpected arguments to Security.Request");
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    debug("Request(%s, %s)", dev_path, address);

    if (!btpin_stack_get_handler(address, &service, &path)) {
        /* Fallback */
        gchar *pin;
        debug("No PIN-handler matched. Using fallback");
        pin = read_pin();
        if (!send_pin_reply(connection, message, pin))
            error("Sending PIN reply failed");
        g_free(pin);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    debug("Found matching handler at (%s, %s)", service, path);

    str2ba(address, &ba);

    newmsg = new_dbus_method_call(service, path, PINAGENT_INTERFACE,
                                  "PinRequest");

    append_dbus_args(newmsg,
            DBUS_TYPE_BOOLEAN, &out,
            DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE, &ptr, sizeof(bdaddr_t),
            DBUS_TYPE_INVALID);

    if (!dbus_connection_send_with_reply(connection, newmsg,
                &pending, TIMEOUT))
        die("Out of memory during dbus_connection_send_with_reply");

    debug("PIN request was sent to handler. Waiting for reply.");

    if (!dbus_pending_call_set_notify(pending,
                (DBusPendingCallNotifyFunction)notify_cb,
                message,
                (DBusFreeFunction)dbus_message_unref))
        die("Out of memory during dbus_pending_call_set_notify");

    dbus_message_ref(message);

    dbus_message_unref(newmsg);
    g_free(service);
    g_free(path);

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult btpin_req_handler(DBusConnection     *connection,
                                           DBusMessage        *message,
                                           void               *user_data) {

    char *path, *bda;
    const char *service;

    if (dbus_message_has_signature(message, bda_sgn))
        (void) dbus_message_get_args(message, NULL,
                                     DBUS_TYPE_STRING, &path,
                                     DBUS_TYPE_STRING, &bda,
                                     DBUS_TYPE_INVALID);
    else if (dbus_message_has_signature(message, nobda_sgn)) {
        (void) dbus_message_get_args(message, NULL,
                                     DBUS_TYPE_STRING, &path,
                                     DBUS_TYPE_INVALID);
        bda = NULL;
    }
    else {
        send_invalid_args(connection, message);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    service = dbus_message_get_sender(message);

    /* Handler "register" method call */
    if (dbus_message_is_method_call(message,
                                    BTPIN_INTERFACE,
                                    BTPIN_REQ_REGISTER)) {

        debug("Method call: register(%s, %s, %s)",
                service, path, bda ? bda : "any");

        if (!btpin_stack_register(service, path, bda)) {
            error("Registering PIN-handler failed!");
            if (!send_dbus_error(connection, message, BTPIN_ERROR_DUPLICATE))
                error("Sending error reply failed");
        }
        else {
            DBusMessage *reply;

            reply = new_dbus_method_return(message);
            if (!send_and_unref(connection, reply))
                error("Sending D-BUS reply failed");
        }

        return DBUS_HANDLER_RESULT_HANDLED;
    }

    /* Handler "unregister" method call */
    if (dbus_message_is_method_call(message,
                                    BTPIN_INTERFACE,
                                    BTPIN_REQ_UNREGISTER)) {

        debug("Method call: unregister(%s, %s, %s)",
                service, path, bda ? bda : "any");

        if (!btpin_stack_unregister(service, path, bda)) {
            error("Unregistering PIN-handler failed!");
            if (!send_dbus_error(connection, message, BTPIN_ERROR_NOMATCH))
                error("Sending error reply failed");
        }
        else {
            DBusMessage *reply;

            reply = new_dbus_method_return(message);
            if (!send_and_unref(connection, reply))
                error("Sending D-BUS reply failed");
        }

        return DBUS_HANDLER_RESULT_HANDLED;
    }
                                    
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static gboolean register_default_handler(DBusConnection *conn, const char *path) {
    DBusMessage *msg, *reply;
    DBusError err;

    msg = dbus_message_new_method_call("org.bluez", "/org/bluez",
                                       "org.bluez.Security",
                                       "RegisterDefaultPasskeyAgent");
    if (!msg)
        return FALSE;

    if (!dbus_message_append_args(msg, DBUS_TYPE_STRING, &path,
                                  DBUS_TYPE_INVALID)) {
        dbus_message_unref(msg);
        return FALSE;
    }

    dbus_error_init(&err);
    reply = dbus_connection_send_with_reply_and_block(conn, msg, -1, &err);

    dbus_message_unref(msg);

    if (dbus_error_is_set(&err)) {
        error("%s", err.message);
        dbus_error_free(&err);
        return FALSE;
    }

    dbus_message_unref(reply);

    return TRUE;
}

static DBusObjectPathVTable bluez_req_vtable = {
    .message_function    = bluez_req_handler,
    .unregister_function = NULL
};

static DBusObjectPathVTable btpin_req_vtable = {
    .message_function    = btpin_req_handler,
    .unregister_function = NULL
};

/* Create bindings for D-Bus handlers */
void init_dbus_handlers(DBusConnection *connection) {
    DBusError derr;

    dbus_error_init(&derr);
    dbus_connection_register_object_path(connection,
                                         BLUEZ_PIN_AGENT_PATH,
                                         &bluez_req_vtable,
                                         &derr);
    if (dbus_error_is_set(&derr)) {
        error("register_object_path(%s) failed: %s",
                BLUEZ_PIN_AGENT_PATH, derr.message);
        dbus_error_free(&derr);
    }

    if (!register_default_handler(connection, BLUEZ_PIN_AGENT_PATH)) {
        error("Registering default passkey handler failed");
        return;
    }

    dbus_connection_register_object_path(connection,
                                         BTPIN_PATH,
                                         &btpin_req_vtable,
                                         &derr);
    if (dbus_error_is_set(&derr)) {
        error("register_object_path(%s) failed: %s",
                BTPIN_PATH, derr.message);
        dbus_error_free(&derr);
    }
}

void destroy_dbus_handlers(DBusConnection *connection) {
    dbus_connection_unregister_object_path(connection, BLUEZ_PIN_AGENT_PATH);
}
