/*   ScwView
 *   Copyright 2005,2006  Kalle Vahlman
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
 *
 */

#define USE_RENDER

#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <string.h>

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include "scw_marshal.h"

#include "scwtypes.h"
#include "scwview.h"

#ifdef USE_RENDER
  #include <gdk/gdkx.h>
#endif

#ifdef DEBUG
  #define SCW_DEBUG_ENTER g_print("%s()\n", __FUNCTION__);
#else
  #define SCW_DEBUG_ENTER {}
#endif

#define MAX_COLUMNS 10

#define _SCW_VIEW_ROW_PADDING 2
#define _SCW_VIEW_COLUMN_SPACING 6
#define _SCW_TIMESTAMP_FORMAT "<i>%H:%M</i>"
#define _SCW_ACTION_ATTRIBUTES "underline='single'"
#define _SELECTION_ROW_SEPARATOR "\n"
#define _SELECTION_COLUMN_SEPARATOR " "
#define _SCW_VIEW_ODD_ROW_TINT 0.9
#define _SCW_EMBEDDED_ICON_SIZE 12
#define _SCW_ROW_AGE 12

#define SCW_ACTION_START "<span %s>"
#define SCW_ACTION_END "</span>"

#define ACTION_TAG "<action"
#define ACTION_END_TAG "</action>"
#define ICON_TAG   "<icon"

enum
{
  PROP_0,
  PROP_MODEL,
  PROP_ALIGN_PRESENCES,
  PROP_PRESENCE_ALIGNMENT,
  PROP_SCROLL_ON_APPEND,
  PROP_TIMESTAMP_FORMAT,
  PROP_ACTION_ATTRIBUTES,
  PROP_SELECTION_ROW_SEPARATOR,
  PROP_SELECTION_COLUMN_SEPARATOR,
  PROP_EMBEDDED_ICON_SIZE,
  PROP_INTERACTION,
  PROP_INVERT_PANNING
};

enum
{
  ACTIVATE,
  CONTEXT_REQUEST,
  BUFFER_REQUEST,
  LAST_SIGNAL
};

static guint scw_view_signals[LAST_SIGNAL] = { 0 };

/* Enumeration for activation types
 * Used internally
 */
enum
{
  SCW_ACTIVATION_ACTIVATE,
  SCW_ACTIVATION_PRELIGHT,
  SCW_ACTIVATION_CONTEXT
};

typedef struct _ScwViewPrivate ScwViewPrivate;

struct _ScwViewPrivate
{
  GdkWindow *draw_window;

  gint width;
  gint height;
  
  gboolean selecting;
  gboolean panning;
  gboolean invert_panning;
  gchar *selection;
  GdkRectangle selection_box;
  GdkPoint selection_start;
  GdkPoint selection_end;
  GdkPoint pan_start;
  
  GdkRectangle active_prelight;  
  PangoAttribute *prelight_attr;
  PangoLayout  *prelight_layout;
  
  gboolean align_presences;
  PangoAlignment presence_alignment;
  gboolean scroll_on_append;
  gboolean should_scroll;
  gchar *timestamp_format;
  gchar *action_attributes;
  gchar *selection_column_separator;
  gchar *selection_row_separator;
  gint icon_size;
  gboolean buffer_requested;

  ScwInteraction interaction;

  gint fixed_width[MAX_COLUMNS];
  gboolean fold_column[MAX_COLUMNS];
  gboolean column_visible[MAX_COLUMNS];
  
  guint model_signals[3];
  
  GtkAdjustment *hadjustment;
  GtkAdjustment *vadjustment;
  
  gint scroll_direction;

  GtkTreeModel *model;

  GList *first_row;
  GList *last_row;

  GList *selected_rows;
};


typedef struct _ScwViewRow  ScwViewRow;
struct _ScwViewRow
{
  GList *layouts;
  GList *pixbufs;
  GList *slots;
  GList *selections;
  
  gint target_width;
  gint calculated_height;
  
  gint y;
  gboolean odd;
  
  gint age;
};

typedef struct _ScwSlot ScwSlot;
struct _ScwSlot
{
  gint x;
  gint y;
  gint width;
  gint height;
  GList *embedded_icons;
};

typedef struct _ScwViewSelection ScwViewSelection;
struct _ScwViewSelection
{
  gint start_index;
  gint end_index;
};

typedef struct _ScwSize ScwSize;
struct _ScwSize
{
  gint width;
  gint height;
};


#define SCW_VIEW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), SCW_TYPE_VIEW, ScwViewPrivate))

static void scw_view_row_calculate_height (ScwView *view, ScwViewRow *row);

static void scw_view_class_init  (ScwViewClass    *klass);
static void scw_view_init        (ScwView         *view);
static void scw_view_finalize    (GObject         *object);

/* Methods */

void scw_view_set_property (GObject      *object,
                            guint         prop_id,
                            const GValue *value,
                            GParamSpec   *pspec);
void scw_view_get_property (GObject      *object,
                            guint         prop_id,
                            GValue *value,
                            GParamSpec   *pspec);

static void     scw_view_realize  (GtkWidget         *widget);
static gboolean scw_view_expose   (GtkWidget       *widget,
                                   GdkEventExpose  *event);
static gboolean scw_view_button_press   (GtkWidget        *widget,
                                         GdkEventButton   *event);
static gboolean scw_view_button_release (GtkWidget        *widget,
                                         GdkEventButton   *event);
static gboolean scw_view_key_release    (GtkWidget *widget,
                                         GdkEventKey *event);

static gboolean scw_view_motion   (GtkWidget        *widget,
                                   GdkEventMotion   *event);
static gboolean scw_view_leave_notify   (GtkWidget          *widget,
                                         GdkEventCrossing   *event);

static void scw_view_size_request (GtkWidget      *widget,
		                               GtkRequisition *requisition);
static void scw_view_size_allocate (GtkWidget     *widget,
			                              GtkAllocation *allocation);

static void          scw_view_set_model (ScwView     *view,
                                  GtkTreeModel *model);
static GtkTreeModel *scw_view_get_model (ScwView     *view);

static void scw_view_set_adjustments          (ScwView   *scw_view,
                                               GtkAdjustment *hadj,
                                               GtkAdjustment *vadj);
static void scw_view_adjustment_changed       (GtkAdjustment *adjustment,
                                               ScwView   *scw_view);
static void scw_view_adjustment_value_changed (GtkAdjustment *adjustment,
                                               ScwView   *scw_view);

static void model_row_inserted     (GtkTreeModel *treemodel,
                             GtkTreePath *path,
                             GtkTreeIter *iter,
                             gpointer user_data);
static void model_row_deleted     (GtkTreeModel *treemodel,
                             GtkTreePath *path,
                             gpointer user_data);
static void model_row_changed      (GtkTreeModel *treemodel,
                             GtkTreePath *path,
                             GtkTreeIter *iter,
                             gpointer user_data);

static ScwViewRow *scw_view_row_new ();
static void scw_view_row_destroy (ScwViewRow *row);

static void scw_view_row_select (ScwView *view,
                          ScwViewRow *row);
static void scw_view_row_select_all (ScwView *view,
                              ScwViewRow *row);

static void scw_view_activated (ScwView *view,
                                 gchar *id,
                                 gchar *action_data);

static gboolean scw_view_activation_at_coord (ScwView *view,
                                               ScwViewRow *row,
                                               gint x,
                                               gint y,
                                               guint activation_type);

static ScwViewSelection *scw_view_selection_new ();

static ScwViewRow *scw_view_row_at_height(ScwView *view, gint y);

static gboolean filter_prelight (PangoAttribute *attribute,
                                 gpointer data);

static gboolean decide_oddness_for_item(GList *item);

static gchar *scw_create_timestamp (ScwView *view, time_t stamp);


static GtkWidgetClass *parent_class = NULL;

GType
scw_view_get_type (void)
{
  static GType view_type = 0;

  if (!view_type)
    {
      static const GTypeInfo scwview_info =
      {
        sizeof (ScwViewClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) scw_view_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (ScwView),
        0,              /* n_preallocs */
        (GInstanceInitFunc) scw_view_init,
        NULL,           /* value_table */
      };

      view_type = g_type_register_static (GTK_TYPE_WIDGET, "ScwView",
                                         &scwview_info, 0);
    }

  return view_type;
}

static void
scw_view_class_init (ScwViewClass *scw_view_class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (scw_view_class);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (scw_view_class);
  parent_class = g_type_class_peek_parent (scw_view_class);

  SCW_DEBUG_ENTER
  
  gobject_class->finalize = scw_view_finalize;
  gobject_class->set_property = scw_view_set_property;
  gobject_class->get_property = scw_view_get_property;

  widget_class->realize = scw_view_realize;
  widget_class->expose_event = scw_view_expose;
  widget_class->size_request = scw_view_size_request;
  widget_class->size_allocate = scw_view_size_allocate;
  widget_class->motion_notify_event = scw_view_motion;
  widget_class->leave_notify_event = scw_view_leave_notify;
  widget_class->button_press_event = scw_view_button_press;
  widget_class->button_release_event = scw_view_button_release;
  widget_class->key_release_event = scw_view_key_release;
  
  scw_view_class->set_scroll_adjustments = scw_view_set_adjustments;
  scw_view_class->activate = scw_view_activated;

  scw_view_signals[ACTIVATE] =
    g_signal_new ("activate",
		  G_TYPE_FROM_CLASS (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (ScwViewClass, activate),
		  NULL, NULL, 
		  scw_marshal_VOID__STRING_STRING,
		  G_TYPE_NONE, 2,
		  G_TYPE_STRING, G_TYPE_STRING);

  scw_view_signals[CONTEXT_REQUEST] =
    g_signal_new ("context-request",
		  G_TYPE_FROM_CLASS (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (ScwViewClass, context_request),
		  NULL, NULL, 
		  scw_marshal_VOID__STRING_STRING_INT_INT,
		  G_TYPE_NONE, 4,
		  G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT, G_TYPE_INT);

  scw_view_signals[BUFFER_REQUEST] =
    g_signal_new ("buffer-request",
		  G_TYPE_FROM_CLASS (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (ScwViewClass, buffer_request),
		  NULL, NULL, 
		  scw_marshal_VOID__OBJECT_BOOLEAN,
		  G_TYPE_NONE, 2,
		  GTK_TYPE_TREE_MODEL, G_TYPE_BOOLEAN);

  widget_class->set_scroll_adjustments_signal =
    g_signal_new ("set_scroll_adjustments",
		  G_TYPE_FROM_CLASS (gobject_class),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (ScwViewClass, set_scroll_adjustments),
		  NULL, NULL, 
		  scw_marshal_VOID__OBJECT_OBJECT,
		  G_TYPE_NONE, 2,
		  GTK_TYPE_ADJUSTMENT, GTK_TYPE_ADJUSTMENT);


  /* Properties */

  /**
   * ScwView:model:
   *
   * The model from which the view gets it data
   *
   **/

  g_object_class_install_property (gobject_class,
    PROP_MODEL,
    g_param_spec_object ("model",
                         "Model",
                         "The model from which the view gets it data",
                         GTK_TYPE_TREE_MODEL,
                         G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_ALIGN_PRESENCES,
    g_param_spec_boolean ("align-presences",
                         "Align presence names in the view",
                         "Aligns presence names (the type ScwPresence) " \
                         "added to the view to take up the same space",
                         FALSE,
                         G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_PRESENCE_ALIGNMENT,
    g_param_spec_enum ("presence-alignment",
                      "Alignment of presence names in the view",
                      "Decide which PangoAlignment type will be used "\
                      "for presence names (the type ScwPresence)",
                      PANGO_TYPE_ALIGNMENT,
                      PANGO_ALIGN_RIGHT,
                      G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_SCROLL_ON_APPEND,
    g_param_spec_boolean ("scroll-on-append",
                         "Scrolls to the end of buffer when a row is added",
                         "Scrolls the view to the end of the buffer when " \
                         "new row is added to the model",
                         TRUE,
                         G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_TIMESTAMP_FORMAT,
    g_param_spec_string ("timestamp-format",
                         "The format string used in timestamps",
                         "The strftime-compatible format string " \
                         "to be used with the timestamp columns",
                         NULL,
                         G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_ACTION_ATTRIBUTES,
    g_param_spec_string ("action-attributes",
                         "Attributes to apply for activatable text",
                         "The string used as the attribute section " \
                         "for the pango span tag that replaces action tags",
                         NULL,
                         G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_SELECTION_ROW_SEPARATOR,
    g_param_spec_string ("selection-row-separator",
                         "The string used to separate rows in selections",
                         "The string used to separate rows in selections",
                         NULL,
                         G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_SELECTION_COLUMN_SEPARATOR,
    g_param_spec_string ("selection-column-separator",
                         "The string used to separate columns in selections",
                         "The string used to separate columns in selections",
                         NULL,
                         G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_EMBEDDED_ICON_SIZE,
    g_param_spec_int ("embedded-icon-size",
                      "The size of embedded icons",
                      "The size of icons that are embedded in the text " \
                      "with the <icon> tag",
                      0,
                      G_MAXINT,
                      _SCW_EMBEDDED_ICON_SIZE,
                      G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_INTERACTION,
    g_param_spec_enum ("interaction",
                       "Interaction method",
                       "The method of interaction with the view",
                       SCW_TYPE_INTERACTION,
                       SCW_INTERACTION_SELECT,
                       G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_INVERT_PANNING,
    g_param_spec_boolean ("invert-panning",
                          "Inverted panning",
                          "Should the panning work in opposite direction",
                          FALSE,
                          G_PARAM_READWRITE));

  /* Style propertes */

  gtk_widget_class_install_style_property (widget_class,
    g_param_spec_boolean ("alternate-row-colors",
                         "Use alternate colors for even and odd rows",
                         "Controls whether the row colors should alternate " \
                         "between even and odd rows",
                         TRUE,
                         G_PARAM_READWRITE));

  gtk_widget_class_install_style_property (widget_class,
    g_param_spec_boxed ("even-row-color",
                        "Even Row Color",
                        "Color to use for even rows, "
                        "also used as the background color",
                        GDK_TYPE_COLOR,
                        G_PARAM_READABLE));

  gtk_widget_class_install_style_property (widget_class,
    g_param_spec_boxed ("odd-row-color",
                        "Odd Row Color",
                        "Color to use for odd rows",
                        GDK_TYPE_COLOR,
                        G_PARAM_READABLE));

  gtk_widget_class_install_style_property (widget_class,
    g_param_spec_float ("odd-row-tint",
                        "Amount of shading applied to odd rows",
                        "Amount of shading applied to odd rows "
                        "if the odd-row-color is not set",
                        0,
                        1,
                        _SCW_VIEW_ODD_ROW_TINT,
                        G_PARAM_READABLE));

  gtk_widget_class_install_style_property (widget_class,
    g_param_spec_int ("row-padding",
                      "Padding around row contents",
                      "Amount of empty space to leave around row contents",
                      0,
                      G_MAXINT,
                      _SCW_VIEW_ROW_PADDING,
                      G_PARAM_READABLE));

  gtk_widget_class_install_style_property (widget_class,
    g_param_spec_int ("column-spacing",
                      "Space between columns",
                      "Amount of empty space to leave between columns",
                      0,
                      G_MAXINT,
                      _SCW_VIEW_COLUMN_SPACING,
                      G_PARAM_READABLE));


  g_type_class_add_private (gobject_class, sizeof (ScwViewPrivate));
}

static void
scw_view_init (ScwView *view)
{
  gint i;
  ScwViewPrivate *priv;
  
  SCW_DEBUG_ENTER

  priv = SCW_VIEW_GET_PRIVATE (view);

  priv->selecting = FALSE;
  priv->panning = FALSE;
  priv->selection = NULL;
  priv->selection_box.x = 0;
  priv->selection_box.y = 0;
  priv->selection_box.width = 0;
  priv->selection_box.height = 0;
  priv->selection_start.x = 0;
  priv->selection_start.y = 0;
  priv->selection_end.x = 0;
  priv->selection_end.y = 0;
  priv->pan_start.x = 0;
  priv->pan_start.y = 0;

  priv->buffer_requested = FALSE;

  priv->prelight_attr = NULL;
  priv->prelight_layout = NULL;

  priv->width = 0;
  priv->height = 0;
  priv->model = NULL;
  priv->align_presences = FALSE;
  priv->presence_alignment = PANGO_ALIGN_RIGHT;
  priv->scroll_on_append = TRUE;
  priv->should_scroll = FALSE;
  priv->timestamp_format = NULL;
  priv->action_attributes = NULL;
  priv->selection_row_separator = NULL;
  priv->selection_column_separator = NULL;
  priv->icon_size = _SCW_EMBEDDED_ICON_SIZE;
  priv->interaction = SCW_INTERACTION_SELECT;

  priv->first_row = NULL;
  priv->last_row = NULL;

  priv->selected_rows = NULL;

  priv->scroll_direction = 0;

  priv->model_signals[0] = 0;
  priv->model_signals[1] = 0;
  priv->model_signals[2] = 0;

  for (i = 0; i < MAX_COLUMNS; i++)
    {
      priv->fixed_width[i]    = 0;
      priv->fold_column[i]    = FALSE;
      priv->column_visible[i] = TRUE;
    }
}

static void scw_view_finalize    (GObject         *object)
{
  GList *l;
  ScwViewPrivate *priv;
  ScwView *view;
 
  SCW_DEBUG_ENTER

  g_assert (object != NULL);
  g_assert (SCW_IS_VIEW (object));

  view = SCW_VIEW (object);

  priv = SCW_VIEW_GET_PRIVATE (view);

  l = priv->first_row;
  while (l != NULL)
    {
      ScwViewRow *row = (ScwViewRow *)l->data;
      scw_view_row_destroy (row);
      l = l->next;
    }
    
  g_list_free(priv->first_row);
  g_list_free(priv->selected_rows);
  
  g_free (priv->timestamp_format);
  g_free (priv->selection);
  g_free (priv->selection_row_separator);
  g_free (priv->selection_column_separator);
  g_free (priv->action_attributes);
}


void
scw_view_set_property (GObject      *object,
                       guint         prop_id,
                       const GValue *value,
                       GParamSpec   *pspec)
{
  ScwView *view;
  ScwViewPrivate *priv;
  
  SCW_DEBUG_ENTER

  g_assert (object != NULL);
  g_assert (SCW_IS_VIEW (object));

  view = SCW_VIEW (object);
  priv = SCW_VIEW_GET_PRIVATE (view);
  
  switch (prop_id)
  {
    case PROP_MODEL:
      scw_view_set_model (view, GTK_TREE_MODEL (g_value_get_object (value)));
      break;
    case PROP_ALIGN_PRESENCES:
      priv->align_presences = g_value_get_boolean (value);
      break;
    case PROP_PRESENCE_ALIGNMENT:
      priv->presence_alignment = g_value_get_enum (value);
      break;
    case PROP_SCROLL_ON_APPEND:
      priv->scroll_on_append = g_value_get_boolean (value);
      break;
    case PROP_TIMESTAMP_FORMAT:
      g_free (priv->timestamp_format);
      priv->timestamp_format = g_strdup (g_value_get_string (value));
      break;
    case PROP_ACTION_ATTRIBUTES:
      g_free (priv->action_attributes);
      priv->action_attributes = g_strdup (g_value_get_string (value));
      break;
    case PROP_SELECTION_ROW_SEPARATOR:
      g_free (priv->selection_row_separator);
      priv->selection_row_separator = g_strdup (g_value_get_string (value));
      break;
    case PROP_SELECTION_COLUMN_SEPARATOR:
      g_free (priv->selection_column_separator);
      priv->selection_column_separator = g_strdup (g_value_get_string (value));
      break;
    case PROP_EMBEDDED_ICON_SIZE:
      priv->icon_size = g_value_get_int (value);
      break;
    case PROP_INTERACTION:
      priv->interaction = g_value_get_enum (value);
      break;
    case PROP_INVERT_PANNING:
      priv->invert_panning = g_value_get_boolean (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }    
  
}

void
scw_view_get_property (GObject    *object,
                       guint      prop_id,
                       GValue     *value,
                       GParamSpec *pspec)
{
  ScwView *view;
  ScwViewPrivate *priv;
  
  SCW_DEBUG_ENTER

  g_assert (object != NULL);
  g_assert (SCW_IS_VIEW (object));

  view = SCW_VIEW (object);
  priv = SCW_VIEW_GET_PRIVATE (view);
  
  switch (prop_id)
  {
    case PROP_MODEL:
      g_value_set_object (value, G_OBJECT (scw_view_get_model (view)));
      break;
    case PROP_ALIGN_PRESENCES:
      g_value_set_boolean (value, priv->align_presences);
      break;
    case PROP_PRESENCE_ALIGNMENT:
      g_value_set_enum (value, priv->align_presences);
      break;
    case PROP_SCROLL_ON_APPEND:
      g_value_set_boolean (value, priv->scroll_on_append);
      break;
    case PROP_TIMESTAMP_FORMAT:
      g_value_set_string (value, priv->timestamp_format);
      break;
    case PROP_ACTION_ATTRIBUTES:
      g_value_set_string (value, priv->action_attributes);
      break;
    case PROP_SELECTION_ROW_SEPARATOR:
      g_value_set_string (value, priv->selection_row_separator);
      break;
    case PROP_SELECTION_COLUMN_SEPARATOR:
      g_value_set_string (value, priv->selection_column_separator);
      break;
    case PROP_EMBEDDED_ICON_SIZE:
      g_value_set_int (value, priv->icon_size);
      break;
    case PROP_INTERACTION:
      g_value_set_enum (value, priv->interaction);
      break;
    case PROP_INVERT_PANNING:
      g_value_set_boolean (value, priv->invert_panning);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }    
  
}

static gboolean inserted_helper    (GtkTreeModel *treemodel,
                             GtkTreePath *path,
                             GtkTreeIter *iter,
                             gpointer user_data)
{
  SCW_DEBUG_ENTER

  model_row_inserted (treemodel, path, iter, user_data);
  return FALSE;
}

static gboolean changed_helper     (GtkTreeModel *treemodel,
                             GtkTreePath *path,
                             GtkTreeIter *iter,
                             gpointer user_data)
{
  SCW_DEBUG_ENTER

  model_row_changed (treemodel, path, iter, user_data);
  return FALSE;
}

static
void scw_view_set_model (ScwView     *view,
                         GtkTreeModel *model)
{
  ScwViewPrivate *priv;
  
  SCW_DEBUG_ENTER

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));
  g_assert (model != NULL);
  
  priv = SCW_VIEW_GET_PRIVATE (view);
  
  if (priv->model != NULL)
    {
      /* On second thought, this might be a bad idea.
      g_object_unref (priv->model);
      */

      g_list_foreach (priv->first_row, (GFunc)scw_view_row_destroy, NULL);
      g_list_free (priv->first_row);
      
      priv->first_row = NULL;
      priv->last_row = NULL;

      g_signal_handler_disconnect (G_OBJECT (priv->model),
                                   priv->model_signals[0]);
      g_signal_handler_disconnect (G_OBJECT (priv->model),
                                   priv->model_signals[1]);
      g_signal_handler_disconnect (G_OBJECT (priv->model),
                                   priv->model_signals[2]);

      priv->model = NULL;
    }
  
  priv->model = model;

  priv->model_signals[0] = 
                       g_signal_connect (G_OBJECT (model), "row-inserted",
                                         G_CALLBACK (model_row_inserted), view);

  priv->model_signals[1] = 
                       g_signal_connect (G_OBJECT (model), "row-deleted",
                                         G_CALLBACK (model_row_deleted), view);

  priv->model_signals[2] = 
                       g_signal_connect (G_OBJECT (model), "row-changed",
                                         G_CALLBACK (model_row_changed), view);

  gtk_tree_model_foreach (model,
                          (GtkTreeModelForeachFunc) inserted_helper,
                          view);
  gtk_tree_model_foreach (model,
                          (GtkTreeModelForeachFunc) changed_helper,
                          view);

  g_object_notify (G_OBJECT (view), "model");  

  /* Make sure we update the view too */
  gtk_widget_queue_resize (GTK_WIDGET(view));
}

GtkTreeModel *scw_view_get_model (ScwView     *view)
{
  ScwViewPrivate *priv;
  
  SCW_DEBUG_ENTER

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));
  
  priv = SCW_VIEW_GET_PRIVATE (view);

  /* Make sure it's not just garbage */
  if (GTK_IS_TREE_MODEL (priv->model))  
    {
      return priv->model;
    }

  return NULL;
}

static void
scw_view_set_adjustments (ScwView   *scw_view,
              		        GtkAdjustment *hadj,
              		        GtkAdjustment *vadj)
{
  gboolean need_adjust = FALSE;
  ScwViewPrivate *priv;
  
  SCW_DEBUG_ENTER
  
  g_assert (scw_view != NULL);
  g_assert (SCW_IS_VIEW (scw_view));

  priv = SCW_VIEW_GET_PRIVATE (SCW_VIEW (scw_view));

  if (hadj)
    {
      g_assert (GTK_IS_ADJUSTMENT (hadj));
      if (GTK_WIDGET_REALIZED(scw_view))
        {
          hadj->page_size = GTK_WIDGET(scw_view)->allocation.width;
          hadj->page_increment = (gint) hadj->page_size * 0.9;
          hadj->step_increment = (gint) hadj->page_size * 0.1;
          hadj->lower = 0;
          hadj->upper = MAX (0, GTK_WIDGET(scw_view)->requisition.width);
        }
    }
  else
    {
      hadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 0.0, 0.0, 0.0, 0.0));
    }
    
  if (vadj)
    {
      g_assert (GTK_IS_ADJUSTMENT (vadj));
      if (GTK_WIDGET_REALIZED(scw_view))
        {
          vadj->page_size = GTK_WIDGET(scw_view)->allocation.height;
          vadj->page_increment = (gint) vadj->page_size * 0.9;
          vadj->step_increment = (gint) vadj->page_size * 0.1;
          vadj->lower = 0;
          vadj->upper = MAX (0, GTK_WIDGET(scw_view)->requisition.height);
        }
    }
  else
    {
      vadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 0.0, 0.0, 0.0, 0.0));
    }

  if (priv->hadjustment && (priv->hadjustment != hadj))
    {
      g_signal_handlers_disconnect_matched (priv->hadjustment,
                                            G_SIGNAL_MATCH_DATA,
                                            0, 0, NULL, NULL,
                                            scw_view);
      g_object_unref (priv->hadjustment);
    }

  if (priv->vadjustment && (priv->vadjustment != vadj))
    {
      g_signal_handlers_disconnect_matched (priv->vadjustment,
                                            G_SIGNAL_MATCH_DATA,
                                            0, 0, NULL, NULL,
                                            scw_view);
      g_object_unref (priv->vadjustment);
    }

  if (priv->hadjustment != hadj)
    {
      priv->hadjustment = hadj;
      g_object_ref (priv->hadjustment);
      gtk_object_sink (GTK_OBJECT (priv->hadjustment));

      g_signal_connect (priv->hadjustment, "value_changed",
			                  G_CALLBACK (scw_view_adjustment_value_changed),
			                  scw_view);
      need_adjust = TRUE;
    }

  if (priv->vadjustment != vadj)
    {
      priv->vadjustment = vadj;
      g_object_ref (priv->vadjustment);
      gtk_object_sink (GTK_OBJECT (priv->vadjustment));

      g_signal_connect (priv->vadjustment, "value_changed",
                  			G_CALLBACK (scw_view_adjustment_value_changed),
                  			scw_view);
      need_adjust = TRUE;
    }

  if (need_adjust)
    {
      scw_view_adjustment_changed (NULL, scw_view);
    }
}

static void
scw_view_adjustment_changed (GtkAdjustment *adjustment,
                             ScwView       *scw_view)
{
  ScwViewPrivate *priv;
  
  SCW_DEBUG_ENTER

  g_assert (scw_view != NULL);
  g_assert (SCW_IS_VIEW (scw_view));

  priv = SCW_VIEW_GET_PRIVATE (scw_view);

  if (priv->hadjustment != NULL)
    {
      priv->hadjustment->value = 0;
    }

}

gboolean request_timeout(ScwViewPrivate *priv)
{
  SCW_DEBUG_ENTER

  priv->buffer_requested = FALSE;

  return FALSE;
}

static void
scw_view_adjustment_value_changed (GtkAdjustment *adjustment,
                                   ScwView       *scw_view)
{
  ScwViewPrivate *priv;
  
  SCW_DEBUG_ENTER

  g_assert (scw_view != NULL);
  g_assert (SCW_IS_VIEW (scw_view));

  priv = SCW_VIEW_GET_PRIVATE (scw_view);

  if (GTK_WIDGET_REALIZED (scw_view))
    {
      GdkRectangle area;
      
      area.x = 0;
      area.y = 0;
      area.width = GTK_WIDGET (scw_view)->allocation.width;
      area.height = GTK_WIDGET (scw_view)->allocation.height;
      gdk_window_invalidate_rect (GTK_WIDGET (scw_view)->window,
                                  &area, FALSE);

      /* Remove prelight */
      if (priv->prelight_attr != NULL
          && priv->prelight_layout != NULL)
        {
          PangoAttrList *list;
          PangoAttrList *new_list;
          
          list = pango_layout_get_attributes (priv->prelight_layout);
          new_list = pango_attr_list_filter (list,
                                             (PangoAttrFilterFunc) filter_prelight,
                                             priv->prelight_attr);
          pango_layout_set_attributes (priv->prelight_layout, new_list);
          pango_attr_list_unref (new_list);
          priv->prelight_attr = NULL;
          priv->prelight_layout = NULL;
          
        }

      if (priv->active_prelight.height != 0)
        {
          gdk_window_invalidate_rect(GTK_WIDGET(scw_view)->window, &priv->active_prelight, FALSE);

          priv->active_prelight.x = 0;
          priv->active_prelight.y = 0;
          priv->active_prelight.width = 0;
          priv->active_prelight.height = 0;
        }


      gdk_window_process_updates (GTK_WIDGET (scw_view)->window, TRUE);

      if (priv->buffer_requested)
        {
          return;
        }

      if (adjustment == priv->vadjustment)
        {
          gboolean start_treshold, end_treshold;
          
          start_treshold = FALSE;
          end_treshold = FALSE;
          
          if ((gint)adjustment->upper - (gint)adjustment->value
              < (gint)adjustment->page_size*2)
            {
              end_treshold = TRUE;
            }
            
          if ((gint)adjustment->value
              < ((gint)adjustment->page_size))
            {
              start_treshold = TRUE;
            }

          if (start_treshold)
            {
              if (!end_treshold 
                   && priv->scroll_direction > (gint) adjustment->value)
                {
                  g_signal_emit_by_name (G_OBJECT(scw_view), "buffer-request",
                                         priv->model,
                                         start_treshold,
                                         G_TYPE_NONE);
                  priv->buffer_requested = TRUE;
                  g_timeout_add(500, (GSourceFunc)request_timeout, priv);
                }
            }
          else if (end_treshold)
            {
              g_signal_emit_by_name (G_OBJECT(scw_view), "buffer-request",
                                     priv->model,
                                     start_treshold,
                                     G_TYPE_NONE);
              priv->buffer_requested = TRUE;
              g_timeout_add(500, (GSourceFunc)request_timeout, priv);
            }

          priv->scroll_direction = (gint) adjustment->value;

        }
    }
  
}

static void scw_view_realize    (GtkWidget         *widget)
{
  GdkWindowAttr attributes;
  gint attributes_mask;
  ScwViewPrivate *priv;
  
  SCW_DEBUG_ENTER

  g_assert (widget != NULL);
  g_assert (SCW_IS_VIEW (widget));

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  GTK_WIDGET_SET_FLAGS (widget, GTK_CAN_FOCUS);

  priv = SCW_VIEW_GET_PRIVATE (SCW_VIEW (widget));

  /* Visible window */
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.event_mask = gtk_widget_get_events (widget) | 
    GDK_EXPOSURE_MASK |
    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK |
    GDK_LEAVE_NOTIFY_MASK;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  widget->window = gdk_window_new (widget->parent->window, &attributes, attributes_mask);

  widget->style = gtk_style_attach (widget->style, widget->window);

  gdk_window_set_user_data (widget->window, widget);

  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
  
}

static void scw_view_size_request (GtkWidget      *widget,
                                   GtkRequisition *requisition)
{
  gint total_height;
  gint max_width;
  gint target_width;
  ScwViewRow *row;
  ScwViewPrivate *priv;
  
  SCW_DEBUG_ENTER

  g_assert (widget != NULL);
  g_assert (SCW_IS_VIEW (widget));

  priv = SCW_VIEW_GET_PRIVATE (SCW_VIEW (widget));

  total_height = 0;
  target_width = 0;

  if (priv->model != NULL)
    {
      GList *row_iter;
      
      target_width = widget->allocation.width;
      
      total_height = 0;
      max_width = 0;

      if (priv->vadjustment != NULL)
        {
          if (priv->vadjustment->value == priv->vadjustment->upper
                                           - priv->vadjustment->page_size)
            {
              priv->should_scroll = TRUE;
            }
        }

      for (row_iter = priv->first_row; row_iter != NULL;
           row_iter = row_iter->next)
        {
          row = row_iter->data;

          if (row != NULL)
            {
              row->target_width = target_width;
          
              scw_view_row_calculate_height (SCW_VIEW (widget), row);
              
              row->y = total_height;
              total_height += row->calculated_height;
            }
        }
    }

  requisition->width = target_width;
  requisition->height = total_height;

}

static void scw_view_size_allocate (GtkWidget     *widget,
                                    GtkAllocation *allocation)
{
  GtkAdjustment *hadjustment;
  GtkAdjustment *vadjustment;
  ScwViewPrivate *priv;
  
  SCW_DEBUG_ENTER

  g_assert (widget != NULL);
  g_assert (SCW_IS_VIEW (widget));

  g_assert (allocation != NULL);

  priv = SCW_VIEW_GET_PRIVATE (SCW_VIEW (widget));

  widget->allocation = *allocation;

  if (GTK_WIDGET_REALIZED (widget))
    {
      gdk_window_move_resize (widget->window,
			      allocation->x, allocation->y,
			      allocation->width, allocation->height);
    }

  /* If the width of the visible area is chaged, we need to recalculate
   * for wrapping correctly */
  if (allocation->width != widget->requisition.width ||
      allocation->height != widget->requisition.height)
    {
      scw_view_size_request (widget, &widget->requisition);
    }

  if (priv->hadjustment == NULL)
    return;
  if (priv->vadjustment == NULL)
    return;

  hadjustment = priv->hadjustment;
  vadjustment = priv->vadjustment;

  hadjustment->page_size = allocation->width;
  hadjustment->page_increment = (gint) hadjustment->page_size * 0.9;
  hadjustment->step_increment = (gint) hadjustment->page_size * 0.1;
  hadjustment->lower = 0;
  hadjustment->upper = MAX (0, widget->requisition.width);

  if (hadjustment->value > hadjustment->upper - hadjustment->page_size)
    gtk_adjustment_set_value (hadjustment, MAX (0,
                                  hadjustment->upper - hadjustment->page_size));

  vadjustment->page_size = allocation->height;
  vadjustment->page_increment = vadjustment->page_size * 0.9;
  vadjustment->step_increment = vadjustment->page_size * 0.1;
  vadjustment->lower = 0;
  vadjustment->upper = MAX (0, widget->requisition.height);

  if (priv->should_scroll)
    {
      vadjustment->value = MAX (0, vadjustment->upper - vadjustment->page_size);

      priv->should_scroll = FALSE;
    }

  if (vadjustment->value > vadjustment->upper - vadjustment->page_size)
    {
      vadjustment->value = MAX (0, vadjustment->upper - vadjustment->page_size);
    }


  gtk_adjustment_changed (hadjustment);
  gtk_adjustment_changed (vadjustment);

}


/* Paints the items in the given row to the area.
 * The area parameter is two-way:
 * x and y will be the point we start drawing
 * and will be set to where the drawing ends
 * width and height will restrict our drawing
 * and will be set to actual space used
 *
 * Return value is TRUE if the path was valid and rendering
 * was succesful, FALSE otherwise
 */
static gboolean scw_view_paint_row (ScwView *view,
                                    GdkGC *gc,
                                    ScwViewRow *row,
                                    GdkRectangle *area,
                                    GdkRegion *clip,
                                    GtkTreeIter *iter)
{
  gboolean alternate_row_colors, free_color;
  gfloat odd_row_tint;
  gint row_padding, column_spacing;
  gint x, y, width, height;
  gint i, columns;
  ScwSlot *slot;
  GdkColor *odd_row_color;
  GdkColor *even_row_color;
  GdkPixbuf *pixbuf;
  PangoLayout *layout;
  GtkWidget *widget;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));
  
  priv = SCW_VIEW_GET_PRIVATE (view);
  widget = GTK_WIDGET (view);

  if (priv->model == NULL)
    {
      return FALSE;
    }

  if (row == NULL)
    {
      return FALSE;
    }

  slot = NULL;
  
  columns = gtk_tree_model_get_n_columns (priv->model);

  even_row_color = NULL;
  odd_row_color = NULL;
  free_color = FALSE;
  
  gtk_widget_style_get (widget,
                        "row-padding", &row_padding,
                        "column-spacing", &column_spacing,
                        "even-row-color", &even_row_color,
                        "odd-row-color", &odd_row_color,
                        "odd-row-tint", &odd_row_tint,
                        "alternate-row-colors", &alternate_row_colors,
                        NULL);
  
  if (even_row_color == NULL)
    {
      even_row_color = &widget->style->base[GTK_STATE_NORMAL];
    }
    
  if (alternate_row_colors)
    {
      if (odd_row_color == NULL)
        {
          odd_row_color = gdk_color_copy (even_row_color);
          free_color = TRUE;
          odd_row_color->red = even_row_color->red * odd_row_tint;
          odd_row_color->green = even_row_color->green * odd_row_tint;
          odd_row_color->blue = even_row_color->blue * odd_row_tint;
        }

      if (row->odd)
        {
          gdk_gc_set_rgb_fg_color (gc, odd_row_color);
        }
      else
        {
          gdk_gc_set_rgb_fg_color (gc, even_row_color);
        }
    }
  else
    {
      gdk_gc_set_rgb_fg_color (gc, even_row_color);
    }

  x = area->x;
  y = area->y;

  height = y;
  width = x;

  gdk_gc_set_clip_region (gc, clip);

  if (iter != NULL)
    {
      for (i = 0; i < columns; i++)
        {

          GType type = gtk_tree_model_get_column_type (priv->model, i);

          if (type == SCW_TYPE_ROW_COLOR)
            {
              gchar *color_spec;
              GdkColor override_color;
              
              color_spec = NULL;
              gtk_tree_model_get (priv->model, iter, i, &color_spec, -1);

              if (color_spec != NULL)
                {
                  if (gdk_color_parse (color_spec, &override_color))
                    {
                      gdk_gc_set_rgb_fg_color (gc, &override_color);
                    }
                }

              g_free (color_spec);
              
            }
          }
    }
    
  gdk_draw_rectangle (GDK_DRAWABLE (widget->window),
                    gc,
                    TRUE,
                    x - row_padding, y,
                    row->target_width,
                    row->calculated_height);

  gdk_gc_set_foreground (gc, &widget->style->text[GTK_WIDGET_STATE(widget)]);


  for (i = 0; i < columns; i++)
    {

      GType type = gtk_tree_model_get_column_type (priv->model, i);

      slot = (ScwSlot *) g_list_nth_data (row->slots, i);

      if (!priv->column_visible[i])
        {
          /* Skip invisible columns */
        }
      else if (type == GDK_TYPE_PIXBUF && iter != NULL)
        {
          gtk_tree_model_get (priv->model, iter, i, &pixbuf, -1);
          
          if (GDK_IS_PIXBUF(pixbuf))
            {
              gdk_draw_pixbuf (GDK_DRAWABLE (widget->window),
                                gc, pixbuf,
                                0, 0, area->x + slot->x, area->y + slot->y,
                                slot->width, slot->height,
                                GDK_RGB_DITHER_NONE, 0, 0);
              g_object_unref (G_OBJECT (pixbuf));
            }
        }
      else if (type == G_TYPE_STRING
               || type == G_TYPE_INT
               || type == SCW_TYPE_TIMESTAMP
               || type == SCW_TYPE_PRESENCE)
        {    
            
          layout = PANGO_LAYOUT (g_list_nth_data (row->layouts, i));

          if (g_utf8_strlen (pango_layout_get_text (layout), -1) > 0)
            {
              PangoLayoutIter *iter;
              GList *a;
              ScwViewSelection *selection;
              
              selection = (ScwViewSelection *)
                            g_list_nth_data (row->selections, i);
              
              gdk_draw_layout (GDK_DRAWABLE (widget->window),
                                gc,
                                area->x + slot->x, area->y + slot->y,
                                layout);

              if (selection->start_index != selection->end_index)
                {
                  gint range[2];
                  GtkStateType state;
                  GdkRegion *selected_area;
                  
                  range[0] = selection->start_index;
                  range[1] = selection->end_index;
                  
                  selected_area = 
                    gdk_pango_layout_get_clip_region (layout,
                                area->x + slot->x, area->y + slot->y,
                                range, 1);
                  
                  gdk_gc_set_clip_region (gc, selected_area);
                  
                  if (!GTK_WIDGET_HAS_FOCUS (widget))
                    {
                      state = GTK_STATE_ACTIVE;
                    }
                  else
                    {
                      state = GTK_STATE_SELECTED;
                    }
                  
                  gdk_draw_layout_with_colors (GDK_DRAWABLE (widget->window),
                                    gc,
                                    area->x + slot->x, area->y + slot->y,
                                    layout,
                                    &widget->style->text[state],
                                    &widget->style->base[state]);
                                    
                  gdk_gc_set_clip_region (gc, clip);

                  gdk_region_destroy (selected_area);                  

                }

              iter = pango_layout_get_iter (layout);

              for (a = slot->embedded_icons; a; a = a->next)
                {
                  PangoRectangle rect;
                  PangoAttrShape *attr = (PangoAttrShape *) a->data;
                  
                  if (strlen (attr->data) < 1)
                    {
                      continue;
                    }
                  
                  do
                    {
                      if (pango_layout_iter_get_index(iter) == attr->attr.start_index)
                        {
                          GtkIconTheme *theme;
                          GdkPixbuf *icon;
                          
                          icon = NULL;
                          theme = gtk_icon_theme_get_default();
                          icon = gtk_icon_theme_load_icon (theme, attr->data,
                                                           priv->icon_size,
                                                           0, NULL);
                          
                          if (icon != NULL)
                            {
                              
                              pango_layout_iter_get_char_extents (iter, &rect);
                              /* FIXME: calculate the center of the line in the
                               * layout and center the icon to it */
                              gdk_draw_pixbuf (GDK_DRAWABLE (widget->window),
                                               gc, icon,
                                               0, 0,
                                               PANGO_PIXELS (rect.x)
                                               + area->x + slot->x,
                                               PANGO_PIXELS (rect.y)
                                               + area->y + slot->y
                                               - priv->icon_size + 2,
                                               priv->icon_size,
                                               priv->icon_size,
                                               GDK_RGB_DITHER_NONE, 0, 0);

                              g_object_unref(icon);
                            
                            }
                         
                          break;
                        }
                    }
                  while (pango_layout_iter_next_char (iter));
                }
              pango_layout_iter_free(iter);
            }
        }
    }


  /* Tell the caller the area we really drew */
  area->x = slot->x + slot->width;
  area->y = area->y + row->calculated_height;
  area->width = slot->x + slot->width;
  area->height = row->calculated_height;

  if (free_color)
    {
      gdk_color_free (odd_row_color);
    }
  
  return TRUE;
}


static gboolean scw_view_expose    (GtkWidget       *widget,
                                    GdkEventExpose  *event)
{
  gint width, height, total_height, draw_height;
  gint offset, padding;
  gint start_height;
  GdkRectangle area;
  GList *start_row;
  ScwViewRow *row;
  ScwViewPrivate *priv;
  GdkGC *gc;
  GdkColor *even_row_color;
  GtkTreeIter iter;
  GtkTreePath *path;

  SCW_DEBUG_ENTER

  g_assert (widget != NULL);
  g_assert (SCW_IS_VIEW (widget));

  if (widget->window != event->window)
    {
      return FALSE;
    }

  priv = SCW_VIEW_GET_PRIVATE (SCW_VIEW (widget));

  if (priv->model == NULL)
    {
      return FALSE;
    }

  row = NULL;
  
  area.x = 0;
  area.y = 0;
  area.width = 0;
  area.height = 0;
  
  draw_height = 0;
  offset = 0;

  if (GTK_WIDGET_REALIZED (widget))
    {
      GList *l;

      total_height = 0;

      if (priv->vadjustment != NULL)
        {
          start_height = (gint) priv->vadjustment->value + event->area.y;
        }
      else
        {
          start_height = event->area.y;
        }

      start_row = priv->first_row;
      l = priv->first_row;

      while (l != NULL)
        {
          row = (ScwViewRow *)l->data;
          if (row->y >= start_height)
            {
              break;
            }

          start_row = l;
          l = l->next;
        }

      if (start_row != NULL)
        {
          row = (ScwViewRow *)start_row->data;
          offset = row->y - start_height + event->area.y;
        }
      else
        {
          row = NULL;
        }
      
    }
  else
    {
      width = widget->allocation.width;
      height = widget->requisition.height;

      start_row = priv->first_row;
      if (start_row != NULL)
        {
          row = (ScwViewRow *)start_row->data;
        }

      offset = row->y;
      scw_view_size_request (widget, &widget->requisition);
    }

  even_row_color = NULL;
  gtk_widget_style_get (widget,
                        "row-padding", &padding,
                        "even-row-color", &even_row_color,
                        NULL);

  if (even_row_color == NULL)
    {
      even_row_color = &widget->style->base[GTK_STATE_NORMAL];
    }

  area.x = padding;
  area.y = offset;
  area.width = widget->allocation.width;
  area.height = widget->allocation.height;      

  gc = gdk_gc_new (GDK_DRAWABLE(widget->window));

  gdk_gc_set_clip_region (gc, event->region);

  gdk_gc_set_rgb_fg_color (gc, even_row_color);
  gdk_draw_rectangle (GDK_DRAWABLE(widget->window),
                      gc,
                      TRUE,
                      event->area.x,
                      event->area.y,
                      event->area.width,
                      event->area.height);

  if (row == NULL)
    {
      return FALSE;
    }

  path = NULL;
  path = gtk_tree_path_new_from_indices(g_list_position(priv->first_row,
                                                        start_row),
                                        -1);
                                        
  if (path == NULL)
    {
      return FALSE;
    }

  if (!gtk_tree_model_get_iter(priv->model, &iter, path))
    {
      gtk_tree_path_free (path);
      return FALSE;
    }

  gtk_tree_path_free (path);
  
  if (!scw_view_paint_row(SCW_VIEW(widget), gc, row,
                          &area, event->region, &iter))
    {
      return FALSE;
    }

  while (gtk_tree_model_iter_next(priv->model, &iter))
    {
/*      SCWD_AREA("Drew", area); */
      if (event->area.y + event->area.height < area.y)
        {
          break;
        }
        
      start_row = g_list_next(start_row);
      if (start_row == NULL)
        {
          break;
        }
        
      row = (ScwViewRow *)start_row->data;
      if (row == NULL)
        {
          break;
        }

      area.x = padding;
      area.y = area.y;
      area.width = widget->allocation.width;
      area.height = widget->allocation.height - area.height;

      if (!scw_view_paint_row(SCW_VIEW(widget), gc, row,
                              &area, event->region, &iter))
        {
          break;
        }
        
    }

  g_object_unref (G_OBJECT(gc));

  return TRUE;
}

static
void model_row_inserted     (GtkTreeModel *treemodel,
                             GtkTreePath *path,
                             GtkTreeIter *iter,
                             gpointer user_data)
{
  gint columns, i;
  gint *indices;
  ScwSlot *slot;
  ScwViewSelection *selection;
  ScwViewPrivate *priv;
  ScwViewRow *row;
  GList *sibling;
  
  SCW_DEBUG_ENTER

  g_assert (treemodel != NULL);
  g_assert (user_data != NULL);
  g_assert (SCW_IS_VIEW (user_data));

  if (gtk_tree_path_get_depth(path) > 1)
    {
      /* g_warning ("Skipping path at depth %i\n", gtk_tree_path_get_depth(path));*/
      return;
    }
  

  priv = SCW_VIEW_GET_PRIVATE (SCW_VIEW (user_data));

  if (priv->model == NULL)
    {
      g_warning ("Got false signal or in inconsistent state! Model is NULL");
      return;
    }

  row = scw_view_row_new ();
  
  columns = gtk_tree_model_get_n_columns (priv->model);

  for (i = 0; i < columns; i++)
    {
      GType type = gtk_tree_model_get_column_type (priv->model, i);

      slot = g_new0 (ScwSlot, 1);
      
      selection = scw_view_selection_new ();

      if (type == GDK_TYPE_PIXBUF)
        {
          row->layouts = g_list_append(row->layouts, NULL);
        }
      else if (type == SCW_TYPE_TIMESTAMP
               || type == SCW_TYPE_PRESENCE
               || type == G_TYPE_STRING
               || type == G_TYPE_INT)
      
        { 
          row->layouts = 
            g_list_append(row->layouts,
                          (gpointer) gtk_widget_create_pango_layout (
                                        GTK_WIDGET (user_data), ""));
        }

      /* We add the size struct when we have the actual pixbuf,
        * to avoid allocating for rows that have none.
        * TODO: Do this to layouts too
        */
      row->pixbufs = g_list_append(row->pixbufs, NULL);
      row->slots = g_list_append (row->slots, slot);
      row->selections = g_list_append(row->selections, selection);
    }

  indices = gtk_tree_path_get_indices(path);
  sibling = g_list_nth (priv->first_row, indices[0]);

  priv->first_row = g_list_insert_before(priv->first_row, sibling, row);
  priv->last_row = g_list_last(priv->first_row);

  sibling =  g_list_nth (priv->first_row,
                         MIN(indices[0], g_list_length(priv->first_row)-1));


  if (sibling != NULL)
    {
      gboolean oddness;

      row = (ScwViewRow *) sibling->data;
      row->odd = decide_oddness_for_item(sibling);

      sibling = sibling->prev;

      oddness = row->odd;

      while (sibling != NULL)
        {
          oddness = row->odd;
          row = (ScwViewRow *) sibling->data;
          row->odd = !oddness;
          sibling = sibling->prev;
        }

    }

  if (priv->vadjustment == NULL)
    {
      return;
    }

  if (priv->scroll_on_append
      && priv->vadjustment->upper < GTK_WIDGET(user_data)->allocation.height)
    {
      priv->should_scroll = TRUE;
    }

  if (priv->scroll_on_append 
      && priv->vadjustment->value == 
                        priv->vadjustment->upper - priv->vadjustment->page_size)
    {
      scw_view_scroll_to_end (SCW_VIEW(user_data));
    }
 
}

static
void model_row_deleted     (GtkTreeModel *treemodel,
                             GtkTreePath *path,
                             gpointer user_data)
{
  gboolean oddness;
  gint *indices;
  GList *l;
  ScwViewRow *row;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (treemodel != NULL);
  g_assert (user_data != NULL);
  g_assert (SCW_IS_VIEW (user_data));

  if (gtk_tree_path_get_depth(path) > 1)
    {
      g_warning ("Skipping path at depth %i\n", gtk_tree_path_get_depth(path));
      return;
    }
  
  priv = SCW_VIEW_GET_PRIVATE (SCW_VIEW (user_data));

  if (priv->model == NULL)
    {
      g_warning ("Got false signal or in inconsistent state! Model is NULL");
      return;
    }

  indices = gtk_tree_path_get_indices(path);


  l = g_list_nth (priv->first_row, indices[0]);
  row = (ScwViewRow *) l->data;

  priv->first_row = g_list_remove(priv->first_row, row);
  priv->last_row = g_list_last(priv->first_row);

  scw_view_row_destroy(row);

  l = g_list_nth (priv->first_row, MIN(indices[0], g_list_length(priv->first_row)-1));

  if (l == NULL)
    {
      gtk_widget_queue_resize (GTK_WIDGET (user_data));
      return;
    }

  row = (ScwViewRow *) l->data;
  row->odd = decide_oddness_for_item(l);

  l = l->prev;

  while (l != NULL)
    {
      oddness = row->odd;
      row = (ScwViewRow *) l->data;
      row->odd = !oddness;
      l = l->prev;
    }

  gtk_widget_queue_resize (GTK_WIDGET (user_data));
 
}

enum
{
  SCW_TAG_ACTION,
  SCW_TAG_ICON
};


static
gboolean _scw_parse_action (ScwView *view,
                            const gchar *markup,
                            PangoAttribute **attr,
                            gchar **clean_str)
{
  gboolean inside_tag, end_tag_found;
  gint start_index, end_index;
  gint id_start_index, id_end_index;
  gint end_tag_start_index;
  gint start_tag_start_index, start_tag_end_index;
  gint escaped_length, end_escaped_length;
  gint tag_type;
  gchar *p;
  gchar *id_i        = NULL,
  *id_end_i          = NULL,
  *activatable_i     = NULL,
  *activatable_end_i = NULL,
  *start_tag_i       = NULL,
  *tail_i            = NULL;
  gchar *id;
  gchar *activatable;
  gchar *start_tag;
  gchar *head;
  gchar *tail = NULL;
  PangoAttribute *new_attr;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));

  priv = SCW_VIEW_GET_PRIVATE (SCW_VIEW (view));
  
  p = (gchar *)markup;
  
  /* Find the first tag */
  start_index = 0;
  start_tag_start_index = 0;
  end_tag_start_index = 0;
  inside_tag = FALSE;
  escaped_length = 0;
  end_escaped_length = 0;
  tag_type = -1;

  while (p != NULL)
    {
      if (g_str_has_prefix (p, "<"))
        {
          if (g_str_has_prefix (p, ACTION_TAG))
            {
              tag_type = SCW_TAG_ACTION;
              start_tag_i = p;
              break;
            }
          else if (g_str_has_prefix (p, ICON_TAG))
            {
              tag_type = SCW_TAG_ICON;
              start_tag_i = p;
              break;
            }
          inside_tag = TRUE;
        }
      else if (inside_tag && g_str_has_prefix (p, ">"))
        {
          start_index--;
          inside_tag = FALSE;
        }
      else if (g_str_has_prefix (p, "&lt;"))
        {
          escaped_length += 3;
        }
      else if (g_str_has_prefix (p, "&gt;"))
        {
          escaped_length += 3;
        }
      else if (g_str_has_prefix (p, "&amp;"))
        {
          escaped_length += 4;
        }
      else if (g_str_has_prefix (p, "&quot;"))
        {
          escaped_length += 5;
        }
      else if (g_str_has_prefix (p, "&apos;"))
        {
          escaped_length += 5;
        }
      p = g_utf8_next_char (p);
      if (!inside_tag)
        {
          start_index++;
        }
      if (g_utf8_strlen (p, -1) == 0)
        {
          return FALSE;
        }
      start_tag_start_index++;
    }

  start_index = MAX(0, start_index);
  start_tag_start_index = MAX(0, start_tag_start_index);
  
  if (p == NULL || g_utf8_strlen (p, -1) == 0)
    {
      return FALSE;
    }

  /* Get the id */
  end_index = start_tag_start_index;
  while (p != NULL)
    {
      if (g_str_has_prefix (p, "id="))
        {
          if (p[3] == '\'' || p[3] == '\"')
            {
              p = p + 4;
              end_index += 4;
              id_i = p;
              break;
            }
          else
            {
              p = p + 3;
              end_index += 3;
              id_i = p;
              break;
            }
        }
      p = g_utf8_next_char (p);
      end_index++;
     }
     
  if (p == NULL || g_utf8_strlen (p, -1) == 0)
    {
      return FALSE;
    }

  id_start_index = end_index;
  id_end_index = id_start_index;

  while (p != NULL)
    {
      if (p[0] == '\'' || p[0] == '\"')
        {
          id_end_index = end_index;
          id_end_i = p;
          break;
        }
      p = g_utf8_next_char (p);
      end_index++;
     }

  if (p == NULL || g_utf8_strlen (p, -1) == 0)
    {
      return FALSE;
    }

  /* Find the tag end */
  while (p != NULL)
    {
      if (g_str_has_prefix (p, ">"))
        {
          end_index++;
          start_tag_end_index = end_index;
          activatable_i = g_utf8_next_char (p);
          tail_i = p + 1;
          break;
        }
      p = g_utf8_next_char (p);
      end_index++;
     }

  if (p == NULL || g_utf8_strlen (p, -1) == 0)
    {
      return FALSE;
    }

  if (tag_type == SCW_TAG_ICON)
    {
      goto skip;
    }

  /* Find the end tag */
  end_tag_found = FALSE;
  while (p != NULL)
    {
      if (g_str_has_prefix (p, ACTION_END_TAG))
        {
          end_tag_found = TRUE;
          end_tag_start_index = end_index - 1;
          end_index += 8;
          tail_i = p + 9;
          activatable_end_i = p;
          break;
        }
      else if (g_str_has_prefix (p, "&lt;"))
        {
          end_escaped_length += 3;
        }
      else if (g_str_has_prefix (p, "&gt;"))
        {
          end_escaped_length += 3;
        }
      else if (g_str_has_prefix (p, "&amp;"))
        {
          end_escaped_length += 4;
        }
      else if (g_str_has_prefix (p, "&quot;"))
        {
          end_escaped_length += 5;
        }
      else if (g_str_has_prefix (p, "&apos;"))
        {
          end_escaped_length += 5;
        }
      p = g_utf8_next_char (p);
      end_index++;
     }

  if (!end_tag_found)
    {
      return FALSE;
    }

skip:

  head = g_strndup (markup, start_tag_i - markup);
  id = g_strndup (id_i, id_end_i - id_i);
  if (tail_i) tail = g_strdup (tail_i);

  if (tag_type == SCW_TAG_ACTION)
    {
      activatable = g_strndup (activatable_i,
                               activatable_end_i - activatable_i);
      new_attr = scw_attr_action_new (id);

      if (priv->action_attributes != NULL)
        {
          start_tag = g_strdup_printf (SCW_ACTION_START, priv->action_attributes);
        }
      else
        {
          start_tag = g_strdup_printf (SCW_ACTION_START, _SCW_ACTION_ATTRIBUTES);
        }
        
      *clean_str = g_strjoin (NULL, head,
                              start_tag,
                              activatable,
                              SCW_ACTION_END,
                              tail, NULL);

      new_attr->start_index = MAX(0, start_index - escaped_length);
      new_attr->end_index = start_index + g_utf8_strlen (activatable, -1)
                                        - escaped_length - end_escaped_length;

      g_free (start_tag);  
      g_free (activatable);  
    }
  else if (tag_type == SCW_TAG_ICON)
    {
      PangoRectangle logical_rect;
      
      logical_rect.x = 0;
      logical_rect.y = -1 * priv->icon_size * PANGO_SCALE;
      logical_rect.width  = priv->icon_size * PANGO_SCALE;
      logical_rect.height = priv->icon_size * PANGO_SCALE;

      new_attr = pango_attr_shape_new_with_data (&logical_rect, &logical_rect,
                                                 g_strdup(id), 
                                               (PangoAttrDataCopyFunc) g_strdup,
                                                 g_free);

/* FIXME: This leaks, why? */

      *clean_str = g_strjoin (NULL, head,
                              " ",
                              tail, NULL);

      new_attr->start_index = MAX(0, start_index - escaped_length);
      new_attr->end_index = start_index + 1 - escaped_length;
    }
  else
    {
      return FALSE;
    }

  *attr = new_attr;

  g_free (head);  
  g_free (id);  
  g_free (tail);  
  
  return TRUE;
}

static
GList *scw_parse_actions (ScwView *view,
                          const gchar *markup,
                          gchar **clean_str)
{
  gchar *str;
  gchar *tmp;
  GList *actions;
  PangoAttribute *attr;
  
  SCW_DEBUG_ENTER

  actions = NULL;
  attr = NULL;
  
  str = g_strdup (markup);
  
  
  while (TRUE)
    {
      if (_scw_parse_action (view, str, &attr, &tmp))
        {
          actions = g_list_append (actions, attr);
          g_free (str);
          str = tmp;
        }
      else
        {
          break;
        }
    }
  
  *clean_str = str;
  
  return actions;
}

static
gboolean decide_oddness_for_item(GList *item)
{
  gboolean odd;
  ScwViewRow *next_row;
  ScwViewRow *prev_row;

  SCW_DEBUG_ENTER

  /* Default is even */
  odd = FALSE;

  if (item == NULL)
    {
      return FALSE;
    }

  next_row = NULL;
  prev_row = NULL;

  if (item->next != NULL)
    {
      next_row = (ScwViewRow *) item->next->data;
    }  
  if (item->prev != NULL)
    {
      prev_row = (ScwViewRow *) item->prev->data;
    }
  
  
  if (next_row != NULL)
    {
      /* Always use the oddness of the rows below if possible, those are more
       *  likely to be visible to the user and thus better to be stable.
       */
      odd = !next_row->odd;
    }
  else if (prev_row != NULL)
    {
      odd = !prev_row->odd;
    }

  return odd;
}

static
void model_row_changed      (GtkTreeModel *treemodel,
                             GtkTreePath *path,
                             GtkTreeIter *iter,
                             gpointer user_data)
{
  gint columns, i;
  gint intvar;
  gint *indices;
  GList *l;
  gchar *strvar;
  PangoLayout *layout;
  ScwViewRow *row;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (treemodel != NULL);
  g_assert (user_data != NULL);
  g_assert (SCW_IS_VIEW (user_data));

  if (gtk_tree_path_get_depth(path) > 1)
    {
      g_warning ("Skipping path at depth %i\n", gtk_tree_path_get_depth(path));
      return;
    }
  
  priv = SCW_VIEW_GET_PRIVATE (SCW_VIEW (user_data));

  if (priv->model == NULL)
    {
      g_warning ("Got false signal or in inconsistent state! Model is NULL");
      return;
    }

  columns = gtk_tree_model_get_n_columns (priv->model);

  indices = gtk_tree_path_get_indices(path);
  
  l = g_list_nth (priv->first_row, indices[0]);
  row = (ScwViewRow *)l->data;

  if (row == NULL)
    {
      strvar = gtk_tree_path_to_string (path);
      g_warning ("Inconsistent state! Row for path %s is NULL", strvar);
      g_free (strvar);
      strvar = NULL;
      return;
    }
    
  for (i = 0; i < columns; i++)
    {
      GType type = gtk_tree_model_get_column_type (priv->model, i);

      strvar = NULL;

      if (!priv->column_visible[i])
        {
          /* Skip invisible columns */
        }
      else if (type == GDK_TYPE_PIXBUF)
        {
          ScwSize *pixbuf_size;
          GdkPixbuf *pixbuf = NULL;

          gtk_tree_model_get (priv->model, iter, i, &pixbuf, -1);
          
          if (GDK_IS_PIXBUF(pixbuf))
            {
              GList *item;
              
              item = g_list_nth (row->pixbufs, i);
              
              if (item->data == NULL)
                {
                  item->data = (gpointer) g_new0 (ScwSize, 1);
                }

              pixbuf_size = (ScwSize *) item->data;
                
              pixbuf_size->height = gdk_pixbuf_get_height (pixbuf);
              pixbuf_size->width = gdk_pixbuf_get_width (pixbuf);
            }
        }      
      else if (type == SCW_TYPE_TIMESTAMP)
        {
          gchar *timestamp;
          layout = PANGO_LAYOUT (g_list_nth_data (row->layouts, i));
          gtk_tree_model_get (priv->model, iter, i, &intvar, -1);
          timestamp = scw_create_timestamp (SCW_VIEW(user_data), intvar),
          pango_layout_set_markup (layout, timestamp, -1);

          g_free (timestamp);

        }
      else if (type == G_TYPE_INT)
        {
          gchar *intstr;
          layout = PANGO_LAYOUT (g_list_nth_data (row->layouts, i));
          gtk_tree_model_get (priv->model, iter, i, &intvar, -1);
          intstr = g_strdup_printf ("%i", intvar);
          pango_layout_set_markup (layout, intstr, -1);

          g_free (intstr);

        }
      else if (type == SCW_TYPE_PRESENCE)
        {
          ScwSlot *slot;
          
          layout = PANGO_LAYOUT (g_list_nth_data (row->layouts, i));
          slot = (ScwSlot *) g_list_nth_data (row->slots, i);

          /* Clear the layout attributes */
          pango_layout_set_attributes (layout, NULL);
          g_list_free (slot->embedded_icons);
          slot->embedded_icons = NULL;

          gtk_tree_model_get (priv->model, iter, i, &strvar, -1);
          if (strvar != NULL)
            {
              gchar *stripped;
              gchar *clean;
              GList *actions;
              GList *a;
              PangoAttrList *list;
              
              actions = NULL;
              actions = scw_parse_actions (SCW_VIEW(user_data),
                                           strvar,
                                           &stripped);
              
              for (a = actions; a; a = a->next)
                {
                  PangoAttribute *attr = (PangoAttribute *)a->data;
                  
                  if (attr->klass->type == PANGO_ATTR_SHAPE)
                    {
                      slot->embedded_icons = g_list_append(slot->embedded_icons,
                                                          attr);
                    }
                }
              
              if (pango_parse_markup (stripped, -1, 0,
                                      &list, &clean,
                                      NULL, NULL))
                {
                  for (a = actions; a; a = a->next)
                    {
                      PangoAttribute *attr = (PangoAttribute *)a->data;
                      if (attr->klass->type == SCW_ATTR_ACTION)
                        {
                          pango_attr_list_insert (list, attr);
                        }
                    }
                  pango_layout_set_text (layout, clean, -1);
                  pango_layout_set_attributes (layout, list);
                  pango_attr_list_unref (list);
                  pango_layout_set_wrap (layout, PANGO_WRAP_WORD);
                  pango_layout_set_alignment (layout, priv->presence_alignment);
                  
                  g_free (clean);
                }

              g_list_free (actions);
              g_free (stripped);
              g_free (strvar);
            }
          
        }
      else if (type == G_TYPE_STRING)
        {
          ScwSlot *slot;
          
          layout = PANGO_LAYOUT (g_list_nth_data (row->layouts, i));
          slot = (ScwSlot *) g_list_nth_data (row->slots, i);
 
           /* Clear the layout attributes */
          pango_layout_set_attributes (layout, NULL);
          g_list_free (slot->embedded_icons);
          slot->embedded_icons = NULL;
         
          gtk_tree_model_get (priv->model, iter, i, &strvar, -1);

          if (strvar != NULL)
            {
              gchar *stripped;
              gchar *clean;
              GList *actions;
              GList *a;
              PangoAttrList *list;
              ScwSlot *slot;
              
              slot = (ScwSlot *) g_list_nth_data (row->slots, i);

              actions = NULL;
              actions = scw_parse_actions (SCW_VIEW(user_data),
                                           strvar,
                                           &stripped);
              
              for (a = actions; a; a = a->next)
                {
                  PangoAttribute *attr = (PangoAttribute *)a->data;

                  if (attr->klass->type == PANGO_ATTR_SHAPE)
                    {
                      slot->embedded_icons = g_list_append(slot->embedded_icons,
                                                           attr);
                    }
                }

              if (pango_parse_markup (stripped, -1, 0,
                                      &list, &clean,
                                      NULL, NULL))
                {
                  for (a = actions; a; a = a->next)
                    {
                      pango_attr_list_insert (list, (PangoAttribute *) a->data);
                    }

                  pango_layout_set_text (layout, clean, -1);
                  pango_layout_set_attributes (layout, list);
                  pango_attr_list_unref (list);
                  pango_layout_set_wrap (layout, PANGO_WRAP_WORD_CHAR);

                  g_free (clean);
                }
              g_list_free (actions);
              g_free (stripped);
              g_free (strvar);
            }
        }
      else if (type == SCW_TYPE_ROW_COLOR)
        {
          /* Silence the error below */
        }
      else
        {
          g_print ("Column %i type %i is not supported\n", i, (gint)type);
        }
    }

  /* Make sure the screen is updated
   * FIXME: Do this only if the insert was really affecting visible stuff
   */
  gtk_widget_queue_resize (GTK_WIDGET (user_data));

}

/* Creates a timestamp */
static
gchar *
scw_create_timestamp (ScwView *view, time_t stamp)
{
  gchar buf[512];
  struct tm *timeinfo;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));

  priv = SCW_VIEW_GET_PRIVATE (view);
  
  if (stamp == 0)
    {
      time (&stamp);
    }
  
  timeinfo = localtime (&stamp);

  if (priv->timestamp_format != NULL)
    {
      strftime (buf, sizeof(buf), priv->timestamp_format, timeinfo);
    }
  else
    {
      strftime (buf, sizeof(buf), _SCW_TIMESTAMP_FORMAT, timeinfo);
    }

  return g_strdup (buf);
}

ScwViewRow * scw_view_row_new ()
{
  ScwViewRow *row;
  
  SCW_DEBUG_ENTER

  row = g_new (ScwViewRow, 1);
  
  row->layouts = NULL;
  row->pixbufs = NULL;
  row->slots = NULL;
  row->selections = NULL;
  row->target_width = 0;
  row->calculated_height = 0;
  row->y = 0;
  row->odd = FALSE;
  row->age = _SCW_ROW_AGE;

  return row;
}

static
void scw_view_row_destroy (ScwViewRow *row)
{
  GList *l;

  SCW_DEBUG_ENTER

  if (row == NULL)
    {
      return;
    }

  for (l = row->layouts; l; l = l->next)
    {
      if (l->data != NULL)
        {
          if (G_IS_OBJECT (l->data))
            {
              g_object_unref (G_OBJECT(l->data));
              l->data = NULL;
            }
        }
    }

  for (l = row->pixbufs; l; l = l->next)
    {
      if (l->data != NULL)
        {
          g_free (l->data);
          l->data = NULL;
        }
    }

  for (l = row->slots; l; l = l->next)
    {
      if (l->data != NULL)
        {
          ScwSlot *slot = l->data;
          
          g_list_free (slot->embedded_icons);
          slot->embedded_icons = NULL;
          
          g_free (l->data);
          l->data = NULL;
        }
    }
  
  for (l = row->selections; l; l = l->next)
    {
      if (l->data != NULL)
        {
          g_free (l->data);
          l->data = NULL;
        }
    }

  g_list_free (row->layouts);
  row->layouts = NULL;
  g_list_free (row->pixbufs);
  row->pixbufs = NULL;
  g_list_free (row->slots);
  row->slots = NULL;
  g_list_free (row->selections);
  row->selections = NULL;

  g_free (row);
  row = NULL;
}

static
ScwViewSelection *scw_view_selection_new ()
{
  ScwViewSelection *selection;
  
  SCW_DEBUG_ENTER

  selection = g_new (ScwViewSelection, 1);
  
  selection->start_index = 0;
  selection->end_index = 0;
  
  return selection;
}

static
void scw_view_row_calculate_height (ScwView *view, ScwViewRow *row)
{
  gboolean must_free;
  gint row_padding, column_spacing;
  gint columns, i, j;
  gint wrap_columns;
  ScwSlot *slot;
  GdkRectangle *prev_slot;
  GdkRectangle *pslot;
  PangoLayout *layout;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));

  priv = SCW_VIEW_GET_PRIVATE (view);
  
  columns = gtk_tree_model_get_n_columns (priv->model);
  
  gtk_widget_style_get (GTK_WIDGET(view),
                        "row-padding", &row_padding,
                        "column-spacing", &column_spacing,
                        NULL);

  row->calculated_height = 0;

  wrap_columns = 0;
  slot = NULL;

  /* Look how many columns we need to wrap for calculations */
  for (i = 0; i < columns; i++)
    {
      GType type = gtk_tree_model_get_column_type (priv->model, i);
      
      if (type == G_TYPE_STRING)
        {
          wrap_columns++;
        }
    }

  for (i = 0; i < columns; i++)
    {
      GType type = gtk_tree_model_get_column_type (priv->model, i);
      must_free = FALSE;

      slot = (ScwSlot *) g_list_nth_data (row->slots, i);

      if (i == 0)
        {
          /* Create an origo slot, this way we don't need to special-case
           * the column 0 further down
           */
          prev_slot = g_new0 (GdkRectangle, 1);
          must_free = TRUE;
        }
      else
        {
          prev_slot = (GdkRectangle *) g_list_nth_data (row->slots, i-1);
          
          /* FIXME: What to do on errors?
           * This won't do anything sensible but won't segfault either
           */
          if (prev_slot == NULL)
            {
              prev_slot = g_new0 (GdkRectangle, 1);
              must_free = TRUE;
            }
        }

      if (!priv->column_visible[i])
        {
      
          if (i == 0)
            {
              slot->x = row_padding;
            }
          else
            {
              slot->x = prev_slot->x + prev_slot->width;
            }

          slot->y      = row_padding;
          slot->width  = 0;
          slot->height = 0;

          if (must_free)
            {
              g_free (prev_slot);
            }
          continue;
        }
      
      if (priv->fold_column[i])
        {
          slot->x = prev_slot->x;
          slot->y = prev_slot->y + prev_slot->height + row_padding;
        }
      else if (i > 0 && priv->fold_column[i-1])
        {
          j = i-1;
          
          slot->x = 0;
          
          while (j > 0 && priv->fold_column[j])
            {
              pslot = (GdkRectangle *) g_list_nth_data (row->slots, j);
              slot->x = MAX (slot->x, pslot->x + pslot->width);
              j--;
            }

          pslot = (GdkRectangle *) g_list_nth_data (row->slots, j);
          slot->x = MAX (slot->x, pslot->x + pslot->width);

          slot->x += column_spacing;
          slot->y = row_padding;
        }
      else
        {
          slot->x = prev_slot->x + prev_slot->width;
          
          if (i == 0)
            {
              /* Padding is all around the row */
              slot->x += row_padding;
            }
          else
            {
              slot->x += column_spacing;
            }
            
          slot->y = row_padding;
        }

      slot->height = 0;
      slot->width = 0;
      
      if (type == G_TYPE_STRING)
        {
          layout = PANGO_LAYOUT (g_list_nth_data (row->layouts, i));

          pango_layout_set_wrap (layout, PANGO_WRAP_WORD_CHAR);

          if (priv->fold_column[i])
            {
              pango_layout_set_width (layout,
                ((row->target_width - slot->x - row_padding) / wrap_columns)
                 * PANGO_SCALE);
            }
          else
            {
              pango_layout_set_width (layout,
                ((row->target_width - slot->x) / wrap_columns) * PANGO_SCALE);
            }
          wrap_columns--;
                        
          if (g_utf8_strlen(pango_layout_get_text(layout), -1) > 0)
            {
              pango_layout_get_pixel_size (layout, &slot->width, &slot->height);
            }
          else
            {
              slot->x = prev_slot->x + prev_slot->width;
              slot->y = prev_slot->y + prev_slot->height;
            }
            
        }
      else if (type == GDK_TYPE_PIXBUF)
        {
          ScwSize *pixbuf_size;
          pixbuf_size = (ScwSize *) g_list_nth_data (row->pixbufs, i);
          if (pixbuf_size != NULL)
            {
              slot->width = pixbuf_size->width;
              slot->height = pixbuf_size->height;
            }
          else
            {
              slot->x = prev_slot->x + prev_slot->width;
              slot->y = prev_slot->y + prev_slot->height;
            }
        }
      else if (type == SCW_TYPE_PRESENCE)
        {
          layout = PANGO_LAYOUT (g_list_nth_data (row->layouts, i));

          pango_layout_set_width (layout, -1);

          if (g_utf8_strlen(pango_layout_get_text(layout), -1) > 0)
            {
              pango_layout_get_pixel_size (layout, &slot->width, &slot->height);
            }
          else
            {
              slot->x = prev_slot->x + prev_slot->width;
              slot->y = prev_slot->y + prev_slot->height;
            }
            
          if (priv->align_presences)
            {
              slot->width = slot->width + priv->icon_size *
                                            g_list_length(slot->embedded_icons);
              slot->width = MAX (slot->width, priv->fixed_width[i]);
              priv->fixed_width[i] = slot->width;
            }

        }
      else if (type == SCW_TYPE_TIMESTAMP)
        {
          layout = PANGO_LAYOUT (g_list_nth_data (row->layouts, i));

          if (g_utf8_strlen(pango_layout_get_text(layout), -1) > 0)
            {
              pango_layout_get_pixel_size (layout, &slot->width, &slot->height);
            }
          else
            {
              slot->x = prev_slot->x + prev_slot->width;
              slot->y = prev_slot->y + prev_slot->height;
            }
        }
        
      row->calculated_height = MAX (row->calculated_height, slot->y + slot->height);
      
      if (must_free)
        {
          g_free (prev_slot);
        }
    }
   row->calculated_height += row_padding;
   
   if (row->calculated_height == row_padding)
    {
      g_warning ("Uh, row with only padding?\n");
    }
   
}

static
void scw_view_do_selection (ScwView *view)
{
  gboolean row_mode, go_down;
  GList *l;
  GdkRectangle row_area;
  ScwViewRow *row;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));

  priv = SCW_VIEW_GET_PRIVATE (view);

  row = NULL;
  
  row = scw_view_row_at_height(view, priv->selection_start.y);

  if (row == NULL)
    {
      return;
    }
  
  row_mode = FALSE;
  
  row_area.x = 0;
  row_area.y = row->y;
  row_area.width = row->target_width;
  row_area.height = row->calculated_height;

  priv->selected_rows = g_list_append (priv->selected_rows, row);

  go_down = TRUE;
  
  /* Check if the selection ends outside this row */
  if (priv->selection_end.y > row_area.y + row_area.height)
    {
      row_mode = TRUE;
      go_down  = TRUE;
    }
  else if (priv->selection_end.y < row_area.y)
    {
      row_mode = TRUE;
      go_down  = FALSE;
    }

  if (row_mode)
    {
    
      l = g_list_find(priv->first_row, row);
      if (go_down)
        {
          l = l->next;
        }
      else
        {
          l = l->prev;
        }
        
      scw_view_row_select_all (view, row);

      while (l != NULL)
        {
          row = (ScwViewRow *)l->data;

          if (go_down
              && priv->selection_end.y < row->y)
            {
              break;
            }
          else if (!go_down
                   && priv->selection_end.y > row->y + row->calculated_height)
            {
              break;
            }

          scw_view_row_select_all (view, row);

          priv->selected_rows = g_list_append (priv->selected_rows, row);

          if (go_down)
            {
              l = l->next;
            }
          else
            {
              l = l->prev;
            }
        }
    }
  else
    {
      scw_view_row_select (view, row);
    }
    
}

static
gboolean filter_prelight (PangoAttribute *attribute,
                          gpointer data)
{
  SCW_DEBUG_ENTER

  if (attribute == data)
    {
      return FALSE;
    }
    
  return TRUE;
}

static gboolean scw_view_leave_notify   (GtkWidget          *widget,
                                         GdkEventCrossing   *event)
{
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (widget != NULL);
  g_assert (SCW_IS_VIEW (widget));

  priv = SCW_VIEW_GET_PRIVATE (widget);

  /* Remove prelight */
  if (priv->prelight_attr != NULL
      && priv->prelight_layout != NULL)
    {
      PangoAttrList *list;
      PangoAttrList *new_list;
      
      list = pango_layout_get_attributes (priv->prelight_layout);
      new_list = pango_attr_list_filter (list,
                                         (PangoAttrFilterFunc) filter_prelight,
                                         priv->prelight_attr);
      pango_layout_set_attributes (priv->prelight_layout, new_list);
      pango_attr_list_unref (new_list);
      priv->prelight_attr = NULL;
      priv->prelight_layout = NULL;
      
      gtk_widget_queue_draw (widget);
    }
  
  return FALSE;
}

static gboolean scw_view_motion   (GtkWidget        *widget,
                                   GdkEventMotion   *event)
{
  gint columns, adj;
  ScwViewRow *row;
  ScwView *view;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (widget != NULL);
  g_assert (SCW_IS_VIEW (widget));

  view = SCW_VIEW (widget);
  priv = SCW_VIEW_GET_PRIVATE (view);

  if (priv->model == NULL)
  {
    return FALSE;
  }

  if (priv->vadjustment == NULL)
    {
      adj = 0;
    }
  else
    {
      adj = (gint) priv->vadjustment->value;
    }

  /* Remove prelight */
  if (priv->prelight_attr != NULL
      && priv->prelight_layout != NULL)
    {
      PangoAttrList *list;
      PangoAttrList *new_list;
      
      list = pango_layout_get_attributes (priv->prelight_layout);
      new_list = pango_attr_list_filter (list,
                                         (PangoAttrFilterFunc) filter_prelight,
                                         priv->prelight_attr);
      pango_layout_set_attributes (priv->prelight_layout, new_list);
      pango_attr_list_unref (new_list);
      priv->prelight_attr = NULL;
      priv->prelight_layout = NULL;
      
    }


  if (event->state & GDK_BUTTON1_MASK
      && !priv->selecting && !priv->panning)
    {
      gint distance, x, y;
      gint limit_x, limit_y;
      gint tolerance = 10;

      /* Relative coordinates with pan_start as the origo */
      x = (gint)event->x - priv->pan_start.x;
      y = priv->pan_start.y - (gint)event->y; /* y axis inverted */

      /* Distance to origo */
      distance = sqrt(pow(x, 2) + pow(y, 2));
      
      if (distance < tolerance)
        {
          gdk_window_get_pointer (widget->window, NULL, NULL, 0);
          return FALSE;
        }


      /* Right-hand upper limit function */
      limit_y = x / 2;

      /* Right-hand left limit function */
      limit_x = (y*-1) / 2;
  
      if (priv->pan_start.x <= (gint)event->x)
        {
          if (limit_x < x && limit_y > y)
            {
              priv->selecting = TRUE;
            }
        }
      else
        {
          if (limit_x > x && abs(limit_y) < y)
            {
              priv->selecting = TRUE;
            }
        }
        
      /* Pan by default */
      if (!priv->selecting)
        {
          priv->panning   = TRUE;
        }
        
    }

  if (priv->selecting)
    {
      GList *l;
      
      columns = gtk_tree_model_get_n_columns (priv->model);
      
      l = priv->selected_rows;
      while (l != NULL)
        {

          row = (ScwViewRow *)l->data;

          if (row != NULL)
            {
              GList *s;
              GdkRectangle area;
              
              area.x = 0;
              area.y = row->y;
              area.width = widget->allocation.width;
              area.height = row->calculated_height;
              
              gdk_window_invalidate_rect(widget->window, &area, FALSE);
              
              s = row->selections;
              
              while (s != NULL)
                {
                  ScwViewSelection *selection = (ScwViewSelection *) s->data;
                  
                  if (selection != NULL)
                    {
                      selection->start_index = 0;
                      selection->end_index = 0;
                    }
                  s = s->next;
                }

            }
          l = l->next;
        }      

      g_list_free(priv->selected_rows);
      priv->selected_rows = NULL;

      g_free(priv->selection);
      priv->selection = NULL;

      priv->selection_box.width = ABS((gint) event->x - priv->selection_box.x);
      priv->selection_box.height = ABS(adj + (gint) event->y - priv->selection_box.y);

      priv->selection_end.x = (gint) event->x;
      priv->selection_end.y = (gint) event->y + adj;

      priv->selection_end.x = MAX(0, priv->selection_end.x);
      priv->selection_end.y = MAX(0, priv->selection_end.y);
      
      scw_view_do_selection (view);

      l = priv->selected_rows;
      while (l != NULL)
        {

          row = (ScwViewRow *)l->data;

          if (row != NULL)
            {
              GdkRectangle area;
              
              area.x = 0;
              area.y = row->y;
              area.width = widget->allocation.width;
              area.height = row->calculated_height;
              
              gdk_window_invalidate_rect(widget->window, &area, FALSE);

            }
          l = l->next;
        }      

      gdk_window_get_pointer (widget->window, NULL, NULL, 0);
    }
  else if (priv->panning)
    {
      gint delta;
      
      delta = (gint) event->y - priv->pan_start.y;
      
      if (delta != 0)
        {
          gint new_adjustment = 0;

          /* Decide which way the user wants to pan */
          if (!priv->invert_panning) {
              /* Cap at the end of rows */
              new_adjustment = MIN((gint) priv->vadjustment->upper 
                                   - (gint) priv->vadjustment->page_size,
                                   adj + delta);
          } else {
              new_adjustment = CLAMP(adj - delta,
                                     (gint) priv->vadjustment->lower,
                                     (gint) priv->vadjustment->upper
                                     - (gint) priv->vadjustment->page_size);
          }
                                      
          gtk_adjustment_set_value(priv->vadjustment, 
                                   new_adjustment);
        }
        
      priv->pan_start.y = (gint) event->y;
      gdk_window_get_pointer (widget->window, NULL, NULL, 0);
    }
  else
    {
      ScwViewRow *row;

      if (priv->active_prelight.height != 0)
        {
          gdk_window_invalidate_rect(widget->window, &priv->active_prelight, FALSE);

          priv->active_prelight.x = 0;
          priv->active_prelight.y = 0;
          priv->active_prelight.width = 0;
          priv->active_prelight.height = 0;
        }
      
      row = scw_view_row_at_height(view, (gint) event->y + adj);

      if (scw_view_activation_at_coord (view, row,
                                        (gint) event->x,
                                        (gint) event->y + adj,
                                        SCW_ACTIVATION_PRELIGHT))
        {
          
          priv->active_prelight.x = 0;
          priv->active_prelight.y = row->y - adj;
          priv->active_prelight.width = row->target_width;
          priv->active_prelight.height = row->calculated_height;
          
          gdk_window_invalidate_rect(widget->window, &priv->active_prelight, FALSE);
        }

      gdk_window_get_pointer (widget->window, NULL, NULL, 0);
    }
    
  /* As we use MOTION_HINT flag, we need to specifically request
   * for more events by calling gdk_window_pointer. A hackish, but
   * suprisingly functional system
   */
   
   return FALSE;
}



static gboolean scw_view_button_press   (GtkWidget        *widget,
                                         GdkEventButton   *event)
{
  gint adj;
  GList *l;
  ScwViewRow *row;
  ScwView *view;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (widget != NULL);
  g_assert (SCW_IS_VIEW (widget));

  view = SCW_VIEW (widget);
  priv = SCW_VIEW_GET_PRIVATE (view);


  if (priv->model == NULL)
  {
    return FALSE;
  }

  if (priv->vadjustment == NULL)
    {
      adj = 0;
    }
  else
    {
      adj = (gint) priv->vadjustment->value;
    }

  gtk_widget_grab_focus (widget);

  if (event->button == 3)
    {
      ScwViewRow *row;
      
      row = scw_view_row_at_height(view, (gint) event->y + adj);
      
      if (!scw_view_activation_at_coord (view, row,
                                         (gint) event->x,
                                         (gint) event->y + adj,
                                         SCW_ACTIVATION_CONTEXT))
        {
          /* It's a global context request */
          g_signal_emit_by_name (G_OBJECT (view), "context-request",
                                  NULL, NULL,
                                  (gint) event->x, (gint) event->y);
        }
      return TRUE;
    }


  l = priv->selected_rows;
              
  while (l != NULL)
    {
      row = (ScwViewRow *)l->data;
      
      if (row != NULL)
        {
          GList *s;
          
          s = row->selections;
          while (s != NULL)
            {
              ScwViewSelection *selection = (ScwViewSelection *)s->data;

              if (selection == NULL)
                {
                  break;
                }

              selection->start_index = 0;
              selection->end_index = 0;
              s = s->next;
            }
        }
      l = l->next;
    }      

  g_list_free(priv->selected_rows);
  priv->selected_rows = NULL;
  g_free(priv->selection);
  priv->selection = NULL;
  
  switch (priv->interaction)
  {
    case SCW_INTERACTION_PAN:
      priv->panning = TRUE;
      break;
    case SCW_INTERACTION_SMART_PAN:
      /* Both being FALSE is interpeted as autodetect in _motion */
      priv->panning = FALSE;
      priv->selecting = FALSE;
      break;
    case SCW_INTERACTION_SELECT:
    default:
      priv->selecting = TRUE;
      break;
  }

  priv->selection_box.x = (gint) event->x;
  priv->selection_box.y = (gint) event->y + adj;
  priv->selection_box.width = 0;
  priv->selection_box.height = 0;

  priv->selection_start.x = (gint) event->x;
  priv->selection_start.y = (gint) event->y + adj;
  priv->selection_end.x = (gint) event->x;
  priv->selection_end.y = (gint) event->y + adj;

/*
  gtk_widget_queue_draw (widget);
*/

  priv->pan_start.x = (gint) event->x;
  priv->pan_start.y = (gint) event->y;
  
  return TRUE;
}

static gboolean scw_view_button_release (GtkWidget        *widget,
                                         GdkEventButton   *event)
{
  GtkClipboard *clipboard;
  ScwView *view;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (widget != NULL);
  g_assert (SCW_IS_VIEW (widget));

  view = SCW_VIEW (widget);
  priv = SCW_VIEW_GET_PRIVATE (view);

  if (priv->model == NULL)
  {
    return FALSE;
  }

  priv->selecting = FALSE;
  priv->panning = FALSE;
  priv->pan_start.x = 0;
  priv->pan_start.y = 0;

  g_free (priv->selection);
  priv->selection = NULL;

  if (priv->selection_box.width < 5 
      && priv->selection_box.height < 5)
    {
      if (event->state & GDK_BUTTON1_MASK)
        {
          ScwViewRow *row;
          
          row = scw_view_row_at_height(view, priv->selection_box.y);
          
          scw_view_activation_at_coord (view,
                                        row,
                                        priv->selection_box.x,
                                        priv->selection_box.y,
                                        SCW_ACTIVATION_ACTIVATE);
        }
    }
  else
    {
      scw_view_do_selection (view);
    }

  /* Set PRIMARY_SELECTION. Just for the record, I hate it */
  if (priv->selection != NULL)
    {
      clipboard = gtk_clipboard_get (GDK_SELECTION_PRIMARY);
      gtk_clipboard_set_text (clipboard, priv->selection, -1);
    }

  gtk_widget_grab_focus (widget);
/* FIXME: UNNECCESSARY DRAWING !!!
  gtk_widget_queue_draw (widget);
*/  
  return TRUE;
}

static gboolean scw_view_key_release    (GtkWidget *widget,
                                         GdkEventKey *event)
{
  ScwView *view;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (widget != NULL);
  g_assert (SCW_IS_VIEW (widget));

  view = SCW_VIEW (widget);
  priv = SCW_VIEW_GET_PRIVATE (view);

  switch (event->keyval)
    {
      case GDK_C:
      case GDK_c:
        if (event->state & GDK_CONTROL_MASK)
          {
            if (priv->selection != NULL)
              {
                GtkClipboard *clipboard = 
                  gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
                gtk_clipboard_set_text (clipboard, priv->selection, -1);
              }
          }
        
    }
  return FALSE;
}

static
void scw_view_row_select (ScwView *view,
                          ScwViewRow *row)
{
  gboolean exact;
  gint columns, i, swap;
  gint row_padding, column_spacing;
  gint trailing, length;
  gint used_width, start_height;
  gint start_index, end_index;
  gint row_y;
  gchar *selection = NULL;
  gchar *tmp = NULL;
  const gchar *str;
  gchar *buf = NULL;
  GdkRectangle area;
  GdkRectangle intersection;
  GdkRectangle *slot;
  PangoLayoutIter *start_iter;
  PangoLayoutIter *end_iter;
  PangoLayout *layout;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));
 
  priv = SCW_VIEW_GET_PRIVATE (view);

  columns = gtk_tree_model_get_n_columns (priv->model);
  
  gtk_widget_style_get (GTK_WIDGET(view),
                        "row-padding", &row_padding,
                        "column-spacing", &column_spacing,
                        NULL);

  used_width = row_padding;
  start_height = row_padding;
  selection = NULL;

  if (priv->selection_box.width < 0)
    {
      area.x = priv->selection_box.x + priv->selection_box.width;
      area.width = ABS (priv->selection_box.width);
    }
  else
    {
      area.x = priv->selection_box.x;
      area.width = priv->selection_box.width;
    }

  row_y = row->y;

  if (priv->selection_box.height < 0)
    {
      area.y = priv->selection_box.y + priv->selection_box.height - row_y;
      area.height = ABS (priv->selection_box.height);
    }
  else
    {
      area.y = priv->selection_box.y - row_y;
      area.height = priv->selection_box.height;
    }

  area.x = MAX(0, area.x);
  area.y = MAX(0, area.y);
  area.width = MAX(0, area.width);
  area.height = MAX(0, area.height);

  row_y = row->y;

  for (i = 0; i < columns; i++)
    {
      GType type = gtk_tree_model_get_column_type (priv->model, i);

      if (type == G_TYPE_STRING
          || type == SCW_TYPE_PRESENCE
          || type == SCW_TYPE_TIMESTAMP)
        {
        
          slot = (GdkRectangle *) g_list_nth_data (row->slots, i);
          layout = PANGO_LAYOUT (g_list_nth_data (row->layouts, i));
          
          if (PANGO_IS_LAYOUT(layout)
              && gdk_rectangle_intersect (slot, &area, &intersection))
            {
              gint pad;
              
              start_iter = NULL;
              end_iter = NULL;
              start_index = 0;
              end_index = 0;
              trailing = 0;
              length = 0;
              
              
              pad = priv->fold_column[i] ? column_spacing : 0;
              
              /* If the selection starts before this layout,
               * select the beginning of the text too
               */
              if (priv->selection_start.x < slot->x + pad)
                {
                  start_index = 0;
                  trailing = 0;
                }
              else              
                {
                  exact = pango_layout_xy_to_index (layout,
                        (priv->selection_start.x - slot->x) * PANGO_SCALE,
                        (priv->selection_start.y - row_y - slot->y) * PANGO_SCALE,
                                                    &start_index, &trailing);
                }

              exact = pango_layout_xy_to_index (layout,
                   (priv->selection_end.x - slot->x) * PANGO_SCALE,
                   (priv->selection_end.y - row_y - slot->y) * PANGO_SCALE,
                                                &end_index, &trailing);

              if (start_index > end_index)
                {
                  swap = start_index;
                  start_index = end_index;
                  end_index = swap;

                  if (start_index > 0)
                    {
                      start_index--;
                    }
                }

              length = end_index - start_index;
              if (length > 0)
                {
                  ScwViewSelection *selection_data;
                  
                  selection_data = (ScwViewSelection *)
                                g_list_nth_data (row->selections, i);
                                
                  selection_data->start_index = start_index;
                  selection_data->end_index = end_index + 1;
                  length++;

                  /* Don't mess with memory unless the selection is finished */
                  if (!priv->selecting)
                    {
                      str = pango_layout_get_text (layout);
                    
                      g_free (buf);
                      
                      buf = g_malloc0 (length);
                      
                      strncpy (buf, g_utf8_offset_to_pointer (str, start_index), length);

                      if (selection == NULL)
                        {
                          selection = g_strdup (buf);
                        }
                      else
                        {
                          tmp = g_strdup (selection);
                          g_free (selection);
                      if (priv->selection_column_separator == NULL)
                        {
                           selection = g_strjoin (_SELECTION_COLUMN_SEPARATOR,
                                                  tmp,
                                                  buf,
                                                  NULL);
                        }
                      else
                        {
                           selection = g_strjoin (
                                               priv->selection_column_separator,
                                                  tmp,
                                                  buf,
                                                  NULL);
                        }
                          g_free (tmp);
                          tmp = NULL;
                        }
                    }
               }
             else
               {
                  ScwViewSelection *selection;
                  
                  selection = (ScwViewSelection *)
                                g_list_nth_data (row->selections, i);
                                
                  selection->start_index = 0;
                  selection->end_index = 0;
                  
               }
          }
       }
    }
      
    if (!priv->selecting && selection != NULL)
      {
        if (priv->selection == NULL)
          {
            priv->selection = g_strdup (selection);
          }
        else
          {
            tmp = g_strdup (priv->selection);
            g_free (priv->selection);
            if (priv->selection_row_separator == NULL)
              {
                priv->selection = g_strjoin (_SELECTION_ROW_SEPARATOR,
                                              tmp,
                                              selection,
                                              NULL);
              }
            else
              {
                priv->selection = g_strjoin (priv->selection_row_separator,
                                              tmp,
                                              selection,
                                              NULL);
              }
            g_free (tmp);
            tmp = NULL;
          }
        
      }    

    g_free (buf);
    buf = NULL;
    g_free (selection);
    selection = NULL;

}

static
void scw_view_row_select_all (ScwView *view,
                              ScwViewRow *row)
{
  gint columns, i;
  gint row_padding, column_spacing;
  gint length;
  gint start_index, end_index;
  gchar *selection = NULL;
  gchar *tmp = NULL;
  const gchar *str;
  gchar *buf = NULL;
  PangoLayoutIter *end_iter;
  PangoLayout *layout = NULL;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));
 
  priv = SCW_VIEW_GET_PRIVATE (view);

  columns = gtk_tree_model_get_n_columns (priv->model);
  
  gtk_widget_style_get (GTK_WIDGET(view),
                        "row-padding", &row_padding,
                        "column-spacing", &column_spacing,
                        NULL);


  for (i = 0; i < columns; i++)
    {
      GType type = gtk_tree_model_get_column_type (priv->model, i);

      if (type == G_TYPE_STRING
          || type == SCW_TYPE_PRESENCE
          || type == SCW_TYPE_TIMESTAMP)
        {
          layout = PANGO_LAYOUT (g_list_nth_data(row->layouts, i));

          start_index = 0;
          end_iter = pango_layout_get_iter(layout);

          while (pango_layout_iter_next_char (end_iter));
          end_index = pango_layout_iter_get_index (end_iter);
          
          length = end_index - start_index;
             
             
          if (length > 0)
            {
              ScwViewSelection *selection_data;
              
              selection_data = (ScwViewSelection *)
                            g_list_nth_data (row->selections, i);
                            
              selection_data->start_index = start_index;
              selection_data->end_index = end_index;

              /* Don't mess with memory unless the selection is finished */
              if (!priv->selecting)
                {
                  str = pango_layout_get_text (layout);
                
                  g_free (buf);
                  
                  buf = g_malloc0 (length);
                  
                  strncpy (buf, g_utf8_offset_to_pointer (str, start_index),
                           length);

                  if (selection == NULL)
                    {
                      selection = g_strdup (buf);
                    }
                  else
                    {
                      tmp = g_strdup (selection);
                      g_free (selection);
                      if (priv->selection_column_separator == NULL)
                        {
                           selection = g_strjoin (_SELECTION_COLUMN_SEPARATOR,
                                                  tmp,
                                                  buf,
                                                  NULL);
                        }
                      else
                        {
                           selection = g_strjoin (
                                               priv->selection_column_separator,
                                                  tmp,
                                                  buf,
                                                  NULL);
                        }
                      g_free (tmp);
                      tmp = NULL;
                    }
                }
            }
          else
            {
              ScwViewSelection *selection;
              
              selection = (ScwViewSelection *)
                            g_list_nth_data (row->selections, i);
                            
              selection->start_index = 0;
              selection->end_index = 0;
              
            }
          pango_layout_iter_free (end_iter);
        }
    }
      
    if (!priv->selecting && selection != NULL)
      {
        if (priv->selection == NULL)
          {
            priv->selection = g_strdup (selection);
          }
        else
          {
            tmp = g_strdup (priv->selection);
            g_free (priv->selection);
            if (priv->selection_row_separator == NULL)
              {
                priv->selection = g_strjoin (_SELECTION_ROW_SEPARATOR,
                                              tmp,
                                              selection,
                                              NULL);
              }
            else
              {
                priv->selection = g_strjoin (priv->selection_row_separator,
                                              tmp,
                                              selection,
                                              NULL);
              }
            g_free (tmp);
            tmp = NULL;
          }
        
      }    

    g_free (buf);
    buf = NULL;
    g_free (selection);
    selection = NULL;

}

/* Activate stuff at the coordinates.
 * Returns TRUE if something was activated, FALSE if not
 */

/* Prelight stuff at the coordinates.
 * Returns TRUE if something was prelight, FALSE if not
 */

/* Context request for action tags at the coordinates.
 * If something activatable is at x,y, emits "context-request" signal with
 * values from that activatable, else with NULL values.
 * Returns TRUE if something activatable was at x,y, FALSE if not
 */

/* @activation defines what will happen if we hit the jackpot:
 * SCW_ACTIVATION_ACTIVATE == activate the item
 * SCW_ACTIVATION_PRELIGHT == prelight the item
 * SCW_ACTIVATION_CONTEXT  == context request on the item
 */

static
gboolean scw_view_activation_at_coord (ScwView *view,
                                       ScwViewRow *row,
                                       gint x,
                                       gint y,
                                       guint activation_type)
{
  gboolean found;
  gint i, max;
  gint index, trailing;
  gint start, end;
  gint event_x, event_y, adj;
  GdkRectangle *slot;
  PangoAttrList *list;
  PangoAttrIterator *aiter;
  PangoLayout *layout;
  GtkWidget *widget;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));
 
  widget = GTK_WIDGET (view);
  priv = SCW_VIEW_GET_PRIVATE (view);
  
  if (row == NULL)
    {
      return FALSE;
    }

  if (row->slots == NULL)
    {
      return FALSE;
    }

  if (priv->vadjustment == NULL)
    {
      adj = 0;
    }
  else
    {
      adj = (gint) priv->vadjustment->value;
    }

  event_x = x;
  event_y = y - adj;
  y = y - row->y;
  
  /* Find the slot */
  
  slot = NULL;
  max = g_list_length (row->slots);
  for (i = 0; i < max; i++)
    {
      slot = (GdkRectangle *) g_list_nth_data (row->slots, i);
      if (slot == NULL)
        {
          break;
        }
      
      if (slot->x <= x && slot->x + slot->width >= x
          && slot->y <= y && slot->y + slot->height >= y)
        {
          break;
        }
      slot = NULL;
    }

  if (slot == NULL)
    {
      return FALSE;
    }
  
  /* Find the index */

  if (GDK_IS_PIXBUF (g_list_nth_data (row->layouts, i)))
    {
      return FALSE;
    }
  
  layout = PANGO_LAYOUT (g_list_nth_data (row->layouts, i));
  
  if (layout == NULL)
    {
      return FALSE;
    }

  if (!pango_layout_xy_to_index (layout,
        (x - slot->x) * PANGO_SCALE,
        (y - slot->y) * PANGO_SCALE,
        &index, &trailing))
    {
      return FALSE;
    }
  /* Find the attribute */

  list = pango_layout_get_attributes (layout);
  
  found = FALSE;
  
  aiter = pango_attr_list_get_iterator (list);

  while (!found)
    {
      
      pango_attr_iterator_range (aiter, &start, &end);
      
      if (start <= index && index <= end)
        {
          GSList *attributes;
          GSList *i;
          
          attributes = pango_attr_iterator_get_attrs (aiter);
          
          for (i = attributes; i; i = i->next)
            {
              PangoAttribute *attr = (PangoAttribute *) i->data;
              if (attr->klass->type == SCW_ATTR_ACTION)
                {
                	ScwAttrAction *action = (ScwAttrAction *) attr;
                	gchar *uri;
                  switch (activation_type)
                    {
                    	
                      case SCW_ACTIVATION_ACTIVATE:
                        uri = g_strndup (g_utf8_offset_to_pointer (
                                                 pango_layout_get_text (layout),
                                                 attr->start_index),
                                           attr->end_index - attr->start_index);
                        
                        g_signal_emit_by_name (G_OBJECT (view), "activate",
                                               action->id,
                                               uri); 
                        g_free (uri);

                        break;
                      case SCW_ACTIVATION_PRELIGHT:
                        priv->prelight_attr = pango_attr_foreground_new (
                                  widget->style->text[GTK_STATE_PRELIGHT].red,
                                  widget->style->text[GTK_STATE_PRELIGHT].green,
                                  widget->style->text[GTK_STATE_PRELIGHT].blue);
                        
                        priv->prelight_attr->start_index = attr->start_index;
                        priv->prelight_attr->end_index = attr->end_index;
                        
                        pango_attr_list_insert (list, priv->prelight_attr);
                        pango_layout_set_attributes (layout, list);
                        
                        priv->prelight_layout = layout;

                        break;
                      case SCW_ACTIVATION_CONTEXT:
                        uri = g_strndup (g_utf8_offset_to_pointer (
                                                 pango_layout_get_text (layout),
                                                 attr->start_index),
                                           attr->end_index - attr->start_index);
                        
                        g_signal_emit_by_name (G_OBJECT (view),
                                               "context-request",
                                               action->id,
                                               uri,
                                               event_x, event_y); 
                        g_free (uri);

                        break;
                      default:
                        break;
                    }
                  
                  found = TRUE;
                }
            }
        }

      if (found || !pango_attr_iterator_next(aiter))
        {
          break;
        }  
    }

  pango_attr_iterator_destroy (aiter);
  
  if (found)
    {
/*      gtk_widget_queue_draw (GTK_WIDGET (view));*/
      return TRUE;
    }
  else
    {
      return FALSE;
    }
}

static
void scw_view_activated (ScwView *view,
                         gchar *id,
                         gchar *action_data)
{
  ScwViewPrivate *priv;

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));
 
  priv = SCW_VIEW_GET_PRIVATE (view);
  
}

static ScwViewRow *scw_view_row_at_height(ScwView *view, gint y)
{
  GList *l;
  GList *target;
  ScwViewRow *row;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_assert (view != NULL);
  g_assert (SCW_IS_VIEW (view));

  priv = SCW_VIEW_GET_PRIVATE (view);

  /* Find the row */

  l = priv->first_row;
  target = priv->first_row;
  while (l != NULL)
    {
      row = (ScwViewRow *)l->data;

      if (row == NULL)
        {
          break;
        }
        
      if (row->y > y)
        {
          break;
        }
      
      target = l;
      l = l->next;
    }
  
  if (target == NULL)
    {
      return NULL;
    }
  
  row = (ScwViewRow *)target->data;

  return row;
}

/****
 * Public API functions should be kept below this
 */

/**
 * scw_view_scroll_to_row:
 * @view: The #ScwView.
 * @path: A string noting the row that should be scrolled to, starting from "0"
 *
 * Scrolls the view so that the row denoted by @path is visible at the top
 * or as close to top of the view as possible.
 *
 */
void scw_view_scroll_to_row (ScwView *view, const gchar *ppath)
{
  gint scroll_height;
  gint *indices;
  GList *item;
  GtkTreePath *path;
  ScwViewRow *row;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_return_if_fail (ppath != NULL);
  g_return_if_fail (view != NULL);
  g_return_if_fail (SCW_IS_VIEW (view));

  priv = SCW_VIEW_GET_PRIVATE (view);

  path = gtk_tree_path_new_from_string (ppath);

  indices = gtk_tree_path_get_indices(path);
  item = g_list_nth (priv->first_row, indices[0]);
  
  gtk_tree_path_free (path);
  
  if (item == NULL)
    {
      return;
    }
  
  row = (ScwViewRow *) item->data;
  
  /* Don't pass the bottom, or the top */
  scroll_height = MIN((gint) priv->vadjustment->upper
                       - (gint) priv->vadjustment->page_size,
                      row->y);

  scroll_height = MAX(0, scroll_height);
  
  gtk_adjustment_set_value(priv->vadjustment, (gdouble) scroll_height);
}

/**
 * scw_view_scroll_to_end:
 * @view: The #ScwView.
 *
 * Scrolls the view so that the last row is visible at the bottom of the view.
 *
 */
void scw_view_scroll_to_end (ScwView *view)
{
  ScwViewRow *row;
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_return_if_fail (view != NULL);
  g_return_if_fail (SCW_IS_VIEW (view));

  priv = SCW_VIEW_GET_PRIVATE (view);

  if (priv->vadjustment == NULL)
    {
      return;
    }

  if (priv->last_row == NULL)
    {
      return;
    }

  row = (ScwViewRow *) priv->last_row->data;

  if (row == NULL)
    {
      return;
    }

  gtk_adjustment_set_value(priv->vadjustment, priv->vadjustment->upper - priv->vadjustment->page_size);
 

/*  priv->should_scroll = TRUE; */
}

/**
 * scw_view_set_column_foldable:
 * @view:     The #ScwView.
 * @column:   Number of the column to set the attribute for.
 * @foldable: Whether the column is foldable or not.
 *
 * Sets the column indicated to either be foldable or not foldable. Columns are
 * not foldable by default.
 * If a column is foldable, it will be rendered below the previous column. This
 * includes already folded columns (so if all columns are folded, they will be
 * rendered as a vertical list, row by row)
 *
 */
void scw_view_set_column_foldable (ScwView *view, gint column, gboolean foldable)
{
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_return_if_fail (view != NULL);
  g_return_if_fail (SCW_IS_VIEW (view));
  g_return_if_fail (column > 0);
  g_return_if_fail (column < MAX_COLUMNS);

  priv = SCW_VIEW_GET_PRIVATE (view);

  priv->fold_column[column] = foldable;
  
}

/**
 * scw_view_get_column_foldable:
 * @view:     The #ScwView.
 * @column:   Number of the column to get the attribute for.
 *
 * Gets the foldable attribute of the column indicated 
 *
 * Returns: Boolean value indicating whether the column is foldable or not.
 *
 */
gboolean scw_view_get_column_foldable (ScwView *view, gint column)
{
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_return_val_if_fail (view != NULL, FALSE);
  g_return_val_if_fail (SCW_IS_VIEW (view), FALSE);
  g_return_val_if_fail (column > 0, FALSE);
  g_return_val_if_fail (column < MAX_COLUMNS, FALSE);

  priv = SCW_VIEW_GET_PRIVATE (view);

  return priv->fold_column[column];
  
}

/**
 * scw_view_set_column_visible:
 * @view:    The #ScwView.
 * @column:  Number of the column to set the attribute for.
 * @visible: Whether the column is visible or not.
 *
 * Sets the column indicated to be visible or not visible. Invisible rows are
 * hidden completely and do not affect the rendering in any way.
 *
 */
void scw_view_set_column_visible (ScwView *view, gint column, gboolean visible)
{
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_return_if_fail (view != NULL);
  g_return_if_fail (SCW_IS_VIEW (view));
  g_return_if_fail (column >= 0);
  g_return_if_fail (column < MAX_COLUMNS);

  priv = SCW_VIEW_GET_PRIVATE (view);

  priv->column_visible[column] = visible;
  
  gtk_widget_queue_resize (GTK_WIDGET(view));
  gtk_widget_queue_draw (GTK_WIDGET(view));
  
}

/**
 * scw_view_get_column_visible:
 * @view:    The #ScwView.
 * @column:  Number of the column to get the attribute for.
 *
 * Gets the visibility of the column indicated.
 *
 * Returns: Boolean value indicating whether the column is visible or not.
 *
 */
gboolean scw_view_get_column_visible (ScwView *view, gint column)
{
  ScwViewPrivate *priv;

  SCW_DEBUG_ENTER

  g_return_val_if_fail (view != NULL, FALSE);
  g_return_val_if_fail (SCW_IS_VIEW (view), FALSE);
  g_return_val_if_fail (column >= 0, FALSE);
  g_return_val_if_fail (column < MAX_COLUMNS, FALSE);

  priv = SCW_VIEW_GET_PRIVATE (view);

  return priv->column_visible[column];
}
