
#include <string.h>

#include <glib.h>

#include "scwsmileys.h"

enum {
    SCW_PARSE_OPEN_TAG,
    SCW_PARSE_CLOSE_TAG,
    SCW_PARSE_OPEN_CLOSE_TAG,
    SCW_PARSE_TEXT
};

typedef struct {
    guint type;
    GString *string;
} ScwParseChunk;

typedef struct {
    ScwSmileyTree *smiley_tree;
    GSList *open_tags;
    GSList *chunks;
} ScwParseContext;

static ScwParseChunk *
_parse_chunk_new (guint type, GString *string)
{
  ScwParseChunk *new = (ScwParseChunk *) g_new0 (ScwParseChunk, 1);
  new->type = type;
  new->string = string;
  return new;
}

static void
_parser_start_element (
  GMarkupParseContext *context,
  const gchar *element_name,
  const gchar **attribute_names,
  const gchar **attribute_values,
  gpointer user_data,
  GError **error)
{
  ScwParseContext *scw_context = (ScwParseContext *) user_data;
  GString *string;
  const gchar **i, **j;

  /* push tag on to the stack */
  scw_context->open_tags = g_slist_prepend (
    scw_context->open_tags, g_strdup (element_name));

  if (0 == strcmp (element_name, "scw"))
    return;

  string = g_string_new ("<");
  g_string_append (string, element_name);

  for (i = attribute_names, j = attribute_values; *i; i++, j++)
    {
      gchar *escaped = g_markup_escape_text (*j, -1);
      g_string_append_printf(string, " %s='%s'", *i, escaped);
      g_free (escaped);
    }

  g_string_append_c (string, '>');
  g_slist_append (scw_context->chunks,
    _parse_chunk_new (SCW_PARSE_OPEN_TAG, string));
}

static void
_parser_end_element (
  GMarkupParseContext *context,
  const gchar *element_name,
  gpointer user_data,
  GError **error)
{
  ScwParseContext *scw_context = (ScwParseContext *) user_data;
  ScwParseChunk *chunk;

  /* close recentmostly opened tag, assuming that the closing tag matches */
  g_assert (NULL != scw_context->open_tags);
  g_assert (NULL != scw_context->open_tags->data);
  g_free (scw_context->open_tags->data);
  scw_context->open_tags = g_slist_remove (
      scw_context->open_tags, scw_context->open_tags->data);

  if (0 == strcmp (element_name, "scw"))
    return;

  g_assert (NULL != scw_context->chunks);
  chunk = (ScwParseChunk *) g_slist_last (scw_context->chunks)->data;
  g_assert (NULL != chunk);

  if (SCW_PARSE_OPEN_TAG == chunk->type)
    {
      guint length = strlen (chunk->string->str);
      g_string_insert_c (chunk->string, length - 1, '/');
      chunk->type = SCW_PARSE_OPEN_CLOSE_TAG;
    }
  else
    {
      GString *string = g_string_new ("");
      g_string_printf (string, "</%s>", element_name);
      g_slist_append (scw_context->chunks,
        _parse_chunk_new (SCW_PARSE_CLOSE_TAG, string));
    }
}

static gboolean
_parse_context_has_open_tag (ScwParseContext *scw_context, gchar *element_name)
{
  GSList *i;

  g_assert (NULL != scw_context);
  g_assert (NULL != scw_context->open_tags);

  for (i = scw_context->open_tags; NULL != i; i = i->next)
    {
      g_assert (NULL != i->data);

      if (0 == strcmp (element_name, (gchar *) i->data))
        return TRUE;
    }

  return FALSE;
}

static void
_parser_text (
  GMarkupParseContext *context,
  const gchar *text,
  gsize text_len,
  gpointer user_data,
  GError **error)
{
  ScwParseContext *scw_context = (ScwParseContext *) user_data;
  GSList *maybes;
  GSList *i;
  GString *string;

  if (_parse_context_has_open_tag (scw_context, "action"))
    {
      scw_context->chunks = g_slist_append (scw_context->chunks,
        _parse_chunk_new (SCW_PARSE_TEXT, g_string_new (text)));
      return;
    }

  maybes = scw_smiley_tree_replace (scw_context->smiley_tree, text);
  string = g_string_new ("");

  for (i = maybes; NULL != i; i = i->next)
    {
      ScwMaybeSmiley *maybe = (ScwMaybeSmiley *) i->data;

      if (maybe->is_smiley)
        {
          g_string_append (string, maybe->str);
        }
      else
        {
          gchar *escaped = g_markup_escape_text (maybe->str, -1);
          g_string_append (string, escaped);
          g_free (escaped);
        }

      g_free (maybe->str);
      g_free (maybe);
    }

  g_slist_free (maybes);
  scw_context->chunks = g_slist_append (scw_context->chunks,
    _parse_chunk_new (SCW_PARSE_TEXT, string));
}

static void
_parser_error (
    GMarkupParseContext *context,
    GError *error,
    gpointer user_data)
  {
    g_error ("SCW parser error: %s\n", error->message);
  }

gchar *
scw_parse (gchar *s, ScwSmileyTree *smiley_tree)
{
  ScwParseContext scw_context;
  GMarkupParser parser;
  GMarkupParseContext *context;
  GString *result = g_string_new ("");
  GSList *i;

  scw_context.smiley_tree = smiley_tree;
  scw_context.open_tags = NULL;
  scw_context.chunks = NULL;

  parser.start_element = _parser_start_element;
  parser.end_element = _parser_end_element;
  parser.text = _parser_text;
  parser.passthrough = NULL;
  parser.error = _parser_error;

  context = g_markup_parse_context_new (&parser, 0, &scw_context, NULL);
  g_markup_parse_context_parse (context, "<scw>", -1, NULL);
  g_markup_parse_context_parse (context, s, -1, NULL);
  g_markup_parse_context_parse (context, "</scw>", -1, NULL);
  g_markup_parse_context_end_parse (context, NULL);
  g_markup_parse_context_free (context);

  for (i = scw_context.chunks; NULL != i; i = i->next)
    {
      ScwParseChunk *chunk = (ScwParseChunk *) i->data;
      g_string_append (result, chunk->string->str);
      g_string_free (chunk->string, TRUE);
      g_free (chunk);
    }

  g_slist_free (scw_context.chunks);
  return g_string_free (result, FALSE);
}

