/*
 * farsight-plugin.c - Source for farsight plugin infrastructure
 *
 * Farsight Voice+Video library
 * Copyright (c) 2005 INdT.
 *   @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 * Copyright 2005,2006 Collabora Ltd.
 * Copyright 2005,2006 Nokia Corp.
 *   @author Rob Taylor <rob.taylor@collabora.co.uk>.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "farsight-plugin.h"

#ifdef G_OS_WIN32
/* TODO verify prefix */
#define PLUGIN_PREFIX     ""
#define PLUGIN_PREFIX_LEN 0
#define PLUGIN_EXT        ".dll"
#else
#define PLUGIN_PREFIX     "lib"
#define PLUGIN_PREFIX_LEN 3
#define PLUGIN_EXT        ".so"
#endif

static void farsight_plugin_destroy (FarsightPlugin * plugin);
static FarsightPlugin *farsight_plugins_find_with_name (const gchar * name);

static GList *plugins = NULL;

/* TODO create function to handle search paths */
static guint search_path_count = 1;
static gchar *search_paths[] = {
  LIBDIR
};

FarsightPlugin *
farsight_plugin_load (const gchar * name)
{
  FarsightPlugin *plugin;
  guint i, len;
  GDir *dir;
  const gchar *file, *ext;
  gchar *path;
  GModule *handle;

  gboolean (*farsight_init_plugin) (FarsightPlugin *);

  g_return_val_if_fail (name != NULL && name[0] != '\0', NULL);

  plugin = farsight_plugins_find_with_name (name);
  if (plugin) {
    ++plugin->ref_count;
    return plugin;
  }

  for (i = 0; i < search_path_count; ++i) {
    g_message("looking for plugins in %s", search_paths[i]);
    dir = g_dir_open (search_paths[i], 0, NULL);
    if (dir == NULL)
      continue;

    while ((file = g_dir_read_name (dir)) != NULL) {
      if ((ext = strrchr (file, '.')) == NULL || strcmp (ext, PLUGIN_EXT) != 0) /* file is not a plugin */
        continue;

      len = strlen (file) - strlen (ext);

      /* the plugin XXX should have the name PLUGIN_PREFIX + XXX + PLUGIN_EXT
       * ex.:
       *   XXX plugin for linux: libXXX.so */
      if (strlen (name) == (len - PLUGIN_PREFIX_LEN) &&
          strncmp (file + PLUGIN_PREFIX_LEN, name, len)) {
        path = g_build_filename (search_paths[i], file, NULL);
        handle = g_module_open (path, 0);
        g_print ("opening module %s %s\n", path,
                 (handle != NULL) ? "succeeded" : g_module_error ());
        g_free (path);

        if (!handle)
          break;

        if (!g_module_symbol (handle,
                "farsight_init_plugin", (gpointer) & farsight_init_plugin)) {
          g_module_close (handle);
          g_print ("could not find init function in plugin\n");
          break;
        }

        plugin = g_new0 (FarsightPlugin, 1);
        plugin->info = NULL;
        plugin->handle = handle;
        plugin->name = g_strdup (name);
        plugin->ref_count = 1;

        plugins = g_list_append (plugins, plugin);
        break;
      }
    }

    g_dir_close (dir);
  }

  if (!plugin) {
    // TODO error handling (plugin not found or without init_function)
    g_print ("Plugin not found or without init_function\n");
    return NULL;
  }

  if (!farsight_init_plugin (plugin) || plugin->info == NULL) {
    /* TODO error handling (init error or no info defined) */
    g_print ("init error or no info defined");
    farsight_plugin_destroy (plugin);
    return NULL;
  }

  if (plugin->info->major_version != FARSIGHT_MAJOR_VERSION ||
      plugin->info->minor_version > FARSIGHT_MINOR_VERSION) {
    /* TODO error handling (version mismatch) */
    g_print ("version mismatch");
    farsight_plugin_destroy (plugin);
    return NULL;
  }

  if (plugin->info->load != NULL)
    plugin->info->load (plugin);

  return plugin;
}

gboolean
farsight_plugin_unload (FarsightPlugin * plugin)
{
  g_return_val_if_fail (plugin != NULL, FALSE);
  g_return_val_if_fail (plugin->ref_count > 0, FALSE);

  if (--plugin->ref_count > 0)
    return TRUE;

  if (plugin->info->unload != NULL)
    plugin->info->unload (plugin);

  farsight_plugin_destroy (plugin);

  return TRUE;
}

static void
farsight_plugin_destroy (FarsightPlugin * plugin)
{
  g_return_if_fail (plugin != NULL);

  plugins = g_list_remove (plugins, plugin);

  if (plugin->handle != NULL)
    g_module_close (plugin->handle);

  if (plugin->name != NULL)
    g_free (plugin->name);

  g_free (plugin);
}

static FarsightPlugin *
farsight_plugins_find_with_name (const gchar * name)
{
  FarsightPlugin *plugin;
  GList *l;

  for (l = plugins; l != NULL; l = l->next) {
    plugin = l->data;

    if (plugin->name != NULL && !strcmp (plugin->name, name))
      return plugin;
  }

  return NULL;
}

/**
 * farsight_plugin_get_name:
 * @plugin: a #FarsightPlugin to get it's name
 *
 * Returns the name of the given plugin plugin.
 *
 * Returns: a string to the name of the plugin, NULL if invalid.
 **/
G_CONST_RETURN gchar *
farsight_plugin_get_name (FarsightPlugin * plugin)
{
  g_return_val_if_fail (plugin != NULL, NULL);

  return plugin->name;
}

/**
 * farsight_plugin_get_description:
 * @plugin: a #FarsightPlugin to get it's description
 *
 * Returns the description of the given plugin plugin.
 *
 * Returns: a string containing the description, NULL if invalid.
 **/
G_CONST_RETURN gchar *
farsight_plugin_get_description (FarsightPlugin * plugin)
{
  g_return_val_if_fail (plugin != NULL, NULL);

  return plugin->info->description;
}

/**
 * farsight_plugin_get_version:
 * @plugin: a #FarsightPlugin to get it's version
 *
 * Returns the version of the given plugin plugin.
 *
 * Returns: a string containing the version, NULL if invalid.
 **/
G_CONST_RETURN gchar *
farsight_plugin_get_version (FarsightPlugin * plugin)
{
  g_return_val_if_fail (plugin != NULL, NULL);

  return plugin->info->version;
}

/**
 * farsight_plugin_get_author:
 * @plugin: a #FarsightPlugin to get it's author
 *
 * Returns the author of the given plugin plugin.
 *
 * Returns: a string containing the author, NULL if invalid.
 **/
G_CONST_RETURN gchar *
farsight_plugin_get_author (FarsightPlugin * plugin)
{
  g_return_val_if_fail (plugin != NULL, NULL);

  return plugin->info->author;
}

/**
 * farsight_plugin_get_homepage:
 * @plugin: a #FarsightPlugin to get it's homepage 
 *
 * Returns the homepage of the given plugin plugin.
 *
 * Returns: a string containing the homepage, NULL if invalid.
 **/
G_CONST_RETURN gchar *
farsight_plugin_get_homepage (FarsightPlugin * plugin)
{
  g_return_val_if_fail (plugin != NULL, NULL);

  return plugin->info->homepage;
}
