/*
 * videosink.c - Source for TpStreamEngineVideoSink
 * Copyright (C) 2006-2008 Collabora Ltd.
 * Copyright (C) 2006-2008 Nokia Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <stdlib.h>

#include <gst/interfaces/xoverlay.h>
#include <gst/video/video.h>
#include <gst/farsight/fs-element-added-notifier.h>

#include <tp-stream-engine.h>

#include <gtk/gtk.h>

#include "tp-stream-engine-signals-marshal.h"
#include "videosink.h"
#include "util.h"

G_DEFINE_ABSTRACT_TYPE (TpStreamEngineVideoSink, tp_stream_engine_video_sink,
    G_TYPE_OBJECT);

struct _TpStreamEngineVideoSinkPrivate
{
  GstElement *sink;
  GtkWidget *plug;

  GstPad *sinkpad;

  guint window_id;

  gboolean is_preview;

  gulong delete_event_handler_id;
  gulong embedded_handler_id;

  GstClockTime last_timestamp;
  gdouble framerate;
  GstElement *textoverlay;

  gulong caps_changed_id;

  GStaticMutex mutex;
  guint size_change_idle_id; /* mutex protected */
};

/* properties */
enum
{
  PROP_0,
  PROP_SINK,
  PROP_WINDOW_ID,
  PROP_IS_PREVIEW
};


/* signals */

enum
{
  PLUG_DELETED,
  SIZE_CHANGED,
  SIGNAL_COUNT
};


static guint signals[SIGNAL_COUNT] = {0};

static void
tp_stream_engine_video_sink_init (TpStreamEngineVideoSink *self)
{
  TpStreamEngineVideoSinkPrivate *priv = G_TYPE_INSTANCE_GET_PRIVATE (self,
      TP_STREAM_ENGINE_TYPE_VIDEO_SINK, TpStreamEngineVideoSinkPrivate);

  self->priv = priv;

  g_static_mutex_init (&priv->mutex);

  priv->last_timestamp = GST_CLOCK_TIME_NONE;
}


static gboolean
fps_qos_probe (GstPad *pad, GstEvent *event, gpointer user_data)
{
  TpStreamEngineVideoSink *self = TP_STREAM_ENGINE_VIDEO_SINK (user_data);
  gdouble proportion;
  GstClockTimeDiff diff;
  GstClockTime timestamp;

  if (GST_EVENT_TYPE (event) != GST_EVENT_QOS)
    return TRUE;

  gst_event_parse_qos (event, &proportion, &diff, &timestamp);

  /* Ignore late buffers */
  if (diff > 0)
    return TRUE;

  if (GST_CLOCK_TIME_IS_VALID (self->priv->last_timestamp))
    {
      GstClockTimeDiff diff = GST_CLOCK_DIFF(self->priv->last_timestamp,
          timestamp);
      gdouble diffs;

      if (diff > GST_SECOND)
        diff = GST_SECOND;

      diffs = (gdouble)diff / (gdouble)GST_SECOND;
      if (self->priv->framerate != 0)
        {
        self->priv->framerate = (1.0/diffs)*(diffs) + self->priv->framerate*(1-diffs);

        if (self->priv->textoverlay)
          {
            gchar *str = g_strdup_printf ("FPS: %f", self->priv->framerate);
            g_object_set (self->priv->textoverlay, "text", str, NULL);
            g_free (str);
          }
        else
          {
            g_debug("Last diff: %lld diffs: %f Instant fps: %f"
                " averaged over 1s: %f\n", diff, diffs, (1.0/diffs),
                self->priv->framerate);
          }
        }
      else
        {
          self->priv->framerate = 1.0/diffs;
        }
    }

  self->priv->last_timestamp = timestamp;

  return TRUE;
}

static gboolean
size_change_idle_cb (gpointer data)
{
  TpStreamEngineVideoSink *self = data;
  gint width, height;

  g_static_mutex_lock (&self->priv->mutex);
  self->priv->size_change_idle_id = 0;
  g_static_mutex_unlock (&self->priv->mutex);

  if (gst_video_get_size (self->priv->sinkpad, &width, &height))
    g_signal_emit (self, signals[SIZE_CHANGED], 0, width, height);

  return FALSE;
}

static void
caps_changed (GObject *obj, GParamSpec *spec, gpointer user_data)
{
  TpStreamEngineVideoSink *self = user_data;

  g_static_mutex_lock (&self->priv->mutex);
  if (!self->priv->size_change_idle_id)
    self->priv->size_change_idle_id =
        g_idle_add ((GSourceFunc) size_change_idle_cb, self);
  g_static_mutex_unlock (&self->priv->mutex);
}


static GstElement *
make_video_sink (TpStreamEngineVideoSink *self, gboolean is_preview)
{
  const gchar *videosink_name;
  GstElement *sink = NULL;
  GstElement *bin, *tmp;
  GstPad *pad;
  GstPad *ghostpad;

  if ((videosink_name = getenv ("PREVIEW_VIDEO_SINK")) ||
      (videosink_name = getenv ("FS_VIDEO_SINK")) ||
      (videosink_name = getenv ("FS_VIDEOSINK")))
    {
      g_debug ("making video sink with pipeline \"%s\"", videosink_name);
      sink = gst_parse_bin_from_description (videosink_name, TRUE, NULL);
    }
  else
    {
      if (sink == NULL)
        sink = gst_element_factory_make ("autovideosink", NULL);

      if (sink == NULL)
        sink = gst_element_factory_make ("xvimagesink", NULL);

      if (sink == NULL)
        sink = gst_element_factory_make ("ximagesink", NULL);
    }

  if (sink == NULL)
    {
      g_debug ("failed to make a video sink");
      return NULL;
    }

  if (g_object_has_property (sink, "force-aspect-ratio"))
    g_object_set (sink, "force-aspect-ratio", TRUE, NULL);


  g_debug ("made video sink element %s", GST_ELEMENT_NAME (sink));

  bin = gst_bin_new (NULL);

  if (!gst_bin_add (GST_BIN (bin), sink))
    {
      g_warning ("Could not add source bin to the pipeline");
      gst_object_unref (bin);
      gst_object_unref (sink);
      return NULL;
    }

  if (g_getenv ("STREAM_ENGINE_FPS") &&
      !strcmp (g_getenv ("STREAM_ENGINE_FPS"), "overlay"))
    {
      tmp = gst_element_factory_make ("textoverlay", NULL);

      if (tmp != NULL)
        {
          self->priv->textoverlay = tmp;
          g_debug ("linking textoverlay");
          if (!gst_bin_add (GST_BIN (bin), tmp))
            {
              g_warning ("Could not add textoverlay to the source bin");
              gst_object_unref (tmp);
              gst_object_unref (bin);
              return NULL;
            }
          if (!gst_element_link (tmp, sink))
            {
              g_warning ("Could not link sink and textoverlay elements");
              gst_object_unref (bin);
              return NULL;
            }
          sink = tmp;
        }
    }

  tmp = gst_element_factory_make ("videoscale", NULL);
  if (tmp != NULL)
    {
      g_object_set (tmp, "qos", FALSE, NULL);
      g_debug ("linking videoscale");
      if (!gst_bin_add (GST_BIN (bin), tmp))
        {
          g_warning ("Could not add videoscale to the source bin");
          gst_object_unref (tmp);
          gst_object_unref (bin);
          return NULL;
        }
      if (!gst_element_link (tmp, sink))
        {
          g_warning ("Could not link sink and videoscale elements");
          gst_object_unref (bin);
          return NULL;
        }
      sink = tmp;
    }

  tmp = gst_element_factory_make ("ffmpegcolorspace", NULL);
  if (tmp != NULL)
    {
      g_object_set (tmp, "qos", FALSE, NULL);
      g_debug ("linking ffmpegcolorspace");

      if (!gst_bin_add (GST_BIN (bin), tmp))
        {
          g_warning ("Could not add ffmpegcolorspace to the source bin");
          gst_object_unref (tmp);
          gst_object_unref (bin);
          return NULL;
        }
      if (!gst_element_link (tmp, sink))
        {
          g_warning ("Could not link sink and ffmpegcolorspace elements");
          gst_object_unref (bin);
          return NULL;
        }
      sink = tmp;
    }

  self->priv->sinkpad = gst_bin_find_unconnected_pad (GST_BIN (bin),
      GST_PAD_SINK);

  if (!pad)
    {
      g_warning ("Could not find unconnected sink pad in the source bin");
      gst_object_unref (bin);
      return NULL;
    }

  self->priv->caps_changed_id = g_signal_connect (self->priv->sinkpad,
      "notify::caps", G_CALLBACK (caps_changed), self);

  if (g_getenv ("STREAM_ENGINE_FPS"))
    gst_pad_add_event_probe (self->priv->sinkpad, G_CALLBACK (fps_qos_probe),
 self);

  ghostpad = gst_ghost_pad_new ("sink", self->priv->sinkpad);
  if (!gst_element_add_pad (bin, ghostpad))
    {
      gst_object_unref (ghostpad);
      g_warning ("Could not add sink ghostpad to the source bin");
      gst_object_unref (bin);
      return NULL;
    }

  sink = bin;

  return sink;
}


static gboolean
delete_event (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
  g_signal_emit (user_data, signals[PLUG_DELETED], 0);

  gtk_widget_hide (widget);
  return TRUE;
}

static void
embedded_event (GtkWidget *widget, gpointer user_data)
{
  gboolean embedded;

  g_object_get (widget, "embedded", &embedded, NULL);

  if (embedded)
    gtk_widget_show (widget);
}

static gboolean
expose_handler (GtkWidget *widget,
    GdkEventExpose *event,
    gpointer        user_data)
{
  return TRUE;
}

static GObject *
tp_stream_engine_video_sink_constructor (GType type,
    guint n_props,
    GObjectConstructParam *props)
{
  GObject *obj;
  TpStreamEngineVideoSink *self = NULL;

  obj = G_OBJECT_CLASS (tp_stream_engine_video_sink_parent_class)->constructor (type, n_props, props);

  self = (TpStreamEngineVideoSink *) obj;

  self->priv->sink = make_video_sink (self, self->priv->is_preview);

  if (self->priv->sink)
    gst_object_ref (self->priv->sink);

  self->priv->plug = gtk_plug_new (0);
  gtk_widget_set_size_request (self->priv->plug, 176, 144);
  gtk_widget_set_double_buffered (self->priv->plug, FALSE);
  gtk_widget_set_app_paintable (self->priv->plug, TRUE);
  self->priv->delete_event_handler_id = g_signal_connect (self->priv->plug,
      "delete-event", G_CALLBACK (delete_event), self);
  self->priv->embedded_handler_id = g_signal_connect (self->priv->plug,
      "embedded", G_CALLBACK (embedded_event), NULL);
  g_signal_connect (self->priv->plug, "expose-event",
      G_CALLBACK (expose_handler), NULL);

  self->priv->window_id = gtk_plug_get_id (GTK_PLUG (self->priv->plug));

  return obj;
}

static void
tp_stream_engine_video_sink_dispose (GObject *object)
{
  TpStreamEngineVideoSink *self = TP_STREAM_ENGINE_VIDEO_SINK (object);

  if (self->priv->sinkpad)
    {
      if (self->priv->caps_changed_id)
        g_signal_handler_disconnect (self->priv->sinkpad,
            self->priv->caps_changed_id);
      self->priv->caps_changed_id = 0;

      gst_object_unref (self->priv->sinkpad);
      self->priv->sinkpad = NULL;
    }

  if (self->priv->sink)
    {
      gst_object_unref (self->priv->sink);
      self->priv->sink = NULL;
    }

  g_static_mutex_lock (&self->priv->mutex);
  if (self->priv->size_change_idle_id)
    g_source_remove (self->priv->size_change_idle_id);
  self->priv->size_change_idle_id = 0;
  g_static_mutex_unlock (&self->priv->mutex);

  if (self->priv->delete_event_handler_id)
    {
      g_signal_handler_disconnect (self->priv->plug,
          self->priv->delete_event_handler_id);
      self->priv->delete_event_handler_id = 0;
    }

  if (self->priv->embedded_handler_id)
    {
      g_signal_handler_disconnect (self->priv->plug,
          self->priv->embedded_handler_id);
      self->priv->embedded_handler_id = 0;
    }

  if (self->priv->plug)
    {
      gtk_widget_destroy (self->priv->plug);
      self->priv->plug = NULL;
    }

  if (G_OBJECT_CLASS (tp_stream_engine_video_sink_parent_class)->dispose)
    G_OBJECT_CLASS (tp_stream_engine_video_sink_parent_class)->dispose (
        object);
}



static void
tp_stream_engine_video_sink_set_property  (GObject *object,
    guint property_id,
    const GValue *value,
    GParamSpec *pspec)
{
  TpStreamEngineVideoSink *self = TP_STREAM_ENGINE_VIDEO_SINK (object);

    switch (property_id)
    {
    case PROP_IS_PREVIEW:
      self->priv->is_preview = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
tp_stream_engine_video_sink_get_property  (GObject *object,
    guint property_id,
    GValue *value,
    GParamSpec *pspec)
{
  TpStreamEngineVideoSink *self = TP_STREAM_ENGINE_VIDEO_SINK (object);

    switch (property_id)
    {
    case PROP_SINK:
      g_value_set_object (value, self->priv->sink);
      break;
    case PROP_WINDOW_ID:
      g_value_set_uint (value, self->priv->window_id);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}


static void
tp_stream_engine_video_sink_class_init (TpStreamEngineVideoSinkClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (TpStreamEngineVideoSinkPrivate));
  object_class->dispose = tp_stream_engine_video_sink_dispose;
  object_class->constructor = tp_stream_engine_video_sink_constructor;
  object_class->set_property = tp_stream_engine_video_sink_set_property;
  object_class->get_property = tp_stream_engine_video_sink_get_property;

  g_object_class_install_property (object_class, PROP_SINK,
      g_param_spec_object ("sink",
          "The video sink element",
          "The GstElement that is used as videosink",
          GST_TYPE_ELEMENT,
          G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (object_class, PROP_WINDOW_ID,
      g_param_spec_uint ("window-id",
          "Window id",
          "The window id to Xembed",
          0, G_MAXUINT, 0,
          G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (object_class, PROP_IS_PREVIEW,
      g_param_spec_boolean ("is-preview",
          "Window id",
          "The window id to Xembed",
          FALSE,
          G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_STRINGS));

  signals[PLUG_DELETED] =
      g_signal_new ("plug-deleted",
          G_OBJECT_CLASS_TYPE (klass),
          G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
          0,
          NULL, NULL,
          g_cclosure_marshal_VOID__VOID,
          G_TYPE_NONE, 0);

  signals[SIZE_CHANGED] =
      g_signal_new ("size-changed",
          G_OBJECT_CLASS_TYPE (klass),
          G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
          0,
          NULL, NULL,
          tp_stream_engine_marshal_VOID__UINT_UINT,
          G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);
}

struct xid_data
{
  GstElement *src;
  gulong window_id;
  gboolean found;
};

static void
set_window_xid (gpointer data, gpointer user_data)
{
  GstXOverlay *xov = GST_X_OVERLAY (data);
  struct xid_data *xiddata = (struct xid_data *) user_data;

  if (GST_ELEMENT_CAST(xov) == xiddata->src) {
      gst_x_overlay_set_xwindow_id (xov, xiddata->window_id);
      xiddata->found = TRUE;
  }

  gst_object_unref (data);
}

gboolean
tp_stream_engine_video_sink_bus_sync_message (
    TpStreamEngineVideoSink *self,
    GstMessage *message)
{
  const GstStructure *s;
  GstIterator *it = NULL;
  struct xid_data xiddata;

  if (GST_MESSAGE_TYPE (message) != GST_MESSAGE_ELEMENT)
    return FALSE;

  s = gst_message_get_structure (message);
  if (!gst_structure_has_name (s, "prepare-xwindow-id"))
    return FALSE;

  xiddata.src = GST_ELEMENT (GST_MESSAGE_SRC (message));
  xiddata.window_id = self->priv->window_id;
  xiddata.found = FALSE;

  it = gst_bin_iterate_all_by_interface (GST_BIN (self->priv->sink),
      GST_TYPE_X_OVERLAY);
  while (gst_iterator_foreach (it, set_window_xid, &xiddata) ==
      GST_ITERATOR_RESYNC)
    gst_iterator_resync (it);
  gst_iterator_free (it);

  return xiddata.found;
}
