/*
 * This file is part of startup-time
 *
 * Copyright (C) 2004, 2005, 2007 Nokia Corporation. 
 *
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


/* ========================================================================= *
 * File: libstartup.c
 *
 * The library intercepts a couple of gtk functions and provides
 * startup time information for gtk based applications.
 *
 * History:
 *
 * 31-Mar-2005 Simo Piiroinen
 * - configurable: emitting gettimeofday() values to log file
 * 
 * 30-Mar-2005 Simo Piiroinen
 * - added mechanism to log custom timestamps
 * 
 * 15-Mar-2005 Simo Piiroinen
 * - some debugging messages written if SHOW_DEBUG_MESSAGES set to nonzero
 * 
 * 10-Mar-2005 Simo Piiroinen
 * - gtk idle time detection improved
 * - TOOL_VERS was missing from the log file header
 * - code cleaned up a bit
 *
 * 29-Nov-2004 Simo Piiroinen
 * - bugfix in sighandler installation loop
 * - settings loaded from /tmp/startup.env
 * 
 * 03-Nov-2004 Simo Piiroinen
 * - almost total rewrite
 *
 * 23-Sep-2004 Simo Piiroinen
 * - version bump -> 0.2.0
 *
 * 15-Jun-2004 Simo Piiroinen
 * - code cleanup
 * - tried to fix issues with 2.6 kernel & 64 bit jiffies
 * ========================================================================= */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <sys/times.h>
#include <stdlib.h>
#include <dlfcn.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <syslog.h>
#include <signal.h>

/* ========================================================================= *
 * Preprocessor Constants
 * ========================================================================= */

/* - - - - - - - - - - - - - - - - - - - *
 * Tool name & release version will be
 * written to log file header
 * - - - - - - - - - - - - - - - - - - - */

// TODO: remove these and use values from config.h instead.
#define TOOL_NAME "sp-startup-time"
#define TOOL_VERS "0.1.0"

/* - - - - - - - - - - - - - - - - - - - *
 * Private environment cache is
 * updated from this file
 * - - - - - - - - - - - - - - - - - - - */

#define ENV_FILE_PATH "/etc/libstartup.conf"

/* - - - - - - - - - - - - - - - - - - - *
 * Calculate number of entries in
 * fixed sized array
 * - - - - - - - - - - - - - - - - - - - */

#define numof(a) (sizeof(a)/sizeof*(a))

/* - - - - - - - - - - - - - - - - - - - *
 * These used to be defined in some
 * standard header...
 * - - - - - - - - - - - - - - - - - - - */

#ifndef FALSE
# define FALSE 0
#endif
#ifndef TRUE
# define TRUE 1
#endif


/* ========================================================================= *
 * Compile Time Configuration
 * ========================================================================= */


#define SHOW_TOD_IN_HEADER 01 /* gettimeofday() compatible timestamp
			       * for process creation time written to
			       * log header -> add to t_real column
			       * values to synchronize with other
			       * logs */

#define SHOW_TOD_IN_EVENT  01 /* gettimeofday() timestamps written for
			       * each log row */

/* ========================================================================= *
 * Debugging 
 * ========================================================================= */

#define SHOW_DEBUG_MESSAGES 0

#if 0
# define HERE fprintf(stderr, "%s:%d: %s ...\n", __FILE__,__LINE__,__FUNCTION__);
#else
# define HERE /* nix */
#endif

/* ========================================================================= *
 * Generic Utilities
 * ========================================================================= */

static void perror_msg(const char *what)
{
  fprintf(stderr, TOOL_NAME": %s: %s\n", what, strerror(errno));
}

/* ------------------------------------------------------------------------- *
 * debug_msg  --  emit formatted message text
 * ------------------------------------------------------------------------- */

#if SHOW_DEBUG_MESSAGES
# define debug_msg(format, ...) \
  do { \
    fprintf(stderr, TOOL_NAME": ");\
    fprintf(stderr, format, ##__VA_ARGS__); fflush(stderr); \
  } while(0)
// QUARANTINE static void debug_msg(const char *fmt, ...)
// QUARANTINE {
// QUARANTINE   if( fmt != 0 )
// QUARANTINE   {
// QUARANTINE     va_list va;
// QUARANTINE     va_start(va, fmt);
// QUARANTINE     vfprintf(stderr, fmt, va);
// QUARANTINE     va_end(va);
// QUARANTINE   }
// QUARANTINE   fflush(stderr);
// QUARANTINE }
#else
# define debug_msg(format, ...) do { } while(0)
#endif

/* ------------------------------------------------------------------------- *
 * white_p  --  string position at white space character
 * ------------------------------------------------------------------------- */

static inline int white_p(const char *s) { return (*s > 0) && (*s <= 32); }

/* ------------------------------------------------------------------------- *
 * black_p  --  string position at non white space character
 * ------------------------------------------------------------------------- */

static inline int black_p(const char *s) { return *(unsigned char *)s > 32; }

/* ------------------------------------------------------------------------- *
 * strip  --  compress white sequences, remove leading and trailing
 * ------------------------------------------------------------------------- */

static char *strip(char *str)
{
  char *si = str;
  char *di = str;
  
  while( white_p(si) ) { ++si; }
  
  for( ;; )
  {
    while( black_p(si) ) { *di++ = *si++; }
    while( white_p(si) ) { ++si; }
    if( *si == 0 ) break;
    *di++ = ' ';
  }
  
  *di = 0;
  return str;
}

/* ------------------------------------------------------------------------- *
 * token  --  split a string by separator character
 * ------------------------------------------------------------------------- */

static char *token(char **ppos, int sep)
{
  char *beg = *ppos;
  char *end = *ppos;
  
  for( ; *end != 0; ++end )
  {
    if( *end == sep ) 
    {
      // do not advance beyond end of string
      if( *end != 0 )
      {
	*end++ = 0;
      }
      break;
    }
  }
  *ppos = end;
  return strip(beg);
}

/* ------------------------------------------------------------------------- *
 * readfile  --  read file content to buffer
 * ------------------------------------------------------------------------- */

static int readfile(const char *path, char *dest, size_t size)
{
  int    done = -1;
  int    file = -1;
  
// Note: we are silent about any errors
  if( (file = open(path,O_RDONLY)) != -1 )
  {
    done = read(file, dest, size-1);
    close(file);
  }
  
  // make sure we return valid c-string
  dest[(done>0) ? done : 0] = 0;
  
  return done;
}


/* ========================================================================= *
 * Problems with setenv avoided by using private environment cache
 * ========================================================================= */


/* ------------------------------------------------------------------------- *
 * env_key  --  environment variables that we cache
 * ------------------------------------------------------------------------- */

static const char * const env_key[] =
{
  "STARTUP_LOG_DIR",   // directory to write logs to

  "STARTUP_LOG_FILE",  // override log path heuristics

  "STARTUP_WRITE_LOG", /* should the log be written
			* default  --  nope
			* yes      -- for gui apps
			* force    -- for all apps
			*/
  
  "STARTUP_IDLE_QUIT", /* if set, exit after gtk idle is reached */
  0
};
/* ------------------------------------------------------------------------- *
 * env_val  --  and the values of cached environmen variables
 * ------------------------------------------------------------------------- */

static char * env_val[numof(env_key)];

/* ------------------------------------------------------------------------- *
 * env_get  --  private replacement for getenv
 * ------------------------------------------------------------------------- */

static const char *env_get(const char *key)
{
  for( int i = 0; env_key[i]; ++i )
  {
    if( !strcmp(env_key[i], key) )
    {
      return env_val[i];
    }
  }
  return 0;
}

/* ------------------------------------------------------------------------- *
 * env_set  --  private replacement for setenv
 * ------------------------------------------------------------------------- */

static void env_set(const char *key, const char *val)
{
  for( int i = 0; env_key[i]; ++i )
  {
    if( !strcmp(env_key[i], key) )
    {
      free(env_val[i]);
      env_val[i] = val ? strdup(val) : 0;
      debug_msg("env set: %s=%s\n", env_key[i], env_val[i] ? env_val[i] : "<null>");
      break;
    }
  }
}

/* ------------------------------------------------------------------------- *
 * env_update_from_real  --  copy real environment to private cache
 * ------------------------------------------------------------------------- */

static void env_update_from_real(void)
{
  for( int i = 0; env_key[i]; ++i )
  {
    const char *val = getenv(env_key[i]);
    if (val != NULL)
    {
      free(env_val[i]);
      env_val[i] = strdup(val);
      debug_msg("env init: %s=%s\n", env_key[i], env_val[i] ? env_val[i] : "<null>");
    }
  }
}

/* ------------------------------------------------------------------------- *
 * env_update_from_file  --  update environment cache from a file
 * ------------------------------------------------------------------------- */

static void env_update_from_file(const char *path)
{
  /* FIXME: If user modifies the configuration file to be larger than 1 kB,
     the rest will currently just be silently ignored... */

  char temp[1024];
  
  if( readfile(path, temp, sizeof temp) != -1 )
  {
    char *pos = temp;
    char *row;
  
    while( *(row = token(&pos, '\n')) )
    {
      char *key = token(&row,'=');
      char *val = token(&row,0);
      env_set(key,val);
    }
  }
}


/* ========================================================================= *
 * Functions for determining process start time
 * ========================================================================= */


/* ------------------------------------------------------------------------- *
 * proc_starttime  --  get process creation time from /proc/self/stat
 * ------------------------------------------------------------------------- */

static double proc_starttime(void)
{
  double secs = 0;
  char temp[1024];

  if( readfile("/proc/self/stat", temp, sizeof temp) != -1 )
  {
    char *pos = temp;
    int   cnt = 20;

    for( pos += strcspn(pos, ")"); *pos; )
    {
      if( *pos++ <= ' ' )
      {
// QUARANTINE   debug_msg("[%d] %g\n", cnt, strtod(pos,0));
        if( --cnt == 0 )
        {
          secs = strtod(pos, 0) * 0.01;
          break;
        }
      }
    }
  }
  return secs;
}

/* ------------------------------------------------------------------------- *
 * fork_starttime  --  get process creation time for newly forken instance
 * ------------------------------------------------------------------------- */

static double fork_starttime(void)
{
  double secs = 0;

  int io[2];

  if( pipe(io) == -1 )
  {
    perror_msg("can't create pipe"); _exit(1);
  }

  fflush(0);

  int pid = fork();
  switch( pid )
  {
  case 0:
    secs = proc_starttime();
    close(io[0]);
    write(io[1], &secs, sizeof secs);
    close(io[1]);
    _exit(0);

  default:
    close(io[1]);
    waitpid(pid,&pid,0);
    read(io[0], &secs, sizeof secs);
    close(io[0]);
    break;

  case -1:
    close(io[0]);
    close(io[1]);
    perror_msg("can't fork");
    _exit(1);
  }
  return secs;
}

/* ------------------------------------------------------------------------- *
 * clock_uptime  --  get uptime from posix monotonic clock
 * ------------------------------------------------------------------------- */

// QUARANTINE static double clock_uptime(void)
// QUARANTINE {
// QUARANTINE   struct timespec t;
// QUARANTINE   clock_gettime(CLOCK_MONOTONIC,&t);
// QUARANTINE   return t.tv_sec + t.tv_nsec * 1e-9;
// QUARANTINE }


/* ------------------------------------------------------------------------- *
 * proc_uptime  --  get uptime from /proc/uptime
 * ------------------------------------------------------------------------- */

// QUARANTINE static double proc_uptime(void)
// QUARANTINE {
// QUARANTINE   double secs = 0;
// QUARANTINE   char temp[256];
// QUARANTINE
// QUARANTINE   if( readfile("/proc/uptime", temp, sizeof temp) != -1 )
// QUARANTINE   {
// QUARANTINE     secs = strtod(temp,0);
// QUARANTINE   }
// QUARANTINE
// QUARANTINE   return secs;
// QUARANTINE }


/* ========================================================================= *
 * Data Types
 * ========================================================================= */


/* ------------------------------------------------------------------------- *
 * avoid including any gtk stuff
 * ------------------------------------------------------------------------- */

typedef int gint;
typedef gint (*GtkFunction)(void *);

/* ------------------------------------------------------------------------- *
 * struct tspec_t  --  time specifier
 * ------------------------------------------------------------------------- */

typedef struct tspec_t
{
  clock_t rtime; // elapsed real time
  clock_t utime; // cpu time in user space
  clock_t stime; // cpu time in system space

#if SHOW_TOD_IN_EVENT
  struct timeval tod;
#endif
} tspec_t;

/* ------------------------------------------------------------------------- *
 * struct finfo_t  --  function information
 * ------------------------------------------------------------------------- */

typedef struct finfo_t
{
  const char *name;
  tspec_t     enter;
  tspec_t     leave;
} finfo_t;

/* ========================================================================= *
 * Configuration
 * ========================================================================= */

enum
{
  FINFO_MAX = 32,  // maximum number of functions to track
};


/* ========================================================================= *
 * Module Data
 * ========================================================================= */


/* ------------------------------------------------------------------------- *
 * function info table
 * ------------------------------------------------------------------------- */

static finfo_t finfo_tab[FINFO_MAX];
static int     finfo_cnt = 0;

/* ------------------------------------------------------------------------- *
 * reference point for real time
 * ------------------------------------------------------------------------- */

static clock_t tspec_ref = 0;

#if SHOW_TOD_IN_HEADER
static struct timeval tspec_tod;
#endif

/* ------------------------------------------------------------------------- *
 * output path for logs
 * ------------------------------------------------------------------------- */

static char lib_logpath[256];
static char lib_appname[256];

/* ------------------------------------------------------------------------- *
 * flag set if gtk_init() was called and gui_idle() reached
 * ------------------------------------------------------------------------- */

static int  was_a_gtk_app;


/* ========================================================================= *
 * Module Code
 * ========================================================================= */


/* ------------------------------------------------------------------------- *
 * finfo_cmp  --  qsort callback for sorting finfo_tab slots
 * ------------------------------------------------------------------------- */

static int finfo_cmp(const void *a, const void *b)
{
  const finfo_t *A = a;
  const finfo_t *B = b;

#define CP(a,b) (((a)>(b))-((a)<(b)))
  int r = CP(A->enter.rtime,B->enter.rtime);
  return r ? r : CP(A,B);
#undef CP
}

/* ------------------------------------------------------------------------- *
 * lib_logpath_set  --  generate path for logfile output
 * ------------------------------------------------------------------------- */

static void lib_logpath_set(void)
{
  int file;

  char dbuf[256];
  const char *dir;

  char bbuf[256];
  const char *base;

  /* - - - - - - - - - - - - - - - - - - - *
   * output directory:
   * 1. $STARTUP_LOG_DIR
   * 2. current dir
   * 3. "/tmp"
   * - - - - - - - - - - - - - - - - - - - */

  if( (dir = env_get("STARTUP_LOG_DIR")) == 0 )
  {
    if( (dir = getcwd(dbuf, sizeof dbuf)) == 0 )
    {
      dir = "/tmp";
    }
  }

  /* - - - - - - - - - - - - - - - - - - - *
   * output basename:
   * 1. basename of command
   * 2. "unknown"
   * - - - - - - - - - - - - - - - - - - - */

  memset(bbuf, 0, sizeof bbuf);
  if( (file = open("/proc/self/cmdline", O_RDONLY)) != -1 )
  {
    read(file, bbuf, sizeof bbuf - 1);
    close(file);
  }
  base = strrchr(bbuf, '/');
  base = base ? base+1 : bbuf;
  base = *base ? base : "unknown";

  snprintf(lib_appname, sizeof lib_appname, "%s", base);

  snprintf(lib_logpath, sizeof lib_logpath,
           "%s/%s--%d.startup", dir, base, (int)getpid());


  /* - - - - - - - - - - - - - - - - - - - *
   * excplicit full path override:
   * 1. $STARTUP_LOG_FILE
   * - - - - - - - - - - - - - - - - - - - */

  if( (base = env_get("STARTUP_LOG_FILE")) != 0 )
  {
    snprintf(lib_logpath, sizeof lib_logpath, "%s", base);
  }
}

/* ------------------------------------------------------------------------- *
 * real_entry  --  locate actual code for named function
 * ------------------------------------------------------------------------- */

static void *real_entry(const char *symb)
{
  void *real = dlsym(RTLD_NEXT, symb);
  if( real == 0 )
  {
    fprintf(stderr, "%s: %s", symb, dlerror());
    exit(1);
  }
  return real;
}

#define GET_REAL_ENTRY if( real == 0 ) { real = real_entry(__FUNCTION__); }

/* ------------------------------------------------------------------------- *
 * tspec_get  --  get current time stamp
 * ------------------------------------------------------------------------- */

static void tspec_get(tspec_t *self)
{
  struct tms tms;
  self->rtime = times(&tms) - tspec_ref;
  self->utime = tms.tms_utime;
  self->stime = tms.tms_stime;
  
#if SHOW_TOD_IN_EVENT
  gettimeofday(&self->tod, 0);
#endif
}

/* ------------------------------------------------------------------------- *
 * finfo_find  --  get function slot by name
 * ------------------------------------------------------------------------- */

static finfo_t *finfo_find(const char *name)
{
  int i;
  for( i = 0; ; ++i )
  {
    if( i == finfo_cnt )
    {
      if( finfo_cnt++ == FINFO_MAX )
      {
        abort();
      }
      finfo_tab[i].name = name;
    }
    if( !strcmp(finfo_tab[i].name, name) )
    {
      break;
    }
  }
  return &finfo_tab[i];
}

/* ------------------------------------------------------------------------- *
 * lib_syslog_idle  --  write to syslog the time taken to reach idle
 * ------------------------------------------------------------------------- */

static void lib_syslog_idle(void)
{
  finfo_t *f = finfo_find("gui_idle");

  /* LOG_PERROR flag duplicates the logging to stderr, but as we're
   * actually outputting just one line per run, it's pretty harmless
   * and saves the day if we don't have a syslog daemon running. */

  openlog("libstartup", LOG_PERROR | LOG_PID, LOG_USER);
  syslog(LOG_MAKEPRI(LOG_USER, LOG_INFO), "%s=%.2f seconds\n",
         lib_appname, f->enter.rtime / (double) sysconf(_SC_CLK_TCK));
  closelog();
}

/* ------------------------------------------------------------------------- *
 * lib_write_log  --  write out the CSV format log file
 * ------------------------------------------------------------------------- */

static void lib_write_log(void)
{
  /* - - - - - - - - - - - - - - - - - - - *
   * make sure the slots are in time order
   * - - - - - - - - - - - - - - - - - - - */

  if( finfo_cnt >= 2 )
  {
    qsort(finfo_tab, finfo_cnt, sizeof *finfo_tab, finfo_cmp);
  }

  /* - - - - - - - - - - - - - - - - - - - *
   * open log file
   * - - - - - - - - - - - - - - - - - - - */

  FILE *f_use = stderr;
  FILE *f_new = 0;

  debug_msg("writing '%s' ...\n", lib_logpath);
  
  if( (f_new = fopen(lib_logpath, "w")) != 0 )
  {
    f_use = f_new;
  }
  else
  {
    perror_msg(lib_logpath);
  }

  /* - - - - - - - - - - - - - - - - - - - *
   * write log data
   * - - - - - - - - - - - - - - - - - - - */

#define FMT "%.3f"
#define TOD "%.6f"

  fprintf(f_use, "generator=%s %s\n", TOOL_NAME, TOOL_VERS);
# if SHOW_TOD_IN_HEADER
  fprintf(f_use, "create_tod="TOD"\n",
	  tspec_tod.tv_sec + tspec_tod.tv_usec * 1e-6);
# endif
  
  fprintf(f_use, "\n");
#if SHOW_TOD_IN_EVENT
  fprintf(f_use, "t_tod,");
#endif
  
  fprintf(f_use, "t_real,t_user,t_sys,f_real,f_user,f_sys,name\n");

  finfo_t *p = 0;
  double t2s = 1.0 / sysconf(_SC_CLK_TCK);

  for( int i = 0; i < finfo_cnt; ++i )
  {
    finfo_t *f = &finfo_tab[i];
    double r,u,s;

    if( p != 0 )
    {
      if( (f->enter.utime != p->leave.utime) ||
          (f->enter.stime != p->leave.stime) )
      {
#if SHOW_TOD_IN_EVENT
	fprintf(f_use, ""TOD",", 
		p->leave.tod.tv_sec + p->leave.tod.tv_usec * 1e-6);
#endif


        r = t2s * p->leave.rtime;
        u = t2s * p->leave.utime;
        s = t2s * p->leave.stime;
        fprintf(f_use,""FMT","FMT","FMT",", r,u,s);

        r = t2s * (f->enter.rtime - p->leave.rtime);
        u = t2s * (f->enter.utime - p->leave.utime);
        s = t2s * (f->enter.stime - p->leave.stime);
        fprintf(f_use,""FMT","FMT","FMT",", r,u,s);

        fprintf(f_use,"%s->%s\n", p->name, f->name);
      }
    }

#if SHOW_TOD_IN_EVENT
    fprintf(f_use, ""TOD",", 
	    f->enter.tod.tv_sec + f->enter.tod.tv_usec * 1e-6);
#endif

    r = t2s * f->enter.rtime;
    u = t2s * f->enter.utime;
    s = t2s * f->enter.stime;
    fprintf(f_use,""FMT","FMT","FMT",", r,u,s);

    r = t2s * (f->leave.rtime - f->enter.rtime);
    u = t2s * (f->leave.utime - f->enter.utime);
    s = t2s * (f->leave.stime - f->enter.stime);
    fprintf(f_use,""FMT","FMT","FMT",", r,u,s);

    fprintf(f_use,"%s\n", f->name);

    p = f;
  }
#undef FMT
#undef TOD

  /* - - - - - - - - - - - - - - - - - - - *
   * flush & cleanup
   * - - - - - - - - - - - - - - - - - - - */

  fflush(f_use);
  if( f_new != 0 ) fclose(f_new);
}

/* ------------------------------------------------------------------------- *
 * log_enter / log_leave  --  hook timings
 * ------------------------------------------------------------------------- */

static void log_enter(const char *name)
{
  tspec_get(&finfo_find(name)->enter);
}

static void log_leave(const char *name)
{
  tspec_get(&finfo_find(name)->leave);
}

static void log_enter_and_leave(const char *name)
{
  finfo_t *f = finfo_find(name);
  tspec_get(&f->enter);
  f->leave = f->enter;
}

#define LOG_ENTER log_enter(__FUNCTION__);
#define LOG_LEAVE log_leave(__FUNCTION__);
#define LOG_ENTER_AND_LEAVE log_enter_and_leave(__FUNCTION__);

/* ------------------------------------------------------------------------- *
 * lib_exit  --  this should end up as the last called atexit handler
 * ------------------------------------------------------------------------- */

static void lib_exit(void)
{
  LOG_ENTER_AND_LEAVE
}

/* ------------------------------------------------------------------------- *
 * lib_init  --  this is called by dynamic loader, just before main()
 * ------------------------------------------------------------------------- */

/* - - - - - - - - - - - - - - - - - - - *
 * list of signals to catch and the
 * function to call when caught
 * - - - - - - - - - - - - - - - - - - - */

static const int lib_signal_list[] =
{
  SIGHUP,SIGINT,SIGQUIT,-1
};

static void lib_signal_handler(int sig)
{
  fprintf(stderr, "%s: unhandled signal %d (%s)\n",
	  lib_appname, sig, strsignal(sig));
  signal(sig, SIG_DFL);
  exit(1);
}

#if SHOW_TOD_IN_HEADER || SHOW_TOD_IN_EVENT
static void secs_to_timeval(struct timeval *tv, double secs)
{
  // we expect to use this only for small values
  // assert( secs >= 0 && secs <= 100 );
  
  // avoid libm functions
  unsigned s = (unsigned)(secs);
  unsigned u = (unsigned)((secs - s) * 1e6);
  
// QUARANTINE   double s = floor(s);
// QUARANTINE   double u = (secs - s) * 1e6;
  
  tv->tv_sec  = (time_t)s;
  tv->tv_usec = (suseconds_t)u;
}
#endif

static void lib_init(void) __attribute__((constructor));

static void lib_init(void)
{
  debug_msg("@ %s ...\n", __FUNCTION__);

  /* - - - - - - - - - - - - - - - - - - - *
   * wait for change in lower accuracy
   * clock source -> makes it more likely
   * that (cpuN - cpu0) == (todN - tod0),
   * but also introduces upto 1 jiffy error
   * (10 ms, depending on kernel config)
   * - - - - - - - - - - - - - - - - - - - */

#if SHOW_TOD_IN_HEADER || SHOW_TOD_IN_EVENT
  {
    clock_t t = times(0);
    while( t == times(0) ) { }
  }
#endif
  
  /* - - - - - - - - - - - - - - - - - - - *
   * calculate creation time difference of
   * current process and forked child
   * -> time it took to load & init libs
   * - - - - - - - - - - - - - - - - - - - */
  
  double  ldtime = fork_starttime() - proc_starttime();
  clock_t ldtick = (clock_t)(ldtime * sysconf(_SC_CLK_TCK) + 0.5);
  
// QUARANTINE   debug_msg("ldtime %.2f\n", ldtime);
// QUARANTINE   debug_msg("ldtick %u\n", ldtick);

  /* - - - - - - - - - - - - - - - - - - - *
   * elapsed real time reference point set
   * to current time - load time
   * - - - - - - - - - - - - - - - - - - - */

  tspec_t t;
  tspec_get(&t);
  tspec_ref = t.rtime - ldtick;

  /* - - - - - - - - - - - - - - - - - - - *
   * the "create" slot will have zero enter
   * times and current values for leave
   * - - - - - - - - - - - - - - - - - - - */

  finfo_t *f = finfo_find("create");
  tspec_get(&f->leave);

  /* - - - - - - - - - - - - - - - - - - - *
   * calculate timeofday for create
   * - - - - - - - - - - - - - - - - - - - */

#if SHOW_TOD_IN_HEADER || SHOW_TOD_IN_EVENT
  struct timeval tv;
  secs_to_timeval(&tv, ldtime);
  timersub(&t.tod, &tv, &tv);
  
# if SHOW_TOD_IN_HEADER
  tspec_tod = tv;
# endif
  
# if SHOW_TOD_IN_EVENT
  f->enter.tod = tv;
# endif
#endif
  
  /* - - - - - - - - - - - - - - - - - - - *
   * then handle "lib_init" slot
   * - - - - - - - - - - - - - - - - - - - */

  LOG_ENTER

  env_update_from_file(ENV_FILE_PATH);
  env_update_from_real();
  
  lib_logpath_set();
  atexit(lib_exit);
  
  for( int i = 0; lib_signal_list[i] != -1; ++i )
  {
    signal(lib_signal_list[i], lib_signal_handler);
  }
  
  LOG_LEAVE
}

/* ------------------------------------------------------------------------- *
 * lib_fini  --  this is called by dynamic loader, after atexit handlers
 * ------------------------------------------------------------------------- */

static void lib_fini(void) __attribute__((destructor));

static void lib_fini(void)
{
  debug_msg("@ %s ...\n", __FUNCTION__);

  /* - - - - - - - - - - - - - - - - - - - *
   * add "lib_fini" slot as the last item
   * - - - - - - - - - - - - - - - - - - - */

  LOG_ENTER_AND_LEAVE

  /* - - - - - - - - - - - - - - - - - - - *
   * write log file if requested
   * - - - - - - - - - - - - - - - - - - - */

  const char *val = env_get("STARTUP_WRITE_LOG");
  
  if( val != 0 )
  {
    // "forced" -> write log for all apps
    // "yes"    -> write log only for gtk apps
    
    if( *val == 'f' || (*val == 'y' && was_a_gtk_app) )
    {
      lib_write_log();
    }
    else
    {
      debug_msg("going to skip writing log\n");
    }
  }
  else
  {
    debug_msg("not configured to write log\n");
  }
}


/* ========================================================================= *
 * Hooked GTK functionality
 * ========================================================================= */

void gtk_main_quit(void)
{
  static void (*real)(void) = 0;

  LOG_ENTER

  GET_REAL_ENTRY

  real();

  LOG_LEAVE
}

/* ------------------------------------------------------------------------- *
 * gui_idle  --  called by gtk when gui is idle
 * ------------------------------------------------------------------------- */

// QUARANTINE static void stuff(const char *fmt, ...)
// QUARANTINE {
// QUARANTINE   FILE *f = fopen("/dev/tty","a");
// QUARANTINE   if( f != 0 )
// QUARANTINE   {
// QUARANTINE     va_list va;
// QUARANTINE     va_start(va, fmt);
// QUARANTINE     vfprintf(f, fmt, va);
// QUARANTINE     va_end(va);
// QUARANTINE     fclose(f);
// QUARANTINE   }
// QUARANTINE }

#if 0
static gint gui_idle_old(void *data)
{
  static clock_t last  = (clock_t)(-1);
  static int     reps  = 0;
  gint           again = TRUE;

  LOG_ENTER_AND_LEAVE

  was_a_gtk_app = 1;

// QUARANTINE   stuff("IDLE\n");
  
  /* TODO: idle time detection should really be better than this */
  
  clock_t curr = times(0);

  if( last != curr )
  {
    last = curr, reps = 0;
  }
  else
  {
    ++reps;
  }

  if( reps >= 4 )
  {
    /* - - - - - - - - - - - - - - - - - - - *
     * wait until we are really idle...
     * that is we get executed N times in
     * row without doing something else in
     * between
     * - - - - - - - - - - - - - - - - - - - */

    again = FALSE;

    lib_syslog_idle();

    if( env_get("STARTUP_IDLE_QUIT") != 0 )
    {
      /* Seems that not all programs like us calling
       * gtk_main_quit() this way, so we'll just call
       * exit() and abandon hope of getting shutdown
       * time values ... */

      //void (*quit)(void) = real_entry("gtk_main_quit");
      //quit();

      gtk_main_quit();

      //exit(0);
    }
  }

  return again;
}
#endif

/* ------------------------------------------------------------------------- *
 * gtk_init  --  must be called before using gtk api, usually from main
 * ------------------------------------------------------------------------- */

static void (*idle_add)(GtkFunction, void*);
static void (*timer_add)(gint, GtkFunction, void*);

static gint gui_idle(void *data);
static gint gui_tick(void *data);

void gtk_init(int *argc, char ***argv)
{
  static void (*real)(int *, char ***) = 0;

// QUARANTINE stuff("IDLE_TICK_TIME = %ld\n", (long)IDLE_TICK_TIME);
// QUARANTINE stuff("IDLE_TICK_RATE = %ld\n", (long)IDLE_TICK_RATE);
// QUARANTINE stuff("IDLE_CPU_USAGE = %ld\n", (long)IDLE_CPU_USAGE);

  LOG_ENTER

  GET_REAL_ENTRY

  
  real(argc, argv);

  idle_add  = real_entry("gtk_idle_add");
  timer_add = real_entry("gtk_timeout_add");
  idle_add(gui_idle, 0);
  
// QUARANTINE   {
// QUARANTINE     void (*idle_add)(GtkFunction, void*);
// QUARANTINE     idle_add = real_entry("gtk_idle_add");
// QUARANTINE     idle_add(gui_idle, 0);
// QUARANTINE   }

  LOG_LEAVE
}

/* ------------------------------------------------------------------------- *
 * gtk_main  --  control is yielded to GTK
 * ------------------------------------------------------------------------- */

void gtk_main(void)
{
  /* - - - - - - - - - - - - - - - - - - - *
   * we do not return for a while, so log
   * only (the first) entry to gtk_main
   * - - - - - - - - - - - - - - - - - - - */

  static void (*real)(void) = 0;

  if( real == 0 )
  {
    LOG_ENTER_AND_LEAVE

    GET_REAL_ENTRY

    real();
  }
  else
  {
    real();
  }
}


/* ========================================================================= *
 * IDLE TIME DETECTION
 * ========================================================================= */

/* ------------------------------------------------------------------------- *
 * Configuration
 * ------------------------------------------------------------------------- */


enum
{
  /* ms interval to check cpu usage after gtk goes idle */
  IDLE_TICK_TIME = 100, 
 
  /* number of idle samples needed for idle detection */
  IDLE_NUM_TICKS = 10,
  
  
  /* % cpu activity allowed in one IDLE_TICK_TIME slot */
  IDLE_TICK_RATE =  10, 

  /* IDLE_TICK_RATE in clock() units */
  
#if 01
  IDLE_CPU_USAGE = (CLOCKS_PER_SEC * 
		    IDLE_TICK_TIME / 1000 * 
		    IDLE_TICK_RATE /100),
#else
  IDLE_CPU_USAGE = (CLOCKS_PER_SEC * 2 / 100),
#endif
  
};

/* - - - - - - - - - - - - - - - - - - - *
 * IDLE_CUMULATIVE == 0
 * 
 *   cpu usage during one sample must be
 *   less than IDLE_CPU_USAGE
 * 
 * IDLE_CUMULATIVE != 0
 * 
 *   total cpu usage during sampling must
 *   be less than IDLE_CPU_USAGE
 * - - - - - - - - - - - - - - - - - - - */

#define IDLE_CUMULATIVE 0

/* ------------------------------------------------------------------------- *
 * Variables
 * ------------------------------------------------------------------------- */

/* - - - - - - - - - - - - - - - - - - - *
 * samples since last idle
 * - - - - - - - - - - - - - - - - - - - */

static size_t  idle_ticks;

/* - - - - - - - - - - - - - - - - - - - *
 * time of last sample (or idle)
 * - - - - - - - - - - - - - - - - - - - */

static clock_t idle_clocks;


/* ------------------------------------------------------------------------- *
 * gui_idle  --  gtk application is idle
 * ------------------------------------------------------------------------- */

static gint gui_idle(void *data)
{
  /* - - - - - - - - - - - - - - - - - - - *
   * update statistics and mark as
   * gtk application
   * - - - - - - - - - - - - - - - - - - - */

  LOG_ENTER_AND_LEAVE
  was_a_gtk_app = 1;

// QUARANTINE   {
// QUARANTINE     finfo_t *f = finfo_find("gui_idle");
// QUARANTINE     stuff("IDLE @ %.3f s\n", f->enter.rtime / (double)sysconf(_SC_CLK_TCK));
// QUARANTINE   }
  
  /* - - - - - - - - - - - - - - - - - - - *
   * start polling cpu usage
   * - - - - - - - - - - - - - - - - - - - */

  idle_ticks  = 0;
  idle_clocks = clock();
  timer_add(IDLE_TICK_TIME, gui_tick, 0);

  /* - - - - - - - - - - - - - - - - - - - *
   * do not call me again
   * - - - - - - - - - - - - - - - - - - - */
  
  return FALSE;
}

/* ------------------------------------------------------------------------- *
 * gui_tick  --  sample cpu usage since last idle
 * ------------------------------------------------------------------------- */

static gint gui_tick(void *data)
{
  /* - - - - - - - - - - - - - - - - - - - *
   * cpu usage since reference point
   * - - - - - - - - - - - - - - - - - - - */

  clock_t d = clock() - idle_clocks;

// QUARANTINE   stuff("TICK %d - %d - %.3f s\n", idle_ticks, d, d/(double)CLOCKS_PER_SEC);

  /* - - - - - - - - - - - - - - - - - - - *
   * if cpu usage is over the threshold,
   * stop sample timer and  wait for the
   * next idle time
   * - - - - - - - - - - - - - - - - - - - */

  if( d >= IDLE_CPU_USAGE )
  {
    idle_add(gui_idle, 0);
    return FALSE;
  }
  
  /* - - - - - - - - - - - - - - - - - - - *
   * if enough samples without significant
   * cpu usage are seen, stop sampling and
   * write syslog entry & possibly logfile
   * - - - - - - - - - - - - - - - - - - - */
  
  if( ++idle_ticks >= IDLE_NUM_TICKS )
  {
    lib_syslog_idle();

    if( env_get("STARTUP_IDLE_QUIT") != 0 )
    {
      gtk_main_quit();
    }
    return FALSE;
  }

  /* - - - - - - - - - - - - - - - - - - - *
   * keep sampling
   * - - - - - - - - - - - - - - - - - - - */
  
#if !IDLE_CUMULATIVE
  idle_clocks += d;
#endif
  return TRUE;
}


/* ========================================================================= *
 * CUSTOM TIMESTAMPS
 * ========================================================================= */

/* ------------------------------------------------------------------------- *
 * libstartup_custom_timestamp  --  emit custom timestamp to log
 * ------------------------------------------------------------------------- */

static void libstartup_custom_timestamp(const char *name)
{
  /* - - - - - - - - - - - - - - - - - - - *
   * duplicate functionality from:
   *   log_enter_and_leave()
   * - - - - - - - - - - - - - - - - - - - */

  finfo_t *f = finfo_find(name);
  tspec_get(&f->enter);
  f->leave = f->enter;
  
  
  /* - - - - - - - - - - - - - - - - - - - *
   * make sure we do not use the caller
   * provided string ...
   * - - - - - - - - - - - - - - - - - - - */
  
  if( f->name == name )
  {
    f->name = strdup(name);
  }
}

/* ------------------------------------------------------------------------- *
 * __cyg_profile_func_enter  --  hijacked gcc instrumentation callback
 * ------------------------------------------------------------------------- */

void __cyg_profile_func_enter (void *this_fn,
			       void *call_site)
{
  /* - - - - - - - - - - - - - - - - - - - *
   * Little explanation:
   * 
   * we can't PRELOAD functions that occur
   * in the main application binary.
   * 
   * We do not want to force modified linking
   * to use custom timestamps.
   * 
   * -> We hijack a debug hook present in
   *    the libc and use it for purposes it
   *    was not meant for while trying to
   *    allow original use too. (not that
   *    you should try to do it..)
   * 
   * - - - - - - - - - - - - - - - - - - - */

  
  if( this_fn == (void *)(-1) )
  {
    libstartup_custom_timestamp(call_site);
  }
  else
  {
    static void (*real)(void *,void*) = 0;
    GET_REAL_ENTRY
    real(this_fn, call_site);
  }
}

